/*!
 * md5.js - MD5 implementation for bcrypto
 * Copyright (c) 2016-2018, Christopher Jeffrey (MIT License).
 * https://github.com/bcoin-org/bcrypto
 * Parts of this software based on hash.js.
 */

'use strict';

const assert = require('bsert');
const HMAC = require('../internal/hmac');

/*
 * Constants
 */

const FINALIZED = -1;
const DESC = Buffer.alloc(8, 0x00);
const PADDING = Buffer.alloc(64, 0x00);

PADDING[0] = 0x80;

const K = new Uint32Array([
  0xd76aa478, 0xe8c7b756, 0x242070db, 0xc1bdceee,
  0xf57c0faf, 0x4787c62a, 0xa8304613, 0xfd469501,
  0x698098d8, 0x8b44f7af, 0xffff5bb1, 0x895cd7be,
  0x6b901122, 0xfd987193, 0xa679438e, 0x49b40821,
  0xf61e2562, 0xc040b340, 0x265e5a51, 0xe9b6c7aa,
  0xd62f105d, 0x02441453, 0xd8a1e681, 0xe7d3fbc8,
  0x21e1cde6, 0xc33707d6, 0xf4d50d87, 0x455a14ed,
  0xa9e3e905, 0xfcefa3f8, 0x676f02d9, 0x8d2a4c8a,
  0xfffa3942, 0x8771f681, 0x6d9d6122, 0xfde5380c,
  0xa4beea44, 0x4bdecfa9, 0xf6bb4b60, 0xbebfbc70,
  0x289b7ec6, 0xeaa127fa, 0xd4ef3085, 0x04881d05,
  0xd9d4d039, 0xe6db99e5, 0x1fa27cf8, 0xc4ac5665,
  0xf4292244, 0x432aff97, 0xab9423a7, 0xfc93a039,
  0x655b59c3, 0x8f0ccc92, 0xffeff47d, 0x85845dd1,
  0x6fa87e4f, 0xfe2ce6e0, 0xa3014314, 0x4e0811a1,
  0xf7537e82, 0xbd3af235, 0x2ad7d2bb, 0xeb86d391
]);

const S = new Uint8Array([
  7, 12, 17, 22, 7, 12, 17, 22, 7, 12, 17, 22,
  7, 12, 17, 22, 5,  9, 14, 20, 5,  9, 14, 20,
  5,  9, 14, 20, 5,  9, 14, 20, 4, 11, 16, 23,
  4, 11, 16, 23, 4, 11, 16, 23, 4, 11, 16, 23,
  6, 10, 15, 21, 6, 10, 15, 21, 6, 10, 15, 21,
  6, 10, 15, 21
]);

/**
 * MD5
 */

class MD5 {
  /**
   * Create a MD5 context.
   * @constructor
   */

  constructor() {
    this.state = new Uint32Array(4);
    this.msg = new Uint32Array(16);
    this.block = Buffer.allocUnsafe(64);
    this.size = FINALIZED;
  }

  /**
   * Initialize MD5 context.
   */

  init() {
    this.state[0] = 0x67452301;
    this.state[1] = 0xefcdab89;
    this.state[2] = 0x98badcfe;
    this.state[3] = 0x10325476;
    this.size = 0;
    return this;
  }

  /**
   * Update MD5 context.
   * @param {Buffer} data
   */

  update(data) {
    assert(Buffer.isBuffer(data));
    this._update(data, data.length);
    return this;
  }

  /**
   * Finalize MD5 context.
   * @returns {Buffer}
   */

  final() {
    return this._final(Buffer.allocUnsafe(16));
  }

  /**
   * Update MD5 context.
   * @private
   * @param {Buffer} data
   * @param {Number} len
   */

  _update(data, len) {
    assert(this.size !== FINALIZED, 'Context already finalized.');

    let pos = this.size & 0x3f;
    let off = 0;

    this.size += len;

    if (pos > 0) {
      let want = 64 - pos;

      if (want > len)
        want = len;

      data.copy(this.block, pos, off, off + want);

      pos += want;
      len -= want;
      off += want;

      if (pos < 64)
        return;

      this.transform(this.block, 0);
    }

    while (len >= 64) {
      this.transform(data, off);
      off += 64;
      len -= 64;
    }

    if (len > 0)
      data.copy(this.block, 0, off, off + len);
  }

  /**
   * Finalize MD5 context.
   * @private
   * @param {Buffer} out
   * @returns {Buffer}
   */

  _final(out) {
    assert(this.size !== FINALIZED, 'Context already finalized.');

    const pos = this.size % 64;
    const len = this.size * 8;

    writeU32(DESC, len, 0);
    writeU32(DESC, len * (1 / 0x100000000), 4);

    this._update(PADDING, 1 + ((119 - pos) % 64));
    this._update(DESC, 8);

    for (let i = 0; i < 4; i++) {
      writeU32(out, this.state[i], i * 4);
      this.state[i] = 0;
    }

    for (let i = 0; i < 16; i++)
      this.msg[i] = 0;

    for (let i = 0; i < 64; i++)
      this.block[i] = 0;

    this.size = FINALIZED;

    return out;
  }

  /**
   * Transform MD5 block.
   * @param {Buffer} chunk
   * @param {Number} pos
   */

  transform(chunk, pos) {
    const W = this.msg;

    let a = this.state[0];
    let b = this.state[1];
    let c = this.state[2];
    let d = this.state[3];
    let i = 0;

    for (; i < 16; i++)
      W[i] = readU32(chunk, pos + i * 4);

    for (i = 0; i < 64; i++) {
      let f, g;

       if (i < 16) {
        f = (b & c) | (~b & d);
        g = i;
      } else if (i < 32) {
        f = (d & b) | (~d & c);
        g = (5 * i + 1) & 15;
      } else if (i < 48) {
        f = b ^ c ^ d;
        g = (3 * i + 5) & 15;
      } else {
        f = c ^ (b | ~d);
        g = (7 * i) & 15;
      }

      f >>>= 0;

      f = f + a + K[i] + W[g];
      f >>>= 0;

      a = d;
      d = c;
      c = b;
      b = b + rotl32(f, S[i]);
      b >>>= 0;
    }

    this.state[0] += a;
    this.state[1] += b;
    this.state[2] += c;
    this.state[3] += d;
  }

  static hash() {
    return new MD5();
  }

  static hmac() {
    return new HMAC(MD5, 64);
  }

  static digest(data) {
    return MD5.ctx.init().update(data).final();
  }

  static root(left, right) {
    assert(Buffer.isBuffer(left) && left.length === 16);
    assert(Buffer.isBuffer(right) && right.length === 16);
    return MD5.ctx.init().update(left).update(right).final();
  }

  static multi(x, y, z) {
    const ctx = MD5.ctx;
    ctx.init();
    ctx.update(x);
    ctx.update(y);
    if (z)
      ctx.update(z);
    return ctx.final();
  }

  static mac(data, key) {
    return MD5.hmac().init(key).update(data).final();
  }
}

MD5.native = 0;
MD5.id = 'MD5';
MD5.size = 16;
MD5.bits = 128;
MD5.blockSize = 64;
MD5.zero = Buffer.alloc(16, 0x00);
MD5.ctx = new MD5();

/*
 * Helpers
 */

function rotl32(w, b) {
  return (w << b) | (w >>> (32 - b));
}

function readU32(buf, offset) {
  return ((buf[offset + 3] & 0xff) * 0x1000000)
    + (((buf[offset + 2] & 0xff) << 16)
    | ((buf[offset + 1] & 0xff) << 8)
    | (buf[offset] & 0xff));
}

function writeU32(buf, value, offset) {
  buf[offset + 3] = value >>> 24;
  buf[offset + 2] = (value >> 16) & 0xff;
  buf[offset + 1] = (value >> 8) & 0xff;
  buf[offset] = value & 0xff;
}

/*
 * Expose
 */

module.exports = MD5;
