/*!
 * whirlpool.js - Whirlpool implementation for bcrypto
 * Copyright (c) 2018, Christopher Jeffrey (MIT License).
 * https://github.com/bcoin-org/bcrypto
 *
 * Parts of this software are based on RustCrypto/hashes:
 *   Copyright (c) 2016-2018, The RustCrypto Authors (MIT License).
 *   https://github.com/RustCrypto/hashes
 *
 * Resources:
 *   https://en.wikipedia.org/wiki/Whirlpool_(hash_function)
 *   https://www.iso.org/standard/39876.html
 *   https://github.com/jzelinskie/whirlpool/blob/master/whirlpool.go
 *   https://github.com/RustCrypto/hashes/blob/master/whirlpool/src/consts.rs
 *   https://github.com/RustCrypto/hashes/blob/master/whirlpool/src/lib.rs
 *   https://github.com/RustCrypto/hashes/blob/master/whirlpool/src/utils.rs
 */

'use strict';

const assert = require('bsert');
const HMAC = require('../internal/hmac');

/*
 * Constants
 */

const FINALIZED = -1;
const DESC = Buffer.alloc(32, 0x00);
const PADDING = Buffer.alloc(64, 0x00);

PADDING[0] = 0x80;

const RC = new Uint32Array([
  0x00000000, 0x00000000,
  0x1823c6e8, 0x87b8014f,
  0x36a6d2f5, 0x796f9152,
  0x60bc9b8e, 0xa30c7b35,
  0x1de0d7c2, 0x2e4bfe57,
  0x157737e5, 0x9ff04ada,
  0x58c9290a, 0xb1a06b85,
  0xbd5d10f4, 0xcb3e0567,
  0xe427418b, 0xa77d95d8,
  0xfbee7c66, 0xdd17479e,
  0xca2dbf07, 0xad5a8333
]);

const C0 = new Uint32Array([
  0x18186018, 0xc07830d8, 0x23238c23, 0x05af4626,
  0xc6c63fc6, 0x7ef991b8, 0xe8e887e8, 0x136fcdfb,
  0x87872687, 0x4ca113cb, 0xb8b8dab8, 0xa9626d11,
  0x01010401, 0x08050209, 0x4f4f214f, 0x426e9e0d,
  0x3636d836, 0xadee6c9b, 0xa6a6a2a6, 0x590451ff,
  0xd2d26fd2, 0xdebdb90c, 0xf5f5f3f5, 0xfb06f70e,
  0x7979f979, 0xef80f296, 0x6f6fa16f, 0x5fcede30,
  0x91917e91, 0xfcef3f6d, 0x52525552, 0xaa07a4f8,
  0x60609d60, 0x27fdc047, 0xbcbccabc, 0x89766535,
  0x9b9b569b, 0xaccd2b37, 0x8e8e028e, 0x048c018a,
  0xa3a3b6a3, 0x71155bd2, 0x0c0c300c, 0x603c186c,
  0x7b7bf17b, 0xff8af684, 0x3535d435, 0xb5e16a80,
  0x1d1d741d, 0xe8693af5, 0xe0e0a7e0, 0x5347ddb3,
  0xd7d77bd7, 0xf6acb321, 0xc2c22fc2, 0x5eed999c,
  0x2e2eb82e, 0x6d965c43, 0x4b4b314b, 0x627a9629,
  0xfefedffe, 0xa321e15d, 0x57574157, 0x8216aed5,
  0x15155415, 0xa8412abd, 0x7777c177, 0x9fb6eee8,
  0x3737dc37, 0xa5eb6e92, 0xe5e5b3e5, 0x7b56d79e,
  0x9f9f469f, 0x8cd92313, 0xf0f0e7f0, 0xd317fd23,
  0x4a4a354a, 0x6a7f9420, 0xdada4fda, 0x9e95a944,
  0x58587d58, 0xfa25b0a2, 0xc9c903c9, 0x06ca8fcf,
  0x2929a429, 0x558d527c, 0x0a0a280a, 0x5022145a,
  0xb1b1feb1, 0xe14f7f50, 0xa0a0baa0, 0x691a5dc9,
  0x6b6bb16b, 0x7fdad614, 0x85852e85, 0x5cab17d9,
  0xbdbdcebd, 0x8173673c, 0x5d5d695d, 0xd234ba8f,
  0x10104010, 0x80502090, 0xf4f4f7f4, 0xf303f507,
  0xcbcb0bcb, 0x16c08bdd, 0x3e3ef83e, 0xedc67cd3,
  0x05051405, 0x28110a2d, 0x67678167, 0x1fe6ce78,
  0xe4e4b7e4, 0x7353d597, 0x27279c27, 0x25bb4e02,
  0x41411941, 0x32588273, 0x8b8b168b, 0x2c9d0ba7,
  0xa7a7a6a7, 0x510153f6, 0x7d7de97d, 0xcf94fab2,
  0x95956e95, 0xdcfb3749, 0xd8d847d8, 0x8e9fad56,
  0xfbfbcbfb, 0x8b30eb70, 0xeeee9fee, 0x2371c1cd,
  0x7c7ced7c, 0xc791f8bb, 0x66668566, 0x17e3cc71,
  0xdddd53dd, 0xa68ea77b, 0x17175c17, 0xb84b2eaf,
  0x47470147, 0x02468e45, 0x9e9e429e, 0x84dc211a,
  0xcaca0fca, 0x1ec589d4, 0x2d2db42d, 0x75995a58,
  0xbfbfc6bf, 0x9179632e, 0x07071c07, 0x381b0e3f,
  0xadad8ead, 0x012347ac, 0x5a5a755a, 0xea2fb4b0,
  0x83833683, 0x6cb51bef, 0x3333cc33, 0x85ff66b6,
  0x63639163, 0x3ff2c65c, 0x02020802, 0x100a0412,
  0xaaaa92aa, 0x39384993, 0x7171d971, 0xafa8e2de,
  0xc8c807c8, 0x0ecf8dc6, 0x19196419, 0xc87d32d1,
  0x49493949, 0x7270923b, 0xd9d943d9, 0x869aaf5f,
  0xf2f2eff2, 0xc31df931, 0xe3e3abe3, 0x4b48dba8,
  0x5b5b715b, 0xe22ab6b9, 0x88881a88, 0x34920dbc,
  0x9a9a529a, 0xa4c8293e, 0x26269826, 0x2dbe4c0b,
  0x3232c832, 0x8dfa64bf, 0xb0b0fab0, 0xe94a7d59,
  0xe9e983e9, 0x1b6acff2, 0x0f0f3c0f, 0x78331e77,
  0xd5d573d5, 0xe6a6b733, 0x80803a80, 0x74ba1df4,
  0xbebec2be, 0x997c6127, 0xcdcd13cd, 0x26de87eb,
  0x3434d034, 0xbde46889, 0x48483d48, 0x7a759032,
  0xffffdbff, 0xab24e354, 0x7a7af57a, 0xf78ff48d,
  0x90907a90, 0xf4ea3d64, 0x5f5f615f, 0xc23ebe9d,
  0x20208020, 0x1da0403d, 0x6868bd68, 0x67d5d00f,
  0x1a1a681a, 0xd07234ca, 0xaeae82ae, 0x192c41b7,
  0xb4b4eab4, 0xc95e757d, 0x54544d54, 0x9a19a8ce,
  0x93937693, 0xece53b7f, 0x22228822, 0x0daa442f,
  0x64648d64, 0x07e9c863, 0xf1f1e3f1, 0xdb12ff2a,
  0x7373d173, 0xbfa2e6cc, 0x12124812, 0x905a2482,
  0x40401d40, 0x3a5d807a, 0x08082008, 0x40281048,
  0xc3c32bc3, 0x56e89b95, 0xecec97ec, 0x337bc5df,
  0xdbdb4bdb, 0x9690ab4d, 0xa1a1bea1, 0x611f5fc0,
  0x8d8d0e8d, 0x1c830791, 0x3d3df43d, 0xf5c97ac8,
  0x97976697, 0xccf1335b, 0x00000000, 0x00000000,
  0xcfcf1bcf, 0x36d483f9, 0x2b2bac2b, 0x4587566e,
  0x7676c576, 0x97b3ece1, 0x82823282, 0x64b019e6,
  0xd6d67fd6, 0xfea9b128, 0x1b1b6c1b, 0xd87736c3,
  0xb5b5eeb5, 0xc15b7774, 0xafaf86af, 0x112943be,
  0x6a6ab56a, 0x77dfd41d, 0x50505d50, 0xba0da0ea,
  0x45450945, 0x124c8a57, 0xf3f3ebf3, 0xcb18fb38,
  0x3030c030, 0x9df060ad, 0xefef9bef, 0x2b74c3c4,
  0x3f3ffc3f, 0xe5c37eda, 0x55554955, 0x921caac7,
  0xa2a2b2a2, 0x791059db, 0xeaea8fea, 0x0365c9e9,
  0x65658965, 0x0fecca6a, 0xbabad2ba, 0xb9686903,
  0x2f2fbc2f, 0x65935e4a, 0xc0c027c0, 0x4ee79d8e,
  0xdede5fde, 0xbe81a160, 0x1c1c701c, 0xe06c38fc,
  0xfdfdd3fd, 0xbb2ee746, 0x4d4d294d, 0x52649a1f,
  0x92927292, 0xe4e03976, 0x7575c975, 0x8fbceafa,
  0x06061806, 0x301e0c36, 0x8a8a128a, 0x249809ae,
  0xb2b2f2b2, 0xf940794b, 0xe6e6bfe6, 0x6359d185,
  0x0e0e380e, 0x70361c7e, 0x1f1f7c1f, 0xf8633ee7,
  0x62629562, 0x37f7c455, 0xd4d477d4, 0xeea3b53a,
  0xa8a89aa8, 0x29324d81, 0x96966296, 0xc4f43152,
  0xf9f9c3f9, 0x9b3aef62, 0xc5c533c5, 0x66f697a3,
  0x25259425, 0x35b14a10, 0x59597959, 0xf220b2ab,
  0x84842a84, 0x54ae15d0, 0x7272d572, 0xb7a7e4c5,
  0x3939e439, 0xd5dd72ec, 0x4c4c2d4c, 0x5a619816,
  0x5e5e655e, 0xca3bbc94, 0x7878fd78, 0xe785f09f,
  0x3838e038, 0xddd870e5, 0x8c8c0a8c, 0x14860598,
  0xd1d163d1, 0xc6b2bf17, 0xa5a5aea5, 0x410b57e4,
  0xe2e2afe2, 0x434dd9a1, 0x61619961, 0x2ff8c24e,
  0xb3b3f6b3, 0xf1457b42, 0x21218421, 0x15a54234,
  0x9c9c4a9c, 0x94d62508, 0x1e1e781e, 0xf0663cee,
  0x43431143, 0x22528661, 0xc7c73bc7, 0x76fc93b1,
  0xfcfcd7fc, 0xb32be54f, 0x04041004, 0x20140824,
  0x51515951, 0xb208a2e3, 0x99995e99, 0xbcc72f25,
  0x6d6da96d, 0x4fc4da22, 0x0d0d340d, 0x68391a65,
  0xfafacffa, 0x8335e979, 0xdfdf5bdf, 0xb684a369,
  0x7e7ee57e, 0xd79bfca9, 0x24249024, 0x3db44819,
  0x3b3bec3b, 0xc5d776fe, 0xabab96ab, 0x313d4b9a,
  0xcece1fce, 0x3ed181f0, 0x11114411, 0x88552299,
  0x8f8f068f, 0x0c890383, 0x4e4e254e, 0x4a6b9c04,
  0xb7b7e6b7, 0xd1517366, 0xebeb8beb, 0x0b60cbe0,
  0x3c3cf03c, 0xfdcc78c1, 0x81813e81, 0x7cbf1ffd,
  0x94946a94, 0xd4fe3540, 0xf7f7fbf7, 0xeb0cf31c,
  0xb9b9deb9, 0xa1676f18, 0x13134c13, 0x985f268b,
  0x2c2cb02c, 0x7d9c5851, 0xd3d36bd3, 0xd6b8bb05,
  0xe7e7bbe7, 0x6b5cd38c, 0x6e6ea56e, 0x57cbdc39,
  0xc4c437c4, 0x6ef395aa, 0x03030c03, 0x180f061b,
  0x56564556, 0x8a13acdc, 0x44440d44, 0x1a49885e,
  0x7f7fe17f, 0xdf9efea0, 0xa9a99ea9, 0x21374f88,
  0x2a2aa82a, 0x4d825467, 0xbbbbd6bb, 0xb16d6b0a,
  0xc1c123c1, 0x46e29f87, 0x53535153, 0xa202a6f1,
  0xdcdc57dc, 0xae8ba572, 0x0b0b2c0b, 0x58271653,
  0x9d9d4e9d, 0x9cd32701, 0x6c6cad6c, 0x47c1d82b,
  0x3131c431, 0x95f562a4, 0x7474cd74, 0x87b9e8f3,
  0xf6f6fff6, 0xe309f115, 0x46460546, 0x0a438c4c,
  0xacac8aac, 0x092645a5, 0x89891e89, 0x3c970fb5,
  0x14145014, 0xa04428b4, 0xe1e1a3e1, 0x5b42dfba,
  0x16165816, 0xb04e2ca6, 0x3a3ae83a, 0xcdd274f7,
  0x6969b969, 0x6fd0d206, 0x09092409, 0x482d1241,
  0x7070dd70, 0xa7ade0d7, 0xb6b6e2b6, 0xd954716f,
  0xd0d067d0, 0xceb7bd1e, 0xeded93ed, 0x3b7ec7d6,
  0xcccc17cc, 0x2edb85e2, 0x42421542, 0x2a578468,
  0x98985a98, 0xb4c22d2c, 0xa4a4aaa4, 0x490e55ed,
  0x2828a028, 0x5d885075, 0x5c5c6d5c, 0xda31b886,
  0xf8f8c7f8, 0x933fed6b, 0x86862286, 0x44a411c2
]);

const C1 = new Uint32Array([
  0xd8181860, 0x18c07830, 0x2623238c, 0x2305af46,
  0xb8c6c63f, 0xc67ef991, 0xfbe8e887, 0xe8136fcd,
  0xcb878726, 0x874ca113, 0x11b8b8da, 0xb8a9626d,
  0x09010104, 0x01080502, 0x0d4f4f21, 0x4f426e9e,
  0x9b3636d8, 0x36adee6c, 0xffa6a6a2, 0xa6590451,
  0x0cd2d26f, 0xd2debdb9, 0x0ef5f5f3, 0xf5fb06f7,
  0x967979f9, 0x79ef80f2, 0x306f6fa1, 0x6f5fcede,
  0x6d91917e, 0x91fcef3f, 0xf8525255, 0x52aa07a4,
  0x4760609d, 0x6027fdc0, 0x35bcbcca, 0xbc897665,
  0x379b9b56, 0x9baccd2b, 0x8a8e8e02, 0x8e048c01,
  0xd2a3a3b6, 0xa371155b, 0x6c0c0c30, 0x0c603c18,
  0x847b7bf1, 0x7bff8af6, 0x803535d4, 0x35b5e16a,
  0xf51d1d74, 0x1de8693a, 0xb3e0e0a7, 0xe05347dd,
  0x21d7d77b, 0xd7f6acb3, 0x9cc2c22f, 0xc25eed99,
  0x432e2eb8, 0x2e6d965c, 0x294b4b31, 0x4b627a96,
  0x5dfefedf, 0xfea321e1, 0xd5575741, 0x578216ae,
  0xbd151554, 0x15a8412a, 0xe87777c1, 0x779fb6ee,
  0x923737dc, 0x37a5eb6e, 0x9ee5e5b3, 0xe57b56d7,
  0x139f9f46, 0x9f8cd923, 0x23f0f0e7, 0xf0d317fd,
  0x204a4a35, 0x4a6a7f94, 0x44dada4f, 0xda9e95a9,
  0xa258587d, 0x58fa25b0, 0xcfc9c903, 0xc906ca8f,
  0x7c2929a4, 0x29558d52, 0x5a0a0a28, 0x0a502214,
  0x50b1b1fe, 0xb1e14f7f, 0xc9a0a0ba, 0xa0691a5d,
  0x146b6bb1, 0x6b7fdad6, 0xd985852e, 0x855cab17,
  0x3cbdbdce, 0xbd817367, 0x8f5d5d69, 0x5dd234ba,
  0x90101040, 0x10805020, 0x07f4f4f7, 0xf4f303f5,
  0xddcbcb0b, 0xcb16c08b, 0xd33e3ef8, 0x3eedc67c,
  0x2d050514, 0x0528110a, 0x78676781, 0x671fe6ce,
  0x97e4e4b7, 0xe47353d5, 0x0227279c, 0x2725bb4e,
  0x73414119, 0x41325882, 0xa78b8b16, 0x8b2c9d0b,
  0xf6a7a7a6, 0xa7510153, 0xb27d7de9, 0x7dcf94fa,
  0x4995956e, 0x95dcfb37, 0x56d8d847, 0xd88e9fad,
  0x70fbfbcb, 0xfb8b30eb, 0xcdeeee9f, 0xee2371c1,
  0xbb7c7ced, 0x7cc791f8, 0x71666685, 0x6617e3cc,
  0x7bdddd53, 0xdda68ea7, 0xaf17175c, 0x17b84b2e,
  0x45474701, 0x4702468e, 0x1a9e9e42, 0x9e84dc21,
  0xd4caca0f, 0xca1ec589, 0x582d2db4, 0x2d75995a,
  0x2ebfbfc6, 0xbf917963, 0x3f07071c, 0x07381b0e,
  0xacadad8e, 0xad012347, 0xb05a5a75, 0x5aea2fb4,
  0xef838336, 0x836cb51b, 0xb63333cc, 0x3385ff66,
  0x5c636391, 0x633ff2c6, 0x12020208, 0x02100a04,
  0x93aaaa92, 0xaa393849, 0xde7171d9, 0x71afa8e2,
  0xc6c8c807, 0xc80ecf8d, 0xd1191964, 0x19c87d32,
  0x3b494939, 0x49727092, 0x5fd9d943, 0xd9869aaf,
  0x31f2f2ef, 0xf2c31df9, 0xa8e3e3ab, 0xe34b48db,
  0xb95b5b71, 0x5be22ab6, 0xbc88881a, 0x8834920d,
  0x3e9a9a52, 0x9aa4c829, 0x0b262698, 0x262dbe4c,
  0xbf3232c8, 0x328dfa64, 0x59b0b0fa, 0xb0e94a7d,
  0xf2e9e983, 0xe91b6acf, 0x770f0f3c, 0x0f78331e,
  0x33d5d573, 0xd5e6a6b7, 0xf480803a, 0x8074ba1d,
  0x27bebec2, 0xbe997c61, 0xebcdcd13, 0xcd26de87,
  0x893434d0, 0x34bde468, 0x3248483d, 0x487a7590,
  0x54ffffdb, 0xffab24e3, 0x8d7a7af5, 0x7af78ff4,
  0x6490907a, 0x90f4ea3d, 0x9d5f5f61, 0x5fc23ebe,
  0x3d202080, 0x201da040, 0x0f6868bd, 0x6867d5d0,
  0xca1a1a68, 0x1ad07234, 0xb7aeae82, 0xae192c41,
  0x7db4b4ea, 0xb4c95e75, 0xce54544d, 0x549a19a8,
  0x7f939376, 0x93ece53b, 0x2f222288, 0x220daa44,
  0x6364648d, 0x6407e9c8, 0x2af1f1e3, 0xf1db12ff,
  0xcc7373d1, 0x73bfa2e6, 0x82121248, 0x12905a24,
  0x7a40401d, 0x403a5d80, 0x48080820, 0x08402810,
  0x95c3c32b, 0xc356e89b, 0xdfecec97, 0xec337bc5,
  0x4ddbdb4b, 0xdb9690ab, 0xc0a1a1be, 0xa1611f5f,
  0x918d8d0e, 0x8d1c8307, 0xc83d3df4, 0x3df5c97a,
  0x5b979766, 0x97ccf133, 0x00000000, 0x00000000,
  0xf9cfcf1b, 0xcf36d483, 0x6e2b2bac, 0x2b458756,
  0xe17676c5, 0x7697b3ec, 0xe6828232, 0x8264b019,
  0x28d6d67f, 0xd6fea9b1, 0xc31b1b6c, 0x1bd87736,
  0x74b5b5ee, 0xb5c15b77, 0xbeafaf86, 0xaf112943,
  0x1d6a6ab5, 0x6a77dfd4, 0xea50505d, 0x50ba0da0,
  0x57454509, 0x45124c8a, 0x38f3f3eb, 0xf3cb18fb,
  0xad3030c0, 0x309df060, 0xc4efef9b, 0xef2b74c3,
  0xda3f3ffc, 0x3fe5c37e, 0xc7555549, 0x55921caa,
  0xdba2a2b2, 0xa2791059, 0xe9eaea8f, 0xea0365c9,
  0x6a656589, 0x650fecca, 0x03babad2, 0xbab96869,
  0x4a2f2fbc, 0x2f65935e, 0x8ec0c027, 0xc04ee79d,
  0x60dede5f, 0xdebe81a1, 0xfc1c1c70, 0x1ce06c38,
  0x46fdfdd3, 0xfdbb2ee7, 0x1f4d4d29, 0x4d52649a,
  0x76929272, 0x92e4e039, 0xfa7575c9, 0x758fbcea,
  0x36060618, 0x06301e0c, 0xae8a8a12, 0x8a249809,
  0x4bb2b2f2, 0xb2f94079, 0x85e6e6bf, 0xe66359d1,
  0x7e0e0e38, 0x0e70361c, 0xe71f1f7c, 0x1ff8633e,
  0x55626295, 0x6237f7c4, 0x3ad4d477, 0xd4eea3b5,
  0x81a8a89a, 0xa829324d, 0x52969662, 0x96c4f431,
  0x62f9f9c3, 0xf99b3aef, 0xa3c5c533, 0xc566f697,
  0x10252594, 0x2535b14a, 0xab595979, 0x59f220b2,
  0xd084842a, 0x8454ae15, 0xc57272d5, 0x72b7a7e4,
  0xec3939e4, 0x39d5dd72, 0x164c4c2d, 0x4c5a6198,
  0x945e5e65, 0x5eca3bbc, 0x9f7878fd, 0x78e785f0,
  0xe53838e0, 0x38ddd870, 0x988c8c0a, 0x8c148605,
  0x17d1d163, 0xd1c6b2bf, 0xe4a5a5ae, 0xa5410b57,
  0xa1e2e2af, 0xe2434dd9, 0x4e616199, 0x612ff8c2,
  0x42b3b3f6, 0xb3f1457b, 0x34212184, 0x2115a542,
  0x089c9c4a, 0x9c94d625, 0xee1e1e78, 0x1ef0663c,
  0x61434311, 0x43225286, 0xb1c7c73b, 0xc776fc93,
  0x4ffcfcd7, 0xfcb32be5, 0x24040410, 0x04201408,
  0xe3515159, 0x51b208a2, 0x2599995e, 0x99bcc72f,
  0x226d6da9, 0x6d4fc4da, 0x650d0d34, 0x0d68391a,
  0x79fafacf, 0xfa8335e9, 0x69dfdf5b, 0xdfb684a3,
  0xa97e7ee5, 0x7ed79bfc, 0x19242490, 0x243db448,
  0xfe3b3bec, 0x3bc5d776, 0x9aabab96, 0xab313d4b,
  0xf0cece1f, 0xce3ed181, 0x99111144, 0x11885522,
  0x838f8f06, 0x8f0c8903, 0x044e4e25, 0x4e4a6b9c,
  0x66b7b7e6, 0xb7d15173, 0xe0ebeb8b, 0xeb0b60cb,
  0xc13c3cf0, 0x3cfdcc78, 0xfd81813e, 0x817cbf1f,
  0x4094946a, 0x94d4fe35, 0x1cf7f7fb, 0xf7eb0cf3,
  0x18b9b9de, 0xb9a1676f, 0x8b13134c, 0x13985f26,
  0x512c2cb0, 0x2c7d9c58, 0x05d3d36b, 0xd3d6b8bb,
  0x8ce7e7bb, 0xe76b5cd3, 0x396e6ea5, 0x6e57cbdc,
  0xaac4c437, 0xc46ef395, 0x1b03030c, 0x03180f06,
  0xdc565645, 0x568a13ac, 0x5e44440d, 0x441a4988,
  0xa07f7fe1, 0x7fdf9efe, 0x88a9a99e, 0xa921374f,
  0x672a2aa8, 0x2a4d8254, 0x0abbbbd6, 0xbbb16d6b,
  0x87c1c123, 0xc146e29f, 0xf1535351, 0x53a202a6,
  0x72dcdc57, 0xdcae8ba5, 0x530b0b2c, 0x0b582716,
  0x019d9d4e, 0x9d9cd327, 0x2b6c6cad, 0x6c47c1d8,
  0xa43131c4, 0x3195f562, 0xf37474cd, 0x7487b9e8,
  0x15f6f6ff, 0xf6e309f1, 0x4c464605, 0x460a438c,
  0xa5acac8a, 0xac092645, 0xb589891e, 0x893c970f,
  0xb4141450, 0x14a04428, 0xbae1e1a3, 0xe15b42df,
  0xa6161658, 0x16b04e2c, 0xf73a3ae8, 0x3acdd274,
  0x066969b9, 0x696fd0d2, 0x41090924, 0x09482d12,
  0xd77070dd, 0x70a7ade0, 0x6fb6b6e2, 0xb6d95471,
  0x1ed0d067, 0xd0ceb7bd, 0xd6eded93, 0xed3b7ec7,
  0xe2cccc17, 0xcc2edb85, 0x68424215, 0x422a5784,
  0x2c98985a, 0x98b4c22d, 0xeda4a4aa, 0xa4490e55,
  0x752828a0, 0x285d8850, 0x865c5c6d, 0x5cda31b8,
  0x6bf8f8c7, 0xf8933fed, 0xc2868622, 0x8644a411
]);

const C2 = new Uint32Array([
  0x30d81818, 0x6018c078, 0x46262323, 0x8c2305af,
  0x91b8c6c6, 0x3fc67ef9, 0xcdfbe8e8, 0x87e8136f,
  0x13cb8787, 0x26874ca1, 0x6d11b8b8, 0xdab8a962,
  0x02090101, 0x04010805, 0x9e0d4f4f, 0x214f426e,
  0x6c9b3636, 0xd836adee, 0x51ffa6a6, 0xa2a65904,
  0xb90cd2d2, 0x6fd2debd, 0xf70ef5f5, 0xf3f5fb06,
  0xf2967979, 0xf979ef80, 0xde306f6f, 0xa16f5fce,
  0x3f6d9191, 0x7e91fcef, 0xa4f85252, 0x5552aa07,
  0xc0476060, 0x9d6027fd, 0x6535bcbc, 0xcabc8976,
  0x2b379b9b, 0x569baccd, 0x018a8e8e, 0x028e048c,
  0x5bd2a3a3, 0xb6a37115, 0x186c0c0c, 0x300c603c,
  0xf6847b7b, 0xf17bff8a, 0x6a803535, 0xd435b5e1,
  0x3af51d1d, 0x741de869, 0xddb3e0e0, 0xa7e05347,
  0xb321d7d7, 0x7bd7f6ac, 0x999cc2c2, 0x2fc25eed,
  0x5c432e2e, 0xb82e6d96, 0x96294b4b, 0x314b627a,
  0xe15dfefe, 0xdffea321, 0xaed55757, 0x41578216,
  0x2abd1515, 0x5415a841, 0xeee87777, 0xc1779fb6,
  0x6e923737, 0xdc37a5eb, 0xd79ee5e5, 0xb3e57b56,
  0x23139f9f, 0x469f8cd9, 0xfd23f0f0, 0xe7f0d317,
  0x94204a4a, 0x354a6a7f, 0xa944dada, 0x4fda9e95,
  0xb0a25858, 0x7d58fa25, 0x8fcfc9c9, 0x03c906ca,
  0x527c2929, 0xa429558d, 0x145a0a0a, 0x280a5022,
  0x7f50b1b1, 0xfeb1e14f, 0x5dc9a0a0, 0xbaa0691a,
  0xd6146b6b, 0xb16b7fda, 0x17d98585, 0x2e855cab,
  0x673cbdbd, 0xcebd8173, 0xba8f5d5d, 0x695dd234,
  0x20901010, 0x40108050, 0xf507f4f4, 0xf7f4f303,
  0x8bddcbcb, 0x0bcb16c0, 0x7cd33e3e, 0xf83eedc6,
  0x0a2d0505, 0x14052811, 0xce786767, 0x81671fe6,
  0xd597e4e4, 0xb7e47353, 0x4e022727, 0x9c2725bb,
  0x82734141, 0x19413258, 0x0ba78b8b, 0x168b2c9d,
  0x53f6a7a7, 0xa6a75101, 0xfab27d7d, 0xe97dcf94,
  0x37499595, 0x6e95dcfb, 0xad56d8d8, 0x47d88e9f,
  0xeb70fbfb, 0xcbfb8b30, 0xc1cdeeee, 0x9fee2371,
  0xf8bb7c7c, 0xed7cc791, 0xcc716666, 0x856617e3,
  0xa77bdddd, 0x53dda68e, 0x2eaf1717, 0x5c17b84b,
  0x8e454747, 0x01470246, 0x211a9e9e, 0x429e84dc,
  0x89d4caca, 0x0fca1ec5, 0x5a582d2d, 0xb42d7599,
  0x632ebfbf, 0xc6bf9179, 0x0e3f0707, 0x1c07381b,
  0x47acadad, 0x8ead0123, 0xb4b05a5a, 0x755aea2f,
  0x1bef8383, 0x36836cb5, 0x66b63333, 0xcc3385ff,
  0xc65c6363, 0x91633ff2, 0x04120202, 0x0802100a,
  0x4993aaaa, 0x92aa3938, 0xe2de7171, 0xd971afa8,
  0x8dc6c8c8, 0x07c80ecf, 0x32d11919, 0x6419c87d,
  0x923b4949, 0x39497270, 0xaf5fd9d9, 0x43d9869a,
  0xf931f2f2, 0xeff2c31d, 0xdba8e3e3, 0xabe34b48,
  0xb6b95b5b, 0x715be22a, 0x0dbc8888, 0x1a883492,
  0x293e9a9a, 0x529aa4c8, 0x4c0b2626, 0x98262dbe,
  0x64bf3232, 0xc8328dfa, 0x7d59b0b0, 0xfab0e94a,
  0xcff2e9e9, 0x83e91b6a, 0x1e770f0f, 0x3c0f7833,
  0xb733d5d5, 0x73d5e6a6, 0x1df48080, 0x3a8074ba,
  0x6127bebe, 0xc2be997c, 0x87ebcdcd, 0x13cd26de,
  0x68893434, 0xd034bde4, 0x90324848, 0x3d487a75,
  0xe354ffff, 0xdbffab24, 0xf48d7a7a, 0xf57af78f,
  0x3d649090, 0x7a90f4ea, 0xbe9d5f5f, 0x615fc23e,
  0x403d2020, 0x80201da0, 0xd00f6868, 0xbd6867d5,
  0x34ca1a1a, 0x681ad072, 0x41b7aeae, 0x82ae192c,
  0x757db4b4, 0xeab4c95e, 0xa8ce5454, 0x4d549a19,
  0x3b7f9393, 0x7693ece5, 0x442f2222, 0x88220daa,
  0xc8636464, 0x8d6407e9, 0xff2af1f1, 0xe3f1db12,
  0xe6cc7373, 0xd173bfa2, 0x24821212, 0x4812905a,
  0x807a4040, 0x1d403a5d, 0x10480808, 0x20084028,
  0x9b95c3c3, 0x2bc356e8, 0xc5dfecec, 0x97ec337b,
  0xab4ddbdb, 0x4bdb9690, 0x5fc0a1a1, 0xbea1611f,
  0x07918d8d, 0x0e8d1c83, 0x7ac83d3d, 0xf43df5c9,
  0x335b9797, 0x6697ccf1, 0x00000000, 0x00000000,
  0x83f9cfcf, 0x1bcf36d4, 0x566e2b2b, 0xac2b4587,
  0xece17676, 0xc57697b3, 0x19e68282, 0x328264b0,
  0xb128d6d6, 0x7fd6fea9, 0x36c31b1b, 0x6c1bd877,
  0x7774b5b5, 0xeeb5c15b, 0x43beafaf, 0x86af1129,
  0xd41d6a6a, 0xb56a77df, 0xa0ea5050, 0x5d50ba0d,
  0x8a574545, 0x0945124c, 0xfb38f3f3, 0xebf3cb18,
  0x60ad3030, 0xc0309df0, 0xc3c4efef, 0x9bef2b74,
  0x7eda3f3f, 0xfc3fe5c3, 0xaac75555, 0x4955921c,
  0x59dba2a2, 0xb2a27910, 0xc9e9eaea, 0x8fea0365,
  0xca6a6565, 0x89650fec, 0x6903baba, 0xd2bab968,
  0x5e4a2f2f, 0xbc2f6593, 0x9d8ec0c0, 0x27c04ee7,
  0xa160dede, 0x5fdebe81, 0x38fc1c1c, 0x701ce06c,
  0xe746fdfd, 0xd3fdbb2e, 0x9a1f4d4d, 0x294d5264,
  0x39769292, 0x7292e4e0, 0xeafa7575, 0xc9758fbc,
  0x0c360606, 0x1806301e, 0x09ae8a8a, 0x128a2498,
  0x794bb2b2, 0xf2b2f940, 0xd185e6e6, 0xbfe66359,
  0x1c7e0e0e, 0x380e7036, 0x3ee71f1f, 0x7c1ff863,
  0xc4556262, 0x956237f7, 0xb53ad4d4, 0x77d4eea3,
  0x4d81a8a8, 0x9aa82932, 0x31529696, 0x6296c4f4,
  0xef62f9f9, 0xc3f99b3a, 0x97a3c5c5, 0x33c566f6,
  0x4a102525, 0x942535b1, 0xb2ab5959, 0x7959f220,
  0x15d08484, 0x2a8454ae, 0xe4c57272, 0xd572b7a7,
  0x72ec3939, 0xe439d5dd, 0x98164c4c, 0x2d4c5a61,
  0xbc945e5e, 0x655eca3b, 0xf09f7878, 0xfd78e785,
  0x70e53838, 0xe038ddd8, 0x05988c8c, 0x0a8c1486,
  0xbf17d1d1, 0x63d1c6b2, 0x57e4a5a5, 0xaea5410b,
  0xd9a1e2e2, 0xafe2434d, 0xc24e6161, 0x99612ff8,
  0x7b42b3b3, 0xf6b3f145, 0x42342121, 0x842115a5,
  0x25089c9c, 0x4a9c94d6, 0x3cee1e1e, 0x781ef066,
  0x86614343, 0x11432252, 0x93b1c7c7, 0x3bc776fc,
  0xe54ffcfc, 0xd7fcb32b, 0x08240404, 0x10042014,
  0xa2e35151, 0x5951b208, 0x2f259999, 0x5e99bcc7,
  0xda226d6d, 0xa96d4fc4, 0x1a650d0d, 0x340d6839,
  0xe979fafa, 0xcffa8335, 0xa369dfdf, 0x5bdfb684,
  0xfca97e7e, 0xe57ed79b, 0x48192424, 0x90243db4,
  0x76fe3b3b, 0xec3bc5d7, 0x4b9aabab, 0x96ab313d,
  0x81f0cece, 0x1fce3ed1, 0x22991111, 0x44118855,
  0x03838f8f, 0x068f0c89, 0x9c044e4e, 0x254e4a6b,
  0x7366b7b7, 0xe6b7d151, 0xcbe0ebeb, 0x8beb0b60,
  0x78c13c3c, 0xf03cfdcc, 0x1ffd8181, 0x3e817cbf,
  0x35409494, 0x6a94d4fe, 0xf31cf7f7, 0xfbf7eb0c,
  0x6f18b9b9, 0xdeb9a167, 0x268b1313, 0x4c13985f,
  0x58512c2c, 0xb02c7d9c, 0xbb05d3d3, 0x6bd3d6b8,
  0xd38ce7e7, 0xbbe76b5c, 0xdc396e6e, 0xa56e57cb,
  0x95aac4c4, 0x37c46ef3, 0x061b0303, 0x0c03180f,
  0xacdc5656, 0x45568a13, 0x885e4444, 0x0d441a49,
  0xfea07f7f, 0xe17fdf9e, 0x4f88a9a9, 0x9ea92137,
  0x54672a2a, 0xa82a4d82, 0x6b0abbbb, 0xd6bbb16d,
  0x9f87c1c1, 0x23c146e2, 0xa6f15353, 0x5153a202,
  0xa572dcdc, 0x57dcae8b, 0x16530b0b, 0x2c0b5827,
  0x27019d9d, 0x4e9d9cd3, 0xd82b6c6c, 0xad6c47c1,
  0x62a43131, 0xc43195f5, 0xe8f37474, 0xcd7487b9,
  0xf115f6f6, 0xfff6e309, 0x8c4c4646, 0x05460a43,
  0x45a5acac, 0x8aac0926, 0x0fb58989, 0x1e893c97,
  0x28b41414, 0x5014a044, 0xdfbae1e1, 0xa3e15b42,
  0x2ca61616, 0x5816b04e, 0x74f73a3a, 0xe83acdd2,
  0xd2066969, 0xb9696fd0, 0x12410909, 0x2409482d,
  0xe0d77070, 0xdd70a7ad, 0x716fb6b6, 0xe2b6d954,
  0xbd1ed0d0, 0x67d0ceb7, 0xc7d6eded, 0x93ed3b7e,
  0x85e2cccc, 0x17cc2edb, 0x84684242, 0x15422a57,
  0x2d2c9898, 0x5a98b4c2, 0x55eda4a4, 0xaaa4490e,
  0x50752828, 0xa0285d88, 0xb8865c5c, 0x6d5cda31,
  0xed6bf8f8, 0xc7f8933f, 0x11c28686, 0x228644a4
]);

const C3 = new Uint32Array([
  0x7830d818, 0x186018c0, 0xaf462623, 0x238c2305,
  0xf991b8c6, 0xc63fc67e, 0x6fcdfbe8, 0xe887e813,
  0xa113cb87, 0x8726874c, 0x626d11b8, 0xb8dab8a9,
  0x05020901, 0x01040108, 0x6e9e0d4f, 0x4f214f42,
  0xee6c9b36, 0x36d836ad, 0x0451ffa6, 0xa6a2a659,
  0xbdb90cd2, 0xd26fd2de, 0x06f70ef5, 0xf5f3f5fb,
  0x80f29679, 0x79f979ef, 0xcede306f, 0x6fa16f5f,
  0xef3f6d91, 0x917e91fc, 0x07a4f852, 0x525552aa,
  0xfdc04760, 0x609d6027, 0x766535bc, 0xbccabc89,
  0xcd2b379b, 0x9b569bac, 0x8c018a8e, 0x8e028e04,
  0x155bd2a3, 0xa3b6a371, 0x3c186c0c, 0x0c300c60,
  0x8af6847b, 0x7bf17bff, 0xe16a8035, 0x35d435b5,
  0x693af51d, 0x1d741de8, 0x47ddb3e0, 0xe0a7e053,
  0xacb321d7, 0xd77bd7f6, 0xed999cc2, 0xc22fc25e,
  0x965c432e, 0x2eb82e6d, 0x7a96294b, 0x4b314b62,
  0x21e15dfe, 0xfedffea3, 0x16aed557, 0x57415782,
  0x412abd15, 0x155415a8, 0xb6eee877, 0x77c1779f,
  0xeb6e9237, 0x37dc37a5, 0x56d79ee5, 0xe5b3e57b,
  0xd923139f, 0x9f469f8c, 0x17fd23f0, 0xf0e7f0d3,
  0x7f94204a, 0x4a354a6a, 0x95a944da, 0xda4fda9e,
  0x25b0a258, 0x587d58fa, 0xca8fcfc9, 0xc903c906,
  0x8d527c29, 0x29a42955, 0x22145a0a, 0x0a280a50,
  0x4f7f50b1, 0xb1feb1e1, 0x1a5dc9a0, 0xa0baa069,
  0xdad6146b, 0x6bb16b7f, 0xab17d985, 0x852e855c,
  0x73673cbd, 0xbdcebd81, 0x34ba8f5d, 0x5d695dd2,
  0x50209010, 0x10401080, 0x03f507f4, 0xf4f7f4f3,
  0xc08bddcb, 0xcb0bcb16, 0xc67cd33e, 0x3ef83eed,
  0x110a2d05, 0x05140528, 0xe6ce7867, 0x6781671f,
  0x53d597e4, 0xe4b7e473, 0xbb4e0227, 0x279c2725,
  0x58827341, 0x41194132, 0x9d0ba78b, 0x8b168b2c,
  0x0153f6a7, 0xa7a6a751, 0x94fab27d, 0x7de97dcf,
  0xfb374995, 0x956e95dc, 0x9fad56d8, 0xd847d88e,
  0x30eb70fb, 0xfbcbfb8b, 0x71c1cdee, 0xee9fee23,
  0x91f8bb7c, 0x7ced7cc7, 0xe3cc7166, 0x66856617,
  0x8ea77bdd, 0xdd53dda6, 0x4b2eaf17, 0x175c17b8,
  0x468e4547, 0x47014702, 0xdc211a9e, 0x9e429e84,
  0xc589d4ca, 0xca0fca1e, 0x995a582d, 0x2db42d75,
  0x79632ebf, 0xbfc6bf91, 0x1b0e3f07, 0x071c0738,
  0x2347acad, 0xad8ead01, 0x2fb4b05a, 0x5a755aea,
  0xb51bef83, 0x8336836c, 0xff66b633, 0x33cc3385,
  0xf2c65c63, 0x6391633f, 0x0a041202, 0x02080210,
  0x384993aa, 0xaa92aa39, 0xa8e2de71, 0x71d971af,
  0xcf8dc6c8, 0xc807c80e, 0x7d32d119, 0x196419c8,
  0x70923b49, 0x49394972, 0x9aaf5fd9, 0xd943d986,
  0x1df931f2, 0xf2eff2c3, 0x48dba8e3, 0xe3abe34b,
  0x2ab6b95b, 0x5b715be2, 0x920dbc88, 0x881a8834,
  0xc8293e9a, 0x9a529aa4, 0xbe4c0b26, 0x2698262d,
  0xfa64bf32, 0x32c8328d, 0x4a7d59b0, 0xb0fab0e9,
  0x6acff2e9, 0xe983e91b, 0x331e770f, 0x0f3c0f78,
  0xa6b733d5, 0xd573d5e6, 0xba1df480, 0x803a8074,
  0x7c6127be, 0xbec2be99, 0xde87ebcd, 0xcd13cd26,
  0xe4688934, 0x34d034bd, 0x75903248, 0x483d487a,
  0x24e354ff, 0xffdbffab, 0x8ff48d7a, 0x7af57af7,
  0xea3d6490, 0x907a90f4, 0x3ebe9d5f, 0x5f615fc2,
  0xa0403d20, 0x2080201d, 0xd5d00f68, 0x68bd6867,
  0x7234ca1a, 0x1a681ad0, 0x2c41b7ae, 0xae82ae19,
  0x5e757db4, 0xb4eab4c9, 0x19a8ce54, 0x544d549a,
  0xe53b7f93, 0x937693ec, 0xaa442f22, 0x2288220d,
  0xe9c86364, 0x648d6407, 0x12ff2af1, 0xf1e3f1db,
  0xa2e6cc73, 0x73d173bf, 0x5a248212, 0x12481290,
  0x5d807a40, 0x401d403a, 0x28104808, 0x08200840,
  0xe89b95c3, 0xc32bc356, 0x7bc5dfec, 0xec97ec33,
  0x90ab4ddb, 0xdb4bdb96, 0x1f5fc0a1, 0xa1bea161,
  0x8307918d, 0x8d0e8d1c, 0xc97ac83d, 0x3df43df5,
  0xf1335b97, 0x976697cc, 0x00000000, 0x00000000,
  0xd483f9cf, 0xcf1bcf36, 0x87566e2b, 0x2bac2b45,
  0xb3ece176, 0x76c57697, 0xb019e682, 0x82328264,
  0xa9b128d6, 0xd67fd6fe, 0x7736c31b, 0x1b6c1bd8,
  0x5b7774b5, 0xb5eeb5c1, 0x2943beaf, 0xaf86af11,
  0xdfd41d6a, 0x6ab56a77, 0x0da0ea50, 0x505d50ba,
  0x4c8a5745, 0x45094512, 0x18fb38f3, 0xf3ebf3cb,
  0xf060ad30, 0x30c0309d, 0x74c3c4ef, 0xef9bef2b,
  0xc37eda3f, 0x3ffc3fe5, 0x1caac755, 0x55495592,
  0x1059dba2, 0xa2b2a279, 0x65c9e9ea, 0xea8fea03,
  0xecca6a65, 0x6589650f, 0x686903ba, 0xbad2bab9,
  0x935e4a2f, 0x2fbc2f65, 0xe79d8ec0, 0xc027c04e,
  0x81a160de, 0xde5fdebe, 0x6c38fc1c, 0x1c701ce0,
  0x2ee746fd, 0xfdd3fdbb, 0x649a1f4d, 0x4d294d52,
  0xe0397692, 0x927292e4, 0xbceafa75, 0x75c9758f,
  0x1e0c3606, 0x06180630, 0x9809ae8a, 0x8a128a24,
  0x40794bb2, 0xb2f2b2f9, 0x59d185e6, 0xe6bfe663,
  0x361c7e0e, 0x0e380e70, 0x633ee71f, 0x1f7c1ff8,
  0xf7c45562, 0x62956237, 0xa3b53ad4, 0xd477d4ee,
  0x324d81a8, 0xa89aa829, 0xf4315296, 0x966296c4,
  0x3aef62f9, 0xf9c3f99b, 0xf697a3c5, 0xc533c566,
  0xb14a1025, 0x25942535, 0x20b2ab59, 0x597959f2,
  0xae15d084, 0x842a8454, 0xa7e4c572, 0x72d572b7,
  0xdd72ec39, 0x39e439d5, 0x6198164c, 0x4c2d4c5a,
  0x3bbc945e, 0x5e655eca, 0x85f09f78, 0x78fd78e7,
  0xd870e538, 0x38e038dd, 0x8605988c, 0x8c0a8c14,
  0xb2bf17d1, 0xd163d1c6, 0x0b57e4a5, 0xa5aea541,
  0x4dd9a1e2, 0xe2afe243, 0xf8c24e61, 0x6199612f,
  0x457b42b3, 0xb3f6b3f1, 0xa5423421, 0x21842115,
  0xd625089c, 0x9c4a9c94, 0x663cee1e, 0x1e781ef0,
  0x52866143, 0x43114322, 0xfc93b1c7, 0xc73bc776,
  0x2be54ffc, 0xfcd7fcb3, 0x14082404, 0x04100420,
  0x08a2e351, 0x515951b2, 0xc72f2599, 0x995e99bc,
  0xc4da226d, 0x6da96d4f, 0x391a650d, 0x0d340d68,
  0x35e979fa, 0xfacffa83, 0x84a369df, 0xdf5bdfb6,
  0x9bfca97e, 0x7ee57ed7, 0xb4481924, 0x2490243d,
  0xd776fe3b, 0x3bec3bc5, 0x3d4b9aab, 0xab96ab31,
  0xd181f0ce, 0xce1fce3e, 0x55229911, 0x11441188,
  0x8903838f, 0x8f068f0c, 0x6b9c044e, 0x4e254e4a,
  0x517366b7, 0xb7e6b7d1, 0x60cbe0eb, 0xeb8beb0b,
  0xcc78c13c, 0x3cf03cfd, 0xbf1ffd81, 0x813e817c,
  0xfe354094, 0x946a94d4, 0x0cf31cf7, 0xf7fbf7eb,
  0x676f18b9, 0xb9deb9a1, 0x5f268b13, 0x134c1398,
  0x9c58512c, 0x2cb02c7d, 0xb8bb05d3, 0xd36bd3d6,
  0x5cd38ce7, 0xe7bbe76b, 0xcbdc396e, 0x6ea56e57,
  0xf395aac4, 0xc437c46e, 0x0f061b03, 0x030c0318,
  0x13acdc56, 0x5645568a, 0x49885e44, 0x440d441a,
  0x9efea07f, 0x7fe17fdf, 0x374f88a9, 0xa99ea921,
  0x8254672a, 0x2aa82a4d, 0x6d6b0abb, 0xbbd6bbb1,
  0xe29f87c1, 0xc123c146, 0x02a6f153, 0x535153a2,
  0x8ba572dc, 0xdc57dcae, 0x2716530b, 0x0b2c0b58,
  0xd327019d, 0x9d4e9d9c, 0xc1d82b6c, 0x6cad6c47,
  0xf562a431, 0x31c43195, 0xb9e8f374, 0x74cd7487,
  0x09f115f6, 0xf6fff6e3, 0x438c4c46, 0x4605460a,
  0x2645a5ac, 0xac8aac09, 0x970fb589, 0x891e893c,
  0x4428b414, 0x145014a0, 0x42dfbae1, 0xe1a3e15b,
  0x4e2ca616, 0x165816b0, 0xd274f73a, 0x3ae83acd,
  0xd0d20669, 0x69b9696f, 0x2d124109, 0x09240948,
  0xade0d770, 0x70dd70a7, 0x54716fb6, 0xb6e2b6d9,
  0xb7bd1ed0, 0xd067d0ce, 0x7ec7d6ed, 0xed93ed3b,
  0xdb85e2cc, 0xcc17cc2e, 0x57846842, 0x4215422a,
  0xc22d2c98, 0x985a98b4, 0x0e55eda4, 0xa4aaa449,
  0x88507528, 0x28a0285d, 0x31b8865c, 0x5c6d5cda,
  0x3fed6bf8, 0xf8c7f893, 0xa411c286, 0x86228644
]);

const C4 = new Uint32Array([
  0xc07830d8, 0x18186018, 0x05af4626, 0x23238c23,
  0x7ef991b8, 0xc6c63fc6, 0x136fcdfb, 0xe8e887e8,
  0x4ca113cb, 0x87872687, 0xa9626d11, 0xb8b8dab8,
  0x08050209, 0x01010401, 0x426e9e0d, 0x4f4f214f,
  0xadee6c9b, 0x3636d836, 0x590451ff, 0xa6a6a2a6,
  0xdebdb90c, 0xd2d26fd2, 0xfb06f70e, 0xf5f5f3f5,
  0xef80f296, 0x7979f979, 0x5fcede30, 0x6f6fa16f,
  0xfcef3f6d, 0x91917e91, 0xaa07a4f8, 0x52525552,
  0x27fdc047, 0x60609d60, 0x89766535, 0xbcbccabc,
  0xaccd2b37, 0x9b9b569b, 0x048c018a, 0x8e8e028e,
  0x71155bd2, 0xa3a3b6a3, 0x603c186c, 0x0c0c300c,
  0xff8af684, 0x7b7bf17b, 0xb5e16a80, 0x3535d435,
  0xe8693af5, 0x1d1d741d, 0x5347ddb3, 0xe0e0a7e0,
  0xf6acb321, 0xd7d77bd7, 0x5eed999c, 0xc2c22fc2,
  0x6d965c43, 0x2e2eb82e, 0x627a9629, 0x4b4b314b,
  0xa321e15d, 0xfefedffe, 0x8216aed5, 0x57574157,
  0xa8412abd, 0x15155415, 0x9fb6eee8, 0x7777c177,
  0xa5eb6e92, 0x3737dc37, 0x7b56d79e, 0xe5e5b3e5,
  0x8cd92313, 0x9f9f469f, 0xd317fd23, 0xf0f0e7f0,
  0x6a7f9420, 0x4a4a354a, 0x9e95a944, 0xdada4fda,
  0xfa25b0a2, 0x58587d58, 0x06ca8fcf, 0xc9c903c9,
  0x558d527c, 0x2929a429, 0x5022145a, 0x0a0a280a,
  0xe14f7f50, 0xb1b1feb1, 0x691a5dc9, 0xa0a0baa0,
  0x7fdad614, 0x6b6bb16b, 0x5cab17d9, 0x85852e85,
  0x8173673c, 0xbdbdcebd, 0xd234ba8f, 0x5d5d695d,
  0x80502090, 0x10104010, 0xf303f507, 0xf4f4f7f4,
  0x16c08bdd, 0xcbcb0bcb, 0xedc67cd3, 0x3e3ef83e,
  0x28110a2d, 0x05051405, 0x1fe6ce78, 0x67678167,
  0x7353d597, 0xe4e4b7e4, 0x25bb4e02, 0x27279c27,
  0x32588273, 0x41411941, 0x2c9d0ba7, 0x8b8b168b,
  0x510153f6, 0xa7a7a6a7, 0xcf94fab2, 0x7d7de97d,
  0xdcfb3749, 0x95956e95, 0x8e9fad56, 0xd8d847d8,
  0x8b30eb70, 0xfbfbcbfb, 0x2371c1cd, 0xeeee9fee,
  0xc791f8bb, 0x7c7ced7c, 0x17e3cc71, 0x66668566,
  0xa68ea77b, 0xdddd53dd, 0xb84b2eaf, 0x17175c17,
  0x02468e45, 0x47470147, 0x84dc211a, 0x9e9e429e,
  0x1ec589d4, 0xcaca0fca, 0x75995a58, 0x2d2db42d,
  0x9179632e, 0xbfbfc6bf, 0x381b0e3f, 0x07071c07,
  0x012347ac, 0xadad8ead, 0xea2fb4b0, 0x5a5a755a,
  0x6cb51bef, 0x83833683, 0x85ff66b6, 0x3333cc33,
  0x3ff2c65c, 0x63639163, 0x100a0412, 0x02020802,
  0x39384993, 0xaaaa92aa, 0xafa8e2de, 0x7171d971,
  0x0ecf8dc6, 0xc8c807c8, 0xc87d32d1, 0x19196419,
  0x7270923b, 0x49493949, 0x869aaf5f, 0xd9d943d9,
  0xc31df931, 0xf2f2eff2, 0x4b48dba8, 0xe3e3abe3,
  0xe22ab6b9, 0x5b5b715b, 0x34920dbc, 0x88881a88,
  0xa4c8293e, 0x9a9a529a, 0x2dbe4c0b, 0x26269826,
  0x8dfa64bf, 0x3232c832, 0xe94a7d59, 0xb0b0fab0,
  0x1b6acff2, 0xe9e983e9, 0x78331e77, 0x0f0f3c0f,
  0xe6a6b733, 0xd5d573d5, 0x74ba1df4, 0x80803a80,
  0x997c6127, 0xbebec2be, 0x26de87eb, 0xcdcd13cd,
  0xbde46889, 0x3434d034, 0x7a759032, 0x48483d48,
  0xab24e354, 0xffffdbff, 0xf78ff48d, 0x7a7af57a,
  0xf4ea3d64, 0x90907a90, 0xc23ebe9d, 0x5f5f615f,
  0x1da0403d, 0x20208020, 0x67d5d00f, 0x6868bd68,
  0xd07234ca, 0x1a1a681a, 0x192c41b7, 0xaeae82ae,
  0xc95e757d, 0xb4b4eab4, 0x9a19a8ce, 0x54544d54,
  0xece53b7f, 0x93937693, 0x0daa442f, 0x22228822,
  0x07e9c863, 0x64648d64, 0xdb12ff2a, 0xf1f1e3f1,
  0xbfa2e6cc, 0x7373d173, 0x905a2482, 0x12124812,
  0x3a5d807a, 0x40401d40, 0x40281048, 0x08082008,
  0x56e89b95, 0xc3c32bc3, 0x337bc5df, 0xecec97ec,
  0x9690ab4d, 0xdbdb4bdb, 0x611f5fc0, 0xa1a1bea1,
  0x1c830791, 0x8d8d0e8d, 0xf5c97ac8, 0x3d3df43d,
  0xccf1335b, 0x97976697, 0x00000000, 0x00000000,
  0x36d483f9, 0xcfcf1bcf, 0x4587566e, 0x2b2bac2b,
  0x97b3ece1, 0x7676c576, 0x64b019e6, 0x82823282,
  0xfea9b128, 0xd6d67fd6, 0xd87736c3, 0x1b1b6c1b,
  0xc15b7774, 0xb5b5eeb5, 0x112943be, 0xafaf86af,
  0x77dfd41d, 0x6a6ab56a, 0xba0da0ea, 0x50505d50,
  0x124c8a57, 0x45450945, 0xcb18fb38, 0xf3f3ebf3,
  0x9df060ad, 0x3030c030, 0x2b74c3c4, 0xefef9bef,
  0xe5c37eda, 0x3f3ffc3f, 0x921caac7, 0x55554955,
  0x791059db, 0xa2a2b2a2, 0x0365c9e9, 0xeaea8fea,
  0x0fecca6a, 0x65658965, 0xb9686903, 0xbabad2ba,
  0x65935e4a, 0x2f2fbc2f, 0x4ee79d8e, 0xc0c027c0,
  0xbe81a160, 0xdede5fde, 0xe06c38fc, 0x1c1c701c,
  0xbb2ee746, 0xfdfdd3fd, 0x52649a1f, 0x4d4d294d,
  0xe4e03976, 0x92927292, 0x8fbceafa, 0x7575c975,
  0x301e0c36, 0x06061806, 0x249809ae, 0x8a8a128a,
  0xf940794b, 0xb2b2f2b2, 0x6359d185, 0xe6e6bfe6,
  0x70361c7e, 0x0e0e380e, 0xf8633ee7, 0x1f1f7c1f,
  0x37f7c455, 0x62629562, 0xeea3b53a, 0xd4d477d4,
  0x29324d81, 0xa8a89aa8, 0xc4f43152, 0x96966296,
  0x9b3aef62, 0xf9f9c3f9, 0x66f697a3, 0xc5c533c5,
  0x35b14a10, 0x25259425, 0xf220b2ab, 0x59597959,
  0x54ae15d0, 0x84842a84, 0xb7a7e4c5, 0x7272d572,
  0xd5dd72ec, 0x3939e439, 0x5a619816, 0x4c4c2d4c,
  0xca3bbc94, 0x5e5e655e, 0xe785f09f, 0x7878fd78,
  0xddd870e5, 0x3838e038, 0x14860598, 0x8c8c0a8c,
  0xc6b2bf17, 0xd1d163d1, 0x410b57e4, 0xa5a5aea5,
  0x434dd9a1, 0xe2e2afe2, 0x2ff8c24e, 0x61619961,
  0xf1457b42, 0xb3b3f6b3, 0x15a54234, 0x21218421,
  0x94d62508, 0x9c9c4a9c, 0xf0663cee, 0x1e1e781e,
  0x22528661, 0x43431143, 0x76fc93b1, 0xc7c73bc7,
  0xb32be54f, 0xfcfcd7fc, 0x20140824, 0x04041004,
  0xb208a2e3, 0x51515951, 0xbcc72f25, 0x99995e99,
  0x4fc4da22, 0x6d6da96d, 0x68391a65, 0x0d0d340d,
  0x8335e979, 0xfafacffa, 0xb684a369, 0xdfdf5bdf,
  0xd79bfca9, 0x7e7ee57e, 0x3db44819, 0x24249024,
  0xc5d776fe, 0x3b3bec3b, 0x313d4b9a, 0xabab96ab,
  0x3ed181f0, 0xcece1fce, 0x88552299, 0x11114411,
  0x0c890383, 0x8f8f068f, 0x4a6b9c04, 0x4e4e254e,
  0xd1517366, 0xb7b7e6b7, 0x0b60cbe0, 0xebeb8beb,
  0xfdcc78c1, 0x3c3cf03c, 0x7cbf1ffd, 0x81813e81,
  0xd4fe3540, 0x94946a94, 0xeb0cf31c, 0xf7f7fbf7,
  0xa1676f18, 0xb9b9deb9, 0x985f268b, 0x13134c13,
  0x7d9c5851, 0x2c2cb02c, 0xd6b8bb05, 0xd3d36bd3,
  0x6b5cd38c, 0xe7e7bbe7, 0x57cbdc39, 0x6e6ea56e,
  0x6ef395aa, 0xc4c437c4, 0x180f061b, 0x03030c03,
  0x8a13acdc, 0x56564556, 0x1a49885e, 0x44440d44,
  0xdf9efea0, 0x7f7fe17f, 0x21374f88, 0xa9a99ea9,
  0x4d825467, 0x2a2aa82a, 0xb16d6b0a, 0xbbbbd6bb,
  0x46e29f87, 0xc1c123c1, 0xa202a6f1, 0x53535153,
  0xae8ba572, 0xdcdc57dc, 0x58271653, 0x0b0b2c0b,
  0x9cd32701, 0x9d9d4e9d, 0x47c1d82b, 0x6c6cad6c,
  0x95f562a4, 0x3131c431, 0x87b9e8f3, 0x7474cd74,
  0xe309f115, 0xf6f6fff6, 0x0a438c4c, 0x46460546,
  0x092645a5, 0xacac8aac, 0x3c970fb5, 0x89891e89,
  0xa04428b4, 0x14145014, 0x5b42dfba, 0xe1e1a3e1,
  0xb04e2ca6, 0x16165816, 0xcdd274f7, 0x3a3ae83a,
  0x6fd0d206, 0x6969b969, 0x482d1241, 0x09092409,
  0xa7ade0d7, 0x7070dd70, 0xd954716f, 0xb6b6e2b6,
  0xceb7bd1e, 0xd0d067d0, 0x3b7ec7d6, 0xeded93ed,
  0x2edb85e2, 0xcccc17cc, 0x2a578468, 0x42421542,
  0xb4c22d2c, 0x98985a98, 0x490e55ed, 0xa4a4aaa4,
  0x5d885075, 0x2828a028, 0xda31b886, 0x5c5c6d5c,
  0x933fed6b, 0xf8f8c7f8, 0x44a411c2, 0x86862286
]);

const C5 = new Uint32Array([
  0x18c07830, 0xd8181860, 0x2305af46, 0x2623238c,
  0xc67ef991, 0xb8c6c63f, 0xe8136fcd, 0xfbe8e887,
  0x874ca113, 0xcb878726, 0xb8a9626d, 0x11b8b8da,
  0x01080502, 0x09010104, 0x4f426e9e, 0x0d4f4f21,
  0x36adee6c, 0x9b3636d8, 0xa6590451, 0xffa6a6a2,
  0xd2debdb9, 0x0cd2d26f, 0xf5fb06f7, 0x0ef5f5f3,
  0x79ef80f2, 0x967979f9, 0x6f5fcede, 0x306f6fa1,
  0x91fcef3f, 0x6d91917e, 0x52aa07a4, 0xf8525255,
  0x6027fdc0, 0x4760609d, 0xbc897665, 0x35bcbcca,
  0x9baccd2b, 0x379b9b56, 0x8e048c01, 0x8a8e8e02,
  0xa371155b, 0xd2a3a3b6, 0x0c603c18, 0x6c0c0c30,
  0x7bff8af6, 0x847b7bf1, 0x35b5e16a, 0x803535d4,
  0x1de8693a, 0xf51d1d74, 0xe05347dd, 0xb3e0e0a7,
  0xd7f6acb3, 0x21d7d77b, 0xc25eed99, 0x9cc2c22f,
  0x2e6d965c, 0x432e2eb8, 0x4b627a96, 0x294b4b31,
  0xfea321e1, 0x5dfefedf, 0x578216ae, 0xd5575741,
  0x15a8412a, 0xbd151554, 0x779fb6ee, 0xe87777c1,
  0x37a5eb6e, 0x923737dc, 0xe57b56d7, 0x9ee5e5b3,
  0x9f8cd923, 0x139f9f46, 0xf0d317fd, 0x23f0f0e7,
  0x4a6a7f94, 0x204a4a35, 0xda9e95a9, 0x44dada4f,
  0x58fa25b0, 0xa258587d, 0xc906ca8f, 0xcfc9c903,
  0x29558d52, 0x7c2929a4, 0x0a502214, 0x5a0a0a28,
  0xb1e14f7f, 0x50b1b1fe, 0xa0691a5d, 0xc9a0a0ba,
  0x6b7fdad6, 0x146b6bb1, 0x855cab17, 0xd985852e,
  0xbd817367, 0x3cbdbdce, 0x5dd234ba, 0x8f5d5d69,
  0x10805020, 0x90101040, 0xf4f303f5, 0x07f4f4f7,
  0xcb16c08b, 0xddcbcb0b, 0x3eedc67c, 0xd33e3ef8,
  0x0528110a, 0x2d050514, 0x671fe6ce, 0x78676781,
  0xe47353d5, 0x97e4e4b7, 0x2725bb4e, 0x0227279c,
  0x41325882, 0x73414119, 0x8b2c9d0b, 0xa78b8b16,
  0xa7510153, 0xf6a7a7a6, 0x7dcf94fa, 0xb27d7de9,
  0x95dcfb37, 0x4995956e, 0xd88e9fad, 0x56d8d847,
  0xfb8b30eb, 0x70fbfbcb, 0xee2371c1, 0xcdeeee9f,
  0x7cc791f8, 0xbb7c7ced, 0x6617e3cc, 0x71666685,
  0xdda68ea7, 0x7bdddd53, 0x17b84b2e, 0xaf17175c,
  0x4702468e, 0x45474701, 0x9e84dc21, 0x1a9e9e42,
  0xca1ec589, 0xd4caca0f, 0x2d75995a, 0x582d2db4,
  0xbf917963, 0x2ebfbfc6, 0x07381b0e, 0x3f07071c,
  0xad012347, 0xacadad8e, 0x5aea2fb4, 0xb05a5a75,
  0x836cb51b, 0xef838336, 0x3385ff66, 0xb63333cc,
  0x633ff2c6, 0x5c636391, 0x02100a04, 0x12020208,
  0xaa393849, 0x93aaaa92, 0x71afa8e2, 0xde7171d9,
  0xc80ecf8d, 0xc6c8c807, 0x19c87d32, 0xd1191964,
  0x49727092, 0x3b494939, 0xd9869aaf, 0x5fd9d943,
  0xf2c31df9, 0x31f2f2ef, 0xe34b48db, 0xa8e3e3ab,
  0x5be22ab6, 0xb95b5b71, 0x8834920d, 0xbc88881a,
  0x9aa4c829, 0x3e9a9a52, 0x262dbe4c, 0x0b262698,
  0x328dfa64, 0xbf3232c8, 0xb0e94a7d, 0x59b0b0fa,
  0xe91b6acf, 0xf2e9e983, 0x0f78331e, 0x770f0f3c,
  0xd5e6a6b7, 0x33d5d573, 0x8074ba1d, 0xf480803a,
  0xbe997c61, 0x27bebec2, 0xcd26de87, 0xebcdcd13,
  0x34bde468, 0x893434d0, 0x487a7590, 0x3248483d,
  0xffab24e3, 0x54ffffdb, 0x7af78ff4, 0x8d7a7af5,
  0x90f4ea3d, 0x6490907a, 0x5fc23ebe, 0x9d5f5f61,
  0x201da040, 0x3d202080, 0x6867d5d0, 0x0f6868bd,
  0x1ad07234, 0xca1a1a68, 0xae192c41, 0xb7aeae82,
  0xb4c95e75, 0x7db4b4ea, 0x549a19a8, 0xce54544d,
  0x93ece53b, 0x7f939376, 0x220daa44, 0x2f222288,
  0x6407e9c8, 0x6364648d, 0xf1db12ff, 0x2af1f1e3,
  0x73bfa2e6, 0xcc7373d1, 0x12905a24, 0x82121248,
  0x403a5d80, 0x7a40401d, 0x08402810, 0x48080820,
  0xc356e89b, 0x95c3c32b, 0xec337bc5, 0xdfecec97,
  0xdb9690ab, 0x4ddbdb4b, 0xa1611f5f, 0xc0a1a1be,
  0x8d1c8307, 0x918d8d0e, 0x3df5c97a, 0xc83d3df4,
  0x97ccf133, 0x5b979766, 0x00000000, 0x00000000,
  0xcf36d483, 0xf9cfcf1b, 0x2b458756, 0x6e2b2bac,
  0x7697b3ec, 0xe17676c5, 0x8264b019, 0xe6828232,
  0xd6fea9b1, 0x28d6d67f, 0x1bd87736, 0xc31b1b6c,
  0xb5c15b77, 0x74b5b5ee, 0xaf112943, 0xbeafaf86,
  0x6a77dfd4, 0x1d6a6ab5, 0x50ba0da0, 0xea50505d,
  0x45124c8a, 0x57454509, 0xf3cb18fb, 0x38f3f3eb,
  0x309df060, 0xad3030c0, 0xef2b74c3, 0xc4efef9b,
  0x3fe5c37e, 0xda3f3ffc, 0x55921caa, 0xc7555549,
  0xa2791059, 0xdba2a2b2, 0xea0365c9, 0xe9eaea8f,
  0x650fecca, 0x6a656589, 0xbab96869, 0x03babad2,
  0x2f65935e, 0x4a2f2fbc, 0xc04ee79d, 0x8ec0c027,
  0xdebe81a1, 0x60dede5f, 0x1ce06c38, 0xfc1c1c70,
  0xfdbb2ee7, 0x46fdfdd3, 0x4d52649a, 0x1f4d4d29,
  0x92e4e039, 0x76929272, 0x758fbcea, 0xfa7575c9,
  0x06301e0c, 0x36060618, 0x8a249809, 0xae8a8a12,
  0xb2f94079, 0x4bb2b2f2, 0xe66359d1, 0x85e6e6bf,
  0x0e70361c, 0x7e0e0e38, 0x1ff8633e, 0xe71f1f7c,
  0x6237f7c4, 0x55626295, 0xd4eea3b5, 0x3ad4d477,
  0xa829324d, 0x81a8a89a, 0x96c4f431, 0x52969662,
  0xf99b3aef, 0x62f9f9c3, 0xc566f697, 0xa3c5c533,
  0x2535b14a, 0x10252594, 0x59f220b2, 0xab595979,
  0x8454ae15, 0xd084842a, 0x72b7a7e4, 0xc57272d5,
  0x39d5dd72, 0xec3939e4, 0x4c5a6198, 0x164c4c2d,
  0x5eca3bbc, 0x945e5e65, 0x78e785f0, 0x9f7878fd,
  0x38ddd870, 0xe53838e0, 0x8c148605, 0x988c8c0a,
  0xd1c6b2bf, 0x17d1d163, 0xa5410b57, 0xe4a5a5ae,
  0xe2434dd9, 0xa1e2e2af, 0x612ff8c2, 0x4e616199,
  0xb3f1457b, 0x42b3b3f6, 0x2115a542, 0x34212184,
  0x9c94d625, 0x089c9c4a, 0x1ef0663c, 0xee1e1e78,
  0x43225286, 0x61434311, 0xc776fc93, 0xb1c7c73b,
  0xfcb32be5, 0x4ffcfcd7, 0x04201408, 0x24040410,
  0x51b208a2, 0xe3515159, 0x99bcc72f, 0x2599995e,
  0x6d4fc4da, 0x226d6da9, 0x0d68391a, 0x650d0d34,
  0xfa8335e9, 0x79fafacf, 0xdfb684a3, 0x69dfdf5b,
  0x7ed79bfc, 0xa97e7ee5, 0x243db448, 0x19242490,
  0x3bc5d776, 0xfe3b3bec, 0xab313d4b, 0x9aabab96,
  0xce3ed181, 0xf0cece1f, 0x11885522, 0x99111144,
  0x8f0c8903, 0x838f8f06, 0x4e4a6b9c, 0x044e4e25,
  0xb7d15173, 0x66b7b7e6, 0xeb0b60cb, 0xe0ebeb8b,
  0x3cfdcc78, 0xc13c3cf0, 0x817cbf1f, 0xfd81813e,
  0x94d4fe35, 0x4094946a, 0xf7eb0cf3, 0x1cf7f7fb,
  0xb9a1676f, 0x18b9b9de, 0x13985f26, 0x8b13134c,
  0x2c7d9c58, 0x512c2cb0, 0xd3d6b8bb, 0x05d3d36b,
  0xe76b5cd3, 0x8ce7e7bb, 0x6e57cbdc, 0x396e6ea5,
  0xc46ef395, 0xaac4c437, 0x03180f06, 0x1b03030c,
  0x568a13ac, 0xdc565645, 0x441a4988, 0x5e44440d,
  0x7fdf9efe, 0xa07f7fe1, 0xa921374f, 0x88a9a99e,
  0x2a4d8254, 0x672a2aa8, 0xbbb16d6b, 0x0abbbbd6,
  0xc146e29f, 0x87c1c123, 0x53a202a6, 0xf1535351,
  0xdcae8ba5, 0x72dcdc57, 0x0b582716, 0x530b0b2c,
  0x9d9cd327, 0x019d9d4e, 0x6c47c1d8, 0x2b6c6cad,
  0x3195f562, 0xa43131c4, 0x7487b9e8, 0xf37474cd,
  0xf6e309f1, 0x15f6f6ff, 0x460a438c, 0x4c464605,
  0xac092645, 0xa5acac8a, 0x893c970f, 0xb589891e,
  0x14a04428, 0xb4141450, 0xe15b42df, 0xbae1e1a3,
  0x16b04e2c, 0xa6161658, 0x3acdd274, 0xf73a3ae8,
  0x696fd0d2, 0x066969b9, 0x09482d12, 0x41090924,
  0x70a7ade0, 0xd77070dd, 0xb6d95471, 0x6fb6b6e2,
  0xd0ceb7bd, 0x1ed0d067, 0xed3b7ec7, 0xd6eded93,
  0xcc2edb85, 0xe2cccc17, 0x422a5784, 0x68424215,
  0x98b4c22d, 0x2c98985a, 0xa4490e55, 0xeda4a4aa,
  0x285d8850, 0x752828a0, 0x5cda31b8, 0x865c5c6d,
  0xf8933fed, 0x6bf8f8c7, 0x8644a411, 0xc2868622
]);

const C6 = new Uint32Array([
  0x6018c078, 0x30d81818, 0x8c2305af, 0x46262323,
  0x3fc67ef9, 0x91b8c6c6, 0x87e8136f, 0xcdfbe8e8,
  0x26874ca1, 0x13cb8787, 0xdab8a962, 0x6d11b8b8,
  0x04010805, 0x02090101, 0x214f426e, 0x9e0d4f4f,
  0xd836adee, 0x6c9b3636, 0xa2a65904, 0x51ffa6a6,
  0x6fd2debd, 0xb90cd2d2, 0xf3f5fb06, 0xf70ef5f5,
  0xf979ef80, 0xf2967979, 0xa16f5fce, 0xde306f6f,
  0x7e91fcef, 0x3f6d9191, 0x5552aa07, 0xa4f85252,
  0x9d6027fd, 0xc0476060, 0xcabc8976, 0x6535bcbc,
  0x569baccd, 0x2b379b9b, 0x028e048c, 0x018a8e8e,
  0xb6a37115, 0x5bd2a3a3, 0x300c603c, 0x186c0c0c,
  0xf17bff8a, 0xf6847b7b, 0xd435b5e1, 0x6a803535,
  0x741de869, 0x3af51d1d, 0xa7e05347, 0xddb3e0e0,
  0x7bd7f6ac, 0xb321d7d7, 0x2fc25eed, 0x999cc2c2,
  0xb82e6d96, 0x5c432e2e, 0x314b627a, 0x96294b4b,
  0xdffea321, 0xe15dfefe, 0x41578216, 0xaed55757,
  0x5415a841, 0x2abd1515, 0xc1779fb6, 0xeee87777,
  0xdc37a5eb, 0x6e923737, 0xb3e57b56, 0xd79ee5e5,
  0x469f8cd9, 0x23139f9f, 0xe7f0d317, 0xfd23f0f0,
  0x354a6a7f, 0x94204a4a, 0x4fda9e95, 0xa944dada,
  0x7d58fa25, 0xb0a25858, 0x03c906ca, 0x8fcfc9c9,
  0xa429558d, 0x527c2929, 0x280a5022, 0x145a0a0a,
  0xfeb1e14f, 0x7f50b1b1, 0xbaa0691a, 0x5dc9a0a0,
  0xb16b7fda, 0xd6146b6b, 0x2e855cab, 0x17d98585,
  0xcebd8173, 0x673cbdbd, 0x695dd234, 0xba8f5d5d,
  0x40108050, 0x20901010, 0xf7f4f303, 0xf507f4f4,
  0x0bcb16c0, 0x8bddcbcb, 0xf83eedc6, 0x7cd33e3e,
  0x14052811, 0x0a2d0505, 0x81671fe6, 0xce786767,
  0xb7e47353, 0xd597e4e4, 0x9c2725bb, 0x4e022727,
  0x19413258, 0x82734141, 0x168b2c9d, 0x0ba78b8b,
  0xa6a75101, 0x53f6a7a7, 0xe97dcf94, 0xfab27d7d,
  0x6e95dcfb, 0x37499595, 0x47d88e9f, 0xad56d8d8,
  0xcbfb8b30, 0xeb70fbfb, 0x9fee2371, 0xc1cdeeee,
  0xed7cc791, 0xf8bb7c7c, 0x856617e3, 0xcc716666,
  0x53dda68e, 0xa77bdddd, 0x5c17b84b, 0x2eaf1717,
  0x01470246, 0x8e454747, 0x429e84dc, 0x211a9e9e,
  0x0fca1ec5, 0x89d4caca, 0xb42d7599, 0x5a582d2d,
  0xc6bf9179, 0x632ebfbf, 0x1c07381b, 0x0e3f0707,
  0x8ead0123, 0x47acadad, 0x755aea2f, 0xb4b05a5a,
  0x36836cb5, 0x1bef8383, 0xcc3385ff, 0x66b63333,
  0x91633ff2, 0xc65c6363, 0x0802100a, 0x04120202,
  0x92aa3938, 0x4993aaaa, 0xd971afa8, 0xe2de7171,
  0x07c80ecf, 0x8dc6c8c8, 0x6419c87d, 0x32d11919,
  0x39497270, 0x923b4949, 0x43d9869a, 0xaf5fd9d9,
  0xeff2c31d, 0xf931f2f2, 0xabe34b48, 0xdba8e3e3,
  0x715be22a, 0xb6b95b5b, 0x1a883492, 0x0dbc8888,
  0x529aa4c8, 0x293e9a9a, 0x98262dbe, 0x4c0b2626,
  0xc8328dfa, 0x64bf3232, 0xfab0e94a, 0x7d59b0b0,
  0x83e91b6a, 0xcff2e9e9, 0x3c0f7833, 0x1e770f0f,
  0x73d5e6a6, 0xb733d5d5, 0x3a8074ba, 0x1df48080,
  0xc2be997c, 0x6127bebe, 0x13cd26de, 0x87ebcdcd,
  0xd034bde4, 0x68893434, 0x3d487a75, 0x90324848,
  0xdbffab24, 0xe354ffff, 0xf57af78f, 0xf48d7a7a,
  0x7a90f4ea, 0x3d649090, 0x615fc23e, 0xbe9d5f5f,
  0x80201da0, 0x403d2020, 0xbd6867d5, 0xd00f6868,
  0x681ad072, 0x34ca1a1a, 0x82ae192c, 0x41b7aeae,
  0xeab4c95e, 0x757db4b4, 0x4d549a19, 0xa8ce5454,
  0x7693ece5, 0x3b7f9393, 0x88220daa, 0x442f2222,
  0x8d6407e9, 0xc8636464, 0xe3f1db12, 0xff2af1f1,
  0xd173bfa2, 0xe6cc7373, 0x4812905a, 0x24821212,
  0x1d403a5d, 0x807a4040, 0x20084028, 0x10480808,
  0x2bc356e8, 0x9b95c3c3, 0x97ec337b, 0xc5dfecec,
  0x4bdb9690, 0xab4ddbdb, 0xbea1611f, 0x5fc0a1a1,
  0x0e8d1c83, 0x07918d8d, 0xf43df5c9, 0x7ac83d3d,
  0x6697ccf1, 0x335b9797, 0x00000000, 0x00000000,
  0x1bcf36d4, 0x83f9cfcf, 0xac2b4587, 0x566e2b2b,
  0xc57697b3, 0xece17676, 0x328264b0, 0x19e68282,
  0x7fd6fea9, 0xb128d6d6, 0x6c1bd877, 0x36c31b1b,
  0xeeb5c15b, 0x7774b5b5, 0x86af1129, 0x43beafaf,
  0xb56a77df, 0xd41d6a6a, 0x5d50ba0d, 0xa0ea5050,
  0x0945124c, 0x8a574545, 0xebf3cb18, 0xfb38f3f3,
  0xc0309df0, 0x60ad3030, 0x9bef2b74, 0xc3c4efef,
  0xfc3fe5c3, 0x7eda3f3f, 0x4955921c, 0xaac75555,
  0xb2a27910, 0x59dba2a2, 0x8fea0365, 0xc9e9eaea,
  0x89650fec, 0xca6a6565, 0xd2bab968, 0x6903baba,
  0xbc2f6593, 0x5e4a2f2f, 0x27c04ee7, 0x9d8ec0c0,
  0x5fdebe81, 0xa160dede, 0x701ce06c, 0x38fc1c1c,
  0xd3fdbb2e, 0xe746fdfd, 0x294d5264, 0x9a1f4d4d,
  0x7292e4e0, 0x39769292, 0xc9758fbc, 0xeafa7575,
  0x1806301e, 0x0c360606, 0x128a2498, 0x09ae8a8a,
  0xf2b2f940, 0x794bb2b2, 0xbfe66359, 0xd185e6e6,
  0x380e7036, 0x1c7e0e0e, 0x7c1ff863, 0x3ee71f1f,
  0x956237f7, 0xc4556262, 0x77d4eea3, 0xb53ad4d4,
  0x9aa82932, 0x4d81a8a8, 0x6296c4f4, 0x31529696,
  0xc3f99b3a, 0xef62f9f9, 0x33c566f6, 0x97a3c5c5,
  0x942535b1, 0x4a102525, 0x7959f220, 0xb2ab5959,
  0x2a8454ae, 0x15d08484, 0xd572b7a7, 0xe4c57272,
  0xe439d5dd, 0x72ec3939, 0x2d4c5a61, 0x98164c4c,
  0x655eca3b, 0xbc945e5e, 0xfd78e785, 0xf09f7878,
  0xe038ddd8, 0x70e53838, 0x0a8c1486, 0x05988c8c,
  0x63d1c6b2, 0xbf17d1d1, 0xaea5410b, 0x57e4a5a5,
  0xafe2434d, 0xd9a1e2e2, 0x99612ff8, 0xc24e6161,
  0xf6b3f145, 0x7b42b3b3, 0x842115a5, 0x42342121,
  0x4a9c94d6, 0x25089c9c, 0x781ef066, 0x3cee1e1e,
  0x11432252, 0x86614343, 0x3bc776fc, 0x93b1c7c7,
  0xd7fcb32b, 0xe54ffcfc, 0x10042014, 0x08240404,
  0x5951b208, 0xa2e35151, 0x5e99bcc7, 0x2f259999,
  0xa96d4fc4, 0xda226d6d, 0x340d6839, 0x1a650d0d,
  0xcffa8335, 0xe979fafa, 0x5bdfb684, 0xa369dfdf,
  0xe57ed79b, 0xfca97e7e, 0x90243db4, 0x48192424,
  0xec3bc5d7, 0x76fe3b3b, 0x96ab313d, 0x4b9aabab,
  0x1fce3ed1, 0x81f0cece, 0x44118855, 0x22991111,
  0x068f0c89, 0x03838f8f, 0x254e4a6b, 0x9c044e4e,
  0xe6b7d151, 0x7366b7b7, 0x8beb0b60, 0xcbe0ebeb,
  0xf03cfdcc, 0x78c13c3c, 0x3e817cbf, 0x1ffd8181,
  0x6a94d4fe, 0x35409494, 0xfbf7eb0c, 0xf31cf7f7,
  0xdeb9a167, 0x6f18b9b9, 0x4c13985f, 0x268b1313,
  0xb02c7d9c, 0x58512c2c, 0x6bd3d6b8, 0xbb05d3d3,
  0xbbe76b5c, 0xd38ce7e7, 0xa56e57cb, 0xdc396e6e,
  0x37c46ef3, 0x95aac4c4, 0x0c03180f, 0x061b0303,
  0x45568a13, 0xacdc5656, 0x0d441a49, 0x885e4444,
  0xe17fdf9e, 0xfea07f7f, 0x9ea92137, 0x4f88a9a9,
  0xa82a4d82, 0x54672a2a, 0xd6bbb16d, 0x6b0abbbb,
  0x23c146e2, 0x9f87c1c1, 0x5153a202, 0xa6f15353,
  0x57dcae8b, 0xa572dcdc, 0x2c0b5827, 0x16530b0b,
  0x4e9d9cd3, 0x27019d9d, 0xad6c47c1, 0xd82b6c6c,
  0xc43195f5, 0x62a43131, 0xcd7487b9, 0xe8f37474,
  0xfff6e309, 0xf115f6f6, 0x05460a43, 0x8c4c4646,
  0x8aac0926, 0x45a5acac, 0x1e893c97, 0x0fb58989,
  0x5014a044, 0x28b41414, 0xa3e15b42, 0xdfbae1e1,
  0x5816b04e, 0x2ca61616, 0xe83acdd2, 0x74f73a3a,
  0xb9696fd0, 0xd2066969, 0x2409482d, 0x12410909,
  0xdd70a7ad, 0xe0d77070, 0xe2b6d954, 0x716fb6b6,
  0x67d0ceb7, 0xbd1ed0d0, 0x93ed3b7e, 0xc7d6eded,
  0x17cc2edb, 0x85e2cccc, 0x15422a57, 0x84684242,
  0x5a98b4c2, 0x2d2c9898, 0xaaa4490e, 0x55eda4a4,
  0xa0285d88, 0x50752828, 0x6d5cda31, 0xb8865c5c,
  0xc7f8933f, 0xed6bf8f8, 0x228644a4, 0x11c28686
]);

const C7 = new Uint32Array([
  0x186018c0, 0x7830d818, 0x238c2305, 0xaf462623,
  0xc63fc67e, 0xf991b8c6, 0xe887e813, 0x6fcdfbe8,
  0x8726874c, 0xa113cb87, 0xb8dab8a9, 0x626d11b8,
  0x01040108, 0x05020901, 0x4f214f42, 0x6e9e0d4f,
  0x36d836ad, 0xee6c9b36, 0xa6a2a659, 0x0451ffa6,
  0xd26fd2de, 0xbdb90cd2, 0xf5f3f5fb, 0x06f70ef5,
  0x79f979ef, 0x80f29679, 0x6fa16f5f, 0xcede306f,
  0x917e91fc, 0xef3f6d91, 0x525552aa, 0x07a4f852,
  0x609d6027, 0xfdc04760, 0xbccabc89, 0x766535bc,
  0x9b569bac, 0xcd2b379b, 0x8e028e04, 0x8c018a8e,
  0xa3b6a371, 0x155bd2a3, 0x0c300c60, 0x3c186c0c,
  0x7bf17bff, 0x8af6847b, 0x35d435b5, 0xe16a8035,
  0x1d741de8, 0x693af51d, 0xe0a7e053, 0x47ddb3e0,
  0xd77bd7f6, 0xacb321d7, 0xc22fc25e, 0xed999cc2,
  0x2eb82e6d, 0x965c432e, 0x4b314b62, 0x7a96294b,
  0xfedffea3, 0x21e15dfe, 0x57415782, 0x16aed557,
  0x155415a8, 0x412abd15, 0x77c1779f, 0xb6eee877,
  0x37dc37a5, 0xeb6e9237, 0xe5b3e57b, 0x56d79ee5,
  0x9f469f8c, 0xd923139f, 0xf0e7f0d3, 0x17fd23f0,
  0x4a354a6a, 0x7f94204a, 0xda4fda9e, 0x95a944da,
  0x587d58fa, 0x25b0a258, 0xc903c906, 0xca8fcfc9,
  0x29a42955, 0x8d527c29, 0x0a280a50, 0x22145a0a,
  0xb1feb1e1, 0x4f7f50b1, 0xa0baa069, 0x1a5dc9a0,
  0x6bb16b7f, 0xdad6146b, 0x852e855c, 0xab17d985,
  0xbdcebd81, 0x73673cbd, 0x5d695dd2, 0x34ba8f5d,
  0x10401080, 0x50209010, 0xf4f7f4f3, 0x03f507f4,
  0xcb0bcb16, 0xc08bddcb, 0x3ef83eed, 0xc67cd33e,
  0x05140528, 0x110a2d05, 0x6781671f, 0xe6ce7867,
  0xe4b7e473, 0x53d597e4, 0x279c2725, 0xbb4e0227,
  0x41194132, 0x58827341, 0x8b168b2c, 0x9d0ba78b,
  0xa7a6a751, 0x0153f6a7, 0x7de97dcf, 0x94fab27d,
  0x956e95dc, 0xfb374995, 0xd847d88e, 0x9fad56d8,
  0xfbcbfb8b, 0x30eb70fb, 0xee9fee23, 0x71c1cdee,
  0x7ced7cc7, 0x91f8bb7c, 0x66856617, 0xe3cc7166,
  0xdd53dda6, 0x8ea77bdd, 0x175c17b8, 0x4b2eaf17,
  0x47014702, 0x468e4547, 0x9e429e84, 0xdc211a9e,
  0xca0fca1e, 0xc589d4ca, 0x2db42d75, 0x995a582d,
  0xbfc6bf91, 0x79632ebf, 0x071c0738, 0x1b0e3f07,
  0xad8ead01, 0x2347acad, 0x5a755aea, 0x2fb4b05a,
  0x8336836c, 0xb51bef83, 0x33cc3385, 0xff66b633,
  0x6391633f, 0xf2c65c63, 0x02080210, 0x0a041202,
  0xaa92aa39, 0x384993aa, 0x71d971af, 0xa8e2de71,
  0xc807c80e, 0xcf8dc6c8, 0x196419c8, 0x7d32d119,
  0x49394972, 0x70923b49, 0xd943d986, 0x9aaf5fd9,
  0xf2eff2c3, 0x1df931f2, 0xe3abe34b, 0x48dba8e3,
  0x5b715be2, 0x2ab6b95b, 0x881a8834, 0x920dbc88,
  0x9a529aa4, 0xc8293e9a, 0x2698262d, 0xbe4c0b26,
  0x32c8328d, 0xfa64bf32, 0xb0fab0e9, 0x4a7d59b0,
  0xe983e91b, 0x6acff2e9, 0x0f3c0f78, 0x331e770f,
  0xd573d5e6, 0xa6b733d5, 0x803a8074, 0xba1df480,
  0xbec2be99, 0x7c6127be, 0xcd13cd26, 0xde87ebcd,
  0x34d034bd, 0xe4688934, 0x483d487a, 0x75903248,
  0xffdbffab, 0x24e354ff, 0x7af57af7, 0x8ff48d7a,
  0x907a90f4, 0xea3d6490, 0x5f615fc2, 0x3ebe9d5f,
  0x2080201d, 0xa0403d20, 0x68bd6867, 0xd5d00f68,
  0x1a681ad0, 0x7234ca1a, 0xae82ae19, 0x2c41b7ae,
  0xb4eab4c9, 0x5e757db4, 0x544d549a, 0x19a8ce54,
  0x937693ec, 0xe53b7f93, 0x2288220d, 0xaa442f22,
  0x648d6407, 0xe9c86364, 0xf1e3f1db, 0x12ff2af1,
  0x73d173bf, 0xa2e6cc73, 0x12481290, 0x5a248212,
  0x401d403a, 0x5d807a40, 0x08200840, 0x28104808,
  0xc32bc356, 0xe89b95c3, 0xec97ec33, 0x7bc5dfec,
  0xdb4bdb96, 0x90ab4ddb, 0xa1bea161, 0x1f5fc0a1,
  0x8d0e8d1c, 0x8307918d, 0x3df43df5, 0xc97ac83d,
  0x976697cc, 0xf1335b97, 0x00000000, 0x00000000,
  0xcf1bcf36, 0xd483f9cf, 0x2bac2b45, 0x87566e2b,
  0x76c57697, 0xb3ece176, 0x82328264, 0xb019e682,
  0xd67fd6fe, 0xa9b128d6, 0x1b6c1bd8, 0x7736c31b,
  0xb5eeb5c1, 0x5b7774b5, 0xaf86af11, 0x2943beaf,
  0x6ab56a77, 0xdfd41d6a, 0x505d50ba, 0x0da0ea50,
  0x45094512, 0x4c8a5745, 0xf3ebf3cb, 0x18fb38f3,
  0x30c0309d, 0xf060ad30, 0xef9bef2b, 0x74c3c4ef,
  0x3ffc3fe5, 0xc37eda3f, 0x55495592, 0x1caac755,
  0xa2b2a279, 0x1059dba2, 0xea8fea03, 0x65c9e9ea,
  0x6589650f, 0xecca6a65, 0xbad2bab9, 0x686903ba,
  0x2fbc2f65, 0x935e4a2f, 0xc027c04e, 0xe79d8ec0,
  0xde5fdebe, 0x81a160de, 0x1c701ce0, 0x6c38fc1c,
  0xfdd3fdbb, 0x2ee746fd, 0x4d294d52, 0x649a1f4d,
  0x927292e4, 0xe0397692, 0x75c9758f, 0xbceafa75,
  0x06180630, 0x1e0c3606, 0x8a128a24, 0x9809ae8a,
  0xb2f2b2f9, 0x40794bb2, 0xe6bfe663, 0x59d185e6,
  0x0e380e70, 0x361c7e0e, 0x1f7c1ff8, 0x633ee71f,
  0x62956237, 0xf7c45562, 0xd477d4ee, 0xa3b53ad4,
  0xa89aa829, 0x324d81a8, 0x966296c4, 0xf4315296,
  0xf9c3f99b, 0x3aef62f9, 0xc533c566, 0xf697a3c5,
  0x25942535, 0xb14a1025, 0x597959f2, 0x20b2ab59,
  0x842a8454, 0xae15d084, 0x72d572b7, 0xa7e4c572,
  0x39e439d5, 0xdd72ec39, 0x4c2d4c5a, 0x6198164c,
  0x5e655eca, 0x3bbc945e, 0x78fd78e7, 0x85f09f78,
  0x38e038dd, 0xd870e538, 0x8c0a8c14, 0x8605988c,
  0xd163d1c6, 0xb2bf17d1, 0xa5aea541, 0x0b57e4a5,
  0xe2afe243, 0x4dd9a1e2, 0x6199612f, 0xf8c24e61,
  0xb3f6b3f1, 0x457b42b3, 0x21842115, 0xa5423421,
  0x9c4a9c94, 0xd625089c, 0x1e781ef0, 0x663cee1e,
  0x43114322, 0x52866143, 0xc73bc776, 0xfc93b1c7,
  0xfcd7fcb3, 0x2be54ffc, 0x04100420, 0x14082404,
  0x515951b2, 0x08a2e351, 0x995e99bc, 0xc72f2599,
  0x6da96d4f, 0xc4da226d, 0x0d340d68, 0x391a650d,
  0xfacffa83, 0x35e979fa, 0xdf5bdfb6, 0x84a369df,
  0x7ee57ed7, 0x9bfca97e, 0x2490243d, 0xb4481924,
  0x3bec3bc5, 0xd776fe3b, 0xab96ab31, 0x3d4b9aab,
  0xce1fce3e, 0xd181f0ce, 0x11441188, 0x55229911,
  0x8f068f0c, 0x8903838f, 0x4e254e4a, 0x6b9c044e,
  0xb7e6b7d1, 0x517366b7, 0xeb8beb0b, 0x60cbe0eb,
  0x3cf03cfd, 0xcc78c13c, 0x813e817c, 0xbf1ffd81,
  0x946a94d4, 0xfe354094, 0xf7fbf7eb, 0x0cf31cf7,
  0xb9deb9a1, 0x676f18b9, 0x134c1398, 0x5f268b13,
  0x2cb02c7d, 0x9c58512c, 0xd36bd3d6, 0xb8bb05d3,
  0xe7bbe76b, 0x5cd38ce7, 0x6ea56e57, 0xcbdc396e,
  0xc437c46e, 0xf395aac4, 0x030c0318, 0x0f061b03,
  0x5645568a, 0x13acdc56, 0x440d441a, 0x49885e44,
  0x7fe17fdf, 0x9efea07f, 0xa99ea921, 0x374f88a9,
  0x2aa82a4d, 0x8254672a, 0xbbd6bbb1, 0x6d6b0abb,
  0xc123c146, 0xe29f87c1, 0x535153a2, 0x02a6f153,
  0xdc57dcae, 0x8ba572dc, 0x0b2c0b58, 0x2716530b,
  0x9d4e9d9c, 0xd327019d, 0x6cad6c47, 0xc1d82b6c,
  0x31c43195, 0xf562a431, 0x74cd7487, 0xb9e8f374,
  0xf6fff6e3, 0x09f115f6, 0x4605460a, 0x438c4c46,
  0xac8aac09, 0x2645a5ac, 0x891e893c, 0x970fb589,
  0x145014a0, 0x4428b414, 0xe1a3e15b, 0x42dfbae1,
  0x165816b0, 0x4e2ca616, 0x3ae83acd, 0xd274f73a,
  0x69b9696f, 0xd0d20669, 0x09240948, 0x2d124109,
  0x70dd70a7, 0xade0d770, 0xb6e2b6d9, 0x54716fb6,
  0xd067d0ce, 0xb7bd1ed0, 0xed93ed3b, 0x7ec7d6ed,
  0xcc17cc2e, 0xdb85e2cc, 0x4215422a, 0x57846842,
  0x985a98b4, 0xc22d2c98, 0xa4aaa449, 0x0e55eda4,
  0x28a0285d, 0x88507528, 0x5c6d5cda, 0x31b8865c,
  0xf8c7f893, 0x3fed6bf8, 0x86228644, 0xa411c286
]);

/**
 * Whirlpool
 */

class Whirlpool {
  /**
   * Create a Whirlpool context.
   * @constructor
   */

  constructor() {
    this.state = new Uint32Array(16);
    this.msg = new Uint32Array(16);
    this.block = Buffer.allocUnsafe(64);
    this.size = FINALIZED;
  }

  /**
   * Initialize Whirlpool context.
   */

  init() {
    for (let i = 0; i < 16; i++)
      this.state[i] = 0;
    this.size = 0;
    return this;
  }

  /**
   * Update Whirlpool context.
   * @param {Buffer} data
   */

  update(data) {
    assert(Buffer.isBuffer(data));
    this._update(data, data.length);
    return this;
  }

  /**
   * Finalize Whirlpool context.
   * @returns {Buffer}
   */

  final() {
    return this._final(Buffer.allocUnsafe(64));
  }

  /**
   * Update Whirlpool context.
   * @private
   * @param {Buffer} data
   * @param {Number} len
   */

  _update(data, len) {
    assert(this.size !== FINALIZED, 'Context already finalized.');

    let pos = this.size & 63;
    let off = 0;

    this.size += len;

    if (pos > 0) {
      let want = 64 - pos;

      if (want > len)
        want = len;

      data.copy(this.block, pos, off, off + want);

      pos += want;
      len -= want;
      off += want;

      if (pos < 64)
        return;

      this.transform(this.block, 0);
    }

    while (len >= 64) {
      this.transform(data, off);
      off += 64;
      len -= 64;
    }

    if (len > 0)
      data.copy(this.block, 0, off, off + len);
  }

  /**
   * Finalize Whirlpool context.
   * @private
   * @param {Buffer} out
   * @returns {Buffer}
   */

  _final(out) {
    assert(this.size !== FINALIZED, 'Context already finalized.');

    const pos = this.size & 63;
    const len = this.size * 8;

    writeU32(DESC, len * (1 / 0x100000000), 24);
    writeU32(DESC, len, 28);

    this._update(PADDING, 1 + ((95 - pos) & 63));
    this._update(DESC, 32);

    for (let i = 0; i < 16; i++) {
      writeU32(out, this.state[i], i * 4);
      this.state[i] = 0;
    }

    for (let i = 0; i < 16; i++)
      this.msg[i] = 0;

    for (let i = 0; i < 64; i++)
      this.block[i] = 0;

    this.size = FINALIZED;

    return out;
  }

  /**
   * Transform Whirlpool block.
   * @param {Buffer} chunk
   * @param {Number} pos
   */

  transform(chunk, pos) {
    const block = this.msg;

    for (let i = 0; i < 16; i++)
      block[i] = readU32(chunk, pos + i * 4);

    const K = new Uint32Array(16);
    const L = new Uint32Array(16);
    const state = new Uint32Array(16);

    for (let i = 0; i < 16; i++) {
      K[i] = this.state[i];
      state[i] = block[i] ^ K[i];
    }

    for (let r = 1; r < 10 + 1; r++) {
      for (let i = 0; i < 8; i++) {
        L[i * 2 + 0] = 0
          ^ C0[((K[(0 + i) % 8 * 2 + 0] >>> 24) & 0xff) * 2 + 0]
          ^ C1[((K[(7 + i) % 8 * 2 + 0] >>> 16) & 0xff) * 2 + 0]
          ^ C2[((K[(6 + i) % 8 * 2 + 0] >>>  8) & 0xff) * 2 + 0]
          ^ C3[((K[(5 + i) % 8 * 2 + 0] >>>  0) & 0xff) * 2 + 0]
          ^ C4[((K[(4 + i) % 8 * 2 + 1] >>> 24) & 0xff) * 2 + 0]
          ^ C5[((K[(3 + i) % 8 * 2 + 1] >>> 16) & 0xff) * 2 + 0]
          ^ C6[((K[(2 + i) % 8 * 2 + 1] >>>  8) & 0xff) * 2 + 0]
          ^ C7[((K[(1 + i) % 8 * 2 + 1] >>>  0) & 0xff) * 2 + 0]
          ^ (i === 0 ? RC[r * 2 + 0] : 0);

        L[i * 2 + 1] = 0
          ^ C0[((K[(0 + i) % 8 * 2 + 0] >>> 24) & 0xff) * 2 + 1]
          ^ C1[((K[(7 + i) % 8 * 2 + 0] >>> 16) & 0xff) * 2 + 1]
          ^ C2[((K[(6 + i) % 8 * 2 + 0] >>>  8) & 0xff) * 2 + 1]
          ^ C3[((K[(5 + i) % 8 * 2 + 0] >>>  0) & 0xff) * 2 + 1]
          ^ C4[((K[(4 + i) % 8 * 2 + 1] >>> 24) & 0xff) * 2 + 1]
          ^ C5[((K[(3 + i) % 8 * 2 + 1] >>> 16) & 0xff) * 2 + 1]
          ^ C6[((K[(2 + i) % 8 * 2 + 1] >>>  8) & 0xff) * 2 + 1]
          ^ C7[((K[(1 + i) % 8 * 2 + 1] >>>  0) & 0xff) * 2 + 1]
          ^ (i === 0 ? RC[r * 2 + 1] : 0);
      }

      for (let i = 0; i < 16; i++)
        K[i] = L[i];

      for (let i = 0; i < 8; i++) {
        L[i * 2 + 0] = 0
          ^ C0[((state[(0 + i) % 8 * 2 + 0] >>> 24) & 0xff) * 2 + 0]
          ^ C1[((state[(7 + i) % 8 * 2 + 0] >>> 16) & 0xff) * 2 + 0]
          ^ C2[((state[(6 + i) % 8 * 2 + 0] >>>  8) & 0xff) * 2 + 0]
          ^ C3[((state[(5 + i) % 8 * 2 + 0] >>>  0) & 0xff) * 2 + 0]
          ^ C4[((state[(4 + i) % 8 * 2 + 1] >>> 24) & 0xff) * 2 + 0]
          ^ C5[((state[(3 + i) % 8 * 2 + 1] >>> 16) & 0xff) * 2 + 0]
          ^ C6[((state[(2 + i) % 8 * 2 + 1] >>>  8) & 0xff) * 2 + 0]
          ^ C7[((state[(1 + i) % 8 * 2 + 1] >>>  0) & 0xff) * 2 + 0]
          ^ K[i * 2 + 0];

        L[i * 2 + 1] = 0
          ^ C0[((state[(0 + i) % 8 * 2 + 0] >>> 24) & 0xff) * 2 + 1]
          ^ C1[((state[(7 + i) % 8 * 2 + 0] >>> 16) & 0xff) * 2 + 1]
          ^ C2[((state[(6 + i) % 8 * 2 + 0] >>>  8) & 0xff) * 2 + 1]
          ^ C3[((state[(5 + i) % 8 * 2 + 0] >>>  0) & 0xff) * 2 + 1]
          ^ C4[((state[(4 + i) % 8 * 2 + 1] >>> 24) & 0xff) * 2 + 1]
          ^ C5[((state[(3 + i) % 8 * 2 + 1] >>> 16) & 0xff) * 2 + 1]
          ^ C6[((state[(2 + i) % 8 * 2 + 1] >>>  8) & 0xff) * 2 + 1]
          ^ C7[((state[(1 + i) % 8 * 2 + 1] >>>  0) & 0xff) * 2 + 1]
          ^ K[i * 2 + 1];
      }

      for (let i = 0; i < 16; i++)
        state[i] = L[i];
    }

    for (let i = 0; i < 16; i++)
      this.state[i] ^= state[i] ^ block[i];
  }

  static hash() {
    return new Whirlpool();
  }

  static hmac() {
    return new HMAC(Whirlpool, 64);
  }

  static digest(data) {
    return Whirlpool.ctx.init().update(data).final();
  }

  static root(left, right) {
    assert(Buffer.isBuffer(left) && left.length === 64);
    assert(Buffer.isBuffer(right) && right.length === 64);
    return Whirlpool.ctx.init().update(left).update(right).final();
  }

  static multi(x, y, z) {
    const ctx = Whirlpool.ctx;
    ctx.init();
    ctx.update(x);
    ctx.update(y);
    if (z)
      ctx.update(z);
    return ctx.final();
  }

  static mac(data, key) {
    return Whirlpool.hmac().init(key).update(data).final();
  }
}

Whirlpool.native = 0;
Whirlpool.id = 'WHIRLPOOL';
Whirlpool.size = 64;
Whirlpool.bits = 512;
Whirlpool.blockSize = 64;
Whirlpool.zero = Buffer.alloc(64, 0x00);
Whirlpool.ctx = new Whirlpool();

/*
 * Helpers
 */

function readU32(data, off) {
  return (data[off++] * 0x1000000
        + data[off++] * 0x10000
        + data[off++] * 0x100
        + data[off]);
}

function writeU32(data, num, off) {
  data[off++] = num >>> 24;
  data[off++] = num >>> 16;
  data[off++] = num >>> 8;
  data[off++] = num;
  return off;
}

/*
 * Expose
 */

module.exports = Whirlpool;
