/*
 * Dillo Widget
 *
 * Copyright 2005-2007 Sebastian Geerken <sgeerken@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "fltkviewport.hh"
#include "overlay_window.hh"

#include <FL/Fl.H>
#include <FL/fl_draw.H>
#include <FL/names.h>

#include <stdio.h>
#include "../lout/msg.h"
#include "../lout/debug.hh"
#include "../dlib/dlib.h" /* Dstr */

int EXPERIMENTAL_PAGING_MODE = 0;

using namespace lout;
using namespace lout::object;
using namespace lout::container::typed;

namespace dw {
namespace fltk {

/*
 * Lets SHIFT+{Left,Right} go to the parent
 */
class CustScrollbar : public Fl_Scrollbar
{
public:
   CustScrollbar(int x, int y, int w, int h) : Fl_Scrollbar(x,y,w,h) {};
   int handle(int e) {
      if (e == FL_SHORTCUT && Fl::event_state() == FL_SHIFT &&
          (Fl::event_key() == FL_Left || Fl::event_key() == FL_Right))
         return 0;
      return Fl_Scrollbar::handle(e);
   }
};

FltkViewport::FltkViewport (int X, int Y, int W, int H, const char *label):
   FltkWidgetView (X, Y, W, H, label)
{
   DBG_OBJ_CREATE ("dw::fltk::FltkViewport");

   hscrollbar = new CustScrollbar (x (), y (), 1, 1);
   hscrollbar->type(FL_HORIZONTAL);
   hscrollbar->callback (hscrollbarCallback, this);
   hscrollbar->selection_color(FL_SELECTION_COLOR);
   hscrollbar->hide();
   add (hscrollbar);

   vscrollbar = new Fl_Scrollbar (x (), y(), 1, 1);
   vscrollbar->type(FL_VERTICAL);
   vscrollbar->callback (vscrollbarCallback, this);
   vscrollbar->selection_color(FL_SELECTION_COLOR);
   vscrollbar->hide();
   add (vscrollbar);

   dragScrollState = OFF;
   pushed_child = NULL;
   scrollX = scrollY = scrollDX = scrollDY = 0;
   horScrolling = verScrolling = 0;
   preventScrollbars = true;
   Hscrollable = Vscrollable = false;
}

FltkViewport::~FltkViewport ()
{
   DBG_OBJ_DELETE ();
}

void FltkViewport::adjustScrollbarsAllocation ()
{
   int hdiff = 0, vdiff = 0;

   _MSG(" >>FltkViewport::adjustScrollbarsAllocation\n");

   hdiff = Vscrollable ? SCROLLBAR_THICKNESS : 0;
   vdiff = Hscrollable ? SCROLLBAR_THICKNESS : 0;

   if (scrollbarOnLeft) {
      hscrollbar->resize(x () + hdiff, y () + h () - SCROLLBAR_THICKNESS,
                         w () - hdiff, SCROLLBAR_THICKNESS);
      vscrollbar->resize(x (), y (),
                         SCROLLBAR_THICKNESS, h () - vdiff);
   } else {
      hscrollbar->resize(x (), y () + h () - SCROLLBAR_THICKNESS,
                         w () - hdiff, SCROLLBAR_THICKNESS);
      vscrollbar->resize(x () + w () - SCROLLBAR_THICKNESS, y (),
                         SCROLLBAR_THICKNESS, h () - vdiff);
   }
   adjustScrollbarValues();
}

void FltkViewport::adjustScrollbarValues ()
{
   hscrollbar->value (scrollX, hscrollbar->w(), 0, canvasWidth);
   vscrollbar->value (scrollY, vscrollbar->h(), 0, canvasHeight);
}

void FltkViewport::hscrollbarChanged ()
{
   scroll (hscrollbar->value () - scrollX, 0);
}

void FltkViewport::vscrollbarChanged ()
{
   scroll (0, vscrollbar->value () - scrollY);
}

void FltkViewport::vscrollbarCallback (Fl_Widget *vscrollbar,void *viewportPtr)
{
   ((FltkViewport*)viewportPtr)->vscrollbarChanged ();
}

void FltkViewport::hscrollbarCallback (Fl_Widget *hscrollbar,void *viewportPtr)
{
   ((FltkViewport*)viewportPtr)->hscrollbarChanged ();
}

// ----------------------------------------------------------------------

void FltkViewport::resize(int X, int Y, int W, int H)
{
   bool dimension_changed = W != w() || H != h();

   Fl_Group::resize(X, Y, W, H);
   if (dimension_changed) {
      theLayout->viewportSizeChanged (this, W, H);
      adjustScrollbarsAllocation ();
      updateOverlayBounds();
   }
}

void FltkViewport::display_page_position()
{
   /* I am reluctant to fill the overlay with too much crap, but when you navigate by keyboard (or paging mode) and don’t have scrollbars up, it feels important as a user to know where you are on the page */
   if (Hscrollable || Vscrollable) {
      Dstr *str = dStr_new("[");
      if (Hscrollable)
         dStr_sprintfa(str, "%d", scrollX * 100 / (canvasWidth - w()));
      dStr_append_c(str, ':');
      if (Vscrollable)
         dStr_sprintfa(str, "%d", scrollY * 100 / (canvasHeight - h()));
      dStr_append_c(str, ']');
      setOverlayProgressMsg(str->str, 1.0f);
      dStr_free(str, 1);
   }
}

void FltkViewport::draw_area (void *data, int x, int y, int w, int h)
{
  FltkViewport *vp = (FltkViewport*) data;
  fl_push_clip(x, y, w, h);

  vp->FltkWidgetView::draw ();

  fl_pop_clip();
}

/*
 * Draw the viewport.
 *
 *  + Damage flags come in different ways, draw() should cope with them all.
 *  + Damage flags are alive for visible and hidden widgets.
 *  + FL_DAMAGE_CHILD can flag scroll bars or embedded FLTK widgets.
 */
void FltkViewport::draw ()
{
   const int d = damage(),
             vis_vs = vscrollbar->visible () ? SCROLLBAR_THICKNESS : 0,
             vis_hs = hscrollbar->visible () ? SCROLLBAR_THICKNESS : 0,
             draw = d & (FL_DAMAGE_ALL | FL_DAMAGE_EXPOSE),
             draw_vs = vis_vs && vscrollbar->damage (),
             draw_hs = vis_hs && hscrollbar->damage ();

   _MSG("FltkViewport::draw d=%d  =>  ", d);
   // main area
   if (d == FL_DAMAGE_CHILD && (draw_vs || draw_hs)) {
      _MSG("none\n");
   } else if (d == (FL_DAMAGE_SCROLL | FL_DAMAGE_CHILD)) {
      int x = this->x();

      if (scrollbarOnLeft)
         x += vis_vs;
      fl_scroll(x, y(), w() - vis_vs, h() - vis_hs,
                -scrollDX, -scrollDY, draw_area, this);
      _MSG("fl_scroll()\n");
   } else {
      int x = this->x();

      if (scrollbarOnLeft)
         x += vis_vs;
      draw_area(this, x, y(), w() - vis_vs, h() - vis_hs);
      _MSG("draw_area()\n");
   }
   // scrollbars
   if (draw || draw_vs) {
      draw_child (*vscrollbar);
   }
   if (draw || draw_hs) {
      draw_child (*hscrollbar);
   }
   if (draw && vis_vs && vis_hs) {
      fl_color(FL_BACKGROUND_COLOR);
      if (scrollbarOnLeft) {
         fl_rectf(x(), y()+h()-vis_hs, vis_vs, vis_hs);
      } else {
         fl_rectf(x()+w()-vis_vs, y()+h()-vis_hs, vis_vs, vis_hs);
      }
   }
   scrollDX = 0;
   scrollDY = 0;
}

bool FltkViewport::in_heart_of_widget()
{
   return (Fl::event_x() >= x() + 20 &&
           Fl::event_x() <= x() + w() - 20 &&
           Fl::event_y() >= y() + 20 &&
           Fl::event_y() <= y() + h() - 20);
}

/*
 * So many things are happening in handle() in order to make a touchscreen useful.
 *
 * - An experimental paging mode that ignores most events except to scroll based on the screen position touched.
 * - If we are in a long press (comes in the form of a right button press), undo the initial press event.
 * - If events belong to our scrollbar children, pass them on.
 * - If there is a push and release with very little mouse movement in the meantime, hide the scrollbars.
 * - If there is a drag with enough mouse movement, show the scrollbars.
 * - Text selection scrolling, where everything keeps moving under the control of a timeout if the mouse is near the edge of the window or outside it in order to allow the user to extend text selection further.
 * - Let a press through for links, etc., but do what we can to transform that into dragging if the mouse moves far enough. This is not fully solved yet, as a child input widget needs to do its own dragging, but a child complex button should let us drag, and Dillo-world things such as links should let us drag but need the release themselves. The FLTK way is that whatever accepted the PUSH gets the DRAG and the RELEASE.
 *
 *  The way to take care of this is probably to expand handle()’s return from the widget view to be multiple bits rather than merely handled vs. not-handled.
 */

int FltkViewport::handle (int event)
{
   int ret = 0;
   _MSG("FltkViewport::handle %s\n", fl_eventnames[event]);

   switch(event) {
   case FL_KEYBOARD:
      /* When the viewport has focus (and not one of its children), FLTK
       * sends the event here. Returning zero tells FLTK to resend the
       * event as SHORTCUT, which we finally route to the parent. */

      /* As we don't know the exact keybindings set by the user, we ask for
       * all of them (except for the minimum needed to keep form navigation).*/
      if (Fl::event_key() != FL_Tab || Fl::event_ctrl())
         return 0;
      break;

   case FL_SHORTCUT:
      /* send it to the parent (UI) */
      return 0;

   case FL_FOCUS:
      /** \bug Draw focus box. */
      break;

   case FL_UNFOCUS:
      /** \bug Undraw focus box. */
      break;

   case FL_PUSH:
      if (EXPERIMENTAL_PAGING_MODE)
         return 1;
      if (Fl::event_button() == FL_RIGHT_MOUSE && (dragScrollState == POTENTIAL || dragScrollState == ACTIVE)) {
         // Another consequence of long press. We have to send the initial press to get link text, for instance. But you can’t send the button 1 release or you’ve just clicked the link or button or whatever. At other times, we have to undo what that press started. Drag scrolling is one such thing, and surely we’ll encounter more.
         dragScrollState = OFF;
         pushed_child = NULL;
      }
      if (vscrollbar->visible() && Fl::event_inside(vscrollbar)) {
         if (vscrollbar->handle(event))
            verScrolling = 1;
         ret = true;
      } else if (hscrollbar->visible() && Fl::event_inside(hscrollbar)) {
         if (hscrollbar->handle(event))
            horScrolling = 1;
         ret = true;
      } else if (Fl::event_button() == (theLayout->getSelectionMode() ? FL_MIDDLE_MOUSE : FL_LEFT_MOUSE)) {
         pushX = Fl::event_x();
         pushY = Fl::event_y();

         if (dragScrollState != DISABLED) {
            /* receive FL_DRAG and FL_RELEASE */
            dragScrollState = POTENTIAL;
            dragX = pushX;
            dragY = pushY;
            (void)FltkWidgetView::handle(event);
            if (contains(Fl::pushed())) {
               pushed_child = Fl::pushed();
               Fl::pushed(this);
            }
            return true;
         }
      }
      break;

   case FL_DRAG:
      if (pushed_child) {
         if (pushed_child->handle(FL_DRAG))
            return 1;
      }
      if (EXPERIMENTAL_PAGING_MODE)
         return 1;
      if (Fl::event_button() == (theLayout->getSelectionMode() ? FL_LEFT_MOUSE : FL_MIDDLE_MOUSE) && in_heart_of_widget()) {
         Fl::remove_timeout(selectionScroll);
         // If the mouse is on a scrollbar, the overlay doesn’t need to know about it, but it’s not a problem to tell it regardless.
      }
      if (Fl::event_inside(this))
         ((OverlayWindow *)window())->mouse_moved(Fl::event_x(), Fl::event_y());
      if (verScrolling) {
         vscrollbar->handle(event);
         return 1;
      } else if (horScrolling) {
         hscrollbar->handle(event);
         return 1;
      } else if (Fl::event_button() == (theLayout->getSelectionMode() ? FL_LEFT_MOUSE : FL_MIDDLE_MOUSE) && !in_heart_of_widget()) {
         /* Scroll if we’re outside the widget or nearly so (in case there is no outside) so selection can extend further */
         mouse_x = Fl::event_x();
         mouse_y = Fl::event_y();
         if (!Fl::has_timeout(selectionScroll, this))
            Fl::add_timeout(0.0, selectionScroll, this);
      } else {
         if (dragScrollState == POTENTIAL) {
            if (abs(Fl::event_x() - dragX) >= dragThreshold ||
                abs(Fl::event_y() - dragY) >= dragThreshold) {

               dragScrollState = ACTIVE;
               dragX = Fl::event_x();
               dragY = Fl::event_y();

               if (pushed_child) {
                  int w = pushed_child->when();
                  pushed_child->when(FL_WHEN_NEVER);
                  pushed_child->handle(FL_RELEASE);
                  pushed_child->when(w);
                  if (Fl::focus() == pushed_child)
                     Fl::focus(this);

                  pushed_child = NULL;
               }
               setCursor (core::style::CURSOR_MOVE);

               if (preventScrollbars) {
                  preventScrollbars = false;
                  if (Hscrollable)
                     hscrollbar->show();
                  if (Vscrollable) {
                     vscrollbar->show();
                     if (Hscrollable)
                        damage(FL_DAMAGE_ALL); // getting that square to draw
                  }
                  if (Hscrollable || Vscrollable)
                     updateOverlayBounds();
               }
            }
         }
         if (dragScrollState == ACTIVE) {
            mouse_x = Fl::event_x();
            mouse_y = Fl::event_y();
            scroll(8 * (mouse_x - dragX), 8 * (mouse_y - dragY));
            dragX = mouse_x;
            dragY = mouse_y;
            return 1;
         }
      }
      break;

   case FL_MOUSEWHEEL:
      ret = (Fl::event_dx() ? hscrollbar : vscrollbar)->handle(event);
      display_page_position();
      return ret;

   case FL_RELEASE:
      if (pushed_child) {
         int ret = pushed_child->handle(event);
         pushed_child = NULL;
         return ret;
      }

      if (EXPERIMENTAL_PAGING_MODE) {
         if (Fl::event_y() < y() + vscrollbar->linesize()) {
            scroll(core::TOP_CMD);
         } else if (Fl::event_y() >= y() + h() - vscrollbar->linesize()) {
            scroll(core::BOTTOM_CMD);
         } else if (Hscrollable && Vscrollable) {
            /* If scrollable both ways, have quadrants. If one way, halves. */
            double xproportion = ((double)Fl::event_x() - x()) / w(),
                   yproportion = ((double)Fl::event_y() - y()) / h();
            if (xproportion > yproportion) {
               if (xproportion + yproportion < 1.0)
                  scroll(core::SCREEN_UP_CMD);
               else
                  scroll(core::SCREEN_RIGHT_CMD);
            } else {
               if (xproportion + yproportion < 1.0)
                  scroll(core::SCREEN_LEFT_CMD);
               else
                  scroll(core::SCREEN_DOWN_CMD);
            }
         } else if (Vscrollable) {
            if (Fl::event_y() - y() < h() / 2)
               scroll(core::SCREEN_UP_CMD);
            else
               scroll(core::SCREEN_DOWN_CMD);
         } else if (Hscrollable) {
            if (Fl::event_x() - x() < w() / 2)
               scroll(core::SCREEN_LEFT_CMD);
            else
               scroll(core::SCREEN_RIGHT_CMD);
         }
         return 1;
      }
      Fl::remove_timeout(selectionScroll);
      if (Fl::event_button() == (theLayout->getSelectionMode() ? FL_MIDDLE_MOUSE : FL_LEFT_MOUSE)) {
         setCursor (core::style::CURSOR_DEFAULT);
      }
      if (verScrolling) {
         ret = vscrollbar->handle(event);
      } else if (horScrolling) {
         ret = hscrollbar->handle(event);
      }
      horScrolling = verScrolling = 0;
      if (dragScrollState == POTENTIAL || dragScrollState == ACTIVE) {
         bool active = dragScrollState == ACTIVE;
         dragScrollState = OFF;
         if (active)
            return 1;
      }
      /* If it looked more like a tap than a drag, hide those scrollbars. */
      if (abs(Fl::event_x() - pushX) < dragThreshold &&
          abs(Fl::event_y() - pushY) < dragThreshold &&
          ! ((vscrollbar->visible() && Fl::event_inside(vscrollbar)) ||
             (hscrollbar->visible() && Fl::event_inside(hscrollbar)))) {
         preventScrollbars = true;
         if (Hscrollable)
            hscrollbar->hide();
         if (Vscrollable) {
            vscrollbar->hide();
            if (Hscrollable)
               damage(FL_DAMAGE_ALL); // I have not yet seen it need this like it was clearly needed when showing the scrollbars
         }
         if (Hscrollable || Vscrollable)
            updateOverlayBounds();
      }
      break;

   case FL_ENTER:
      if (EXPERIMENTAL_PAGING_MODE)
         return 1;
      if (vscrollbar->visible() && Fl::event_inside(vscrollbar))
         return vscrollbar->handle(event);
      if (hscrollbar->visible() && Fl::event_inside(hscrollbar))
         return hscrollbar->handle(event);
      /* could be the result of, e.g., closing another window. */
      mouse_x = Fl::event_x();
      mouse_y = Fl::event_y();
      positionChanged();
      break;

   case FL_MOVE:
      if (EXPERIMENTAL_PAGING_MODE)
         return 1;
      {
         bool in_scrollbar = false;

         /* Use LEAVE in order not to be over a link, etc., anymore. */
         if (vscrollbar->visible() && Fl::event_inside(vscrollbar)) {
            in_scrollbar = true;
            (void)FltkWidgetView::handle(FL_LEAVE);
            return vscrollbar->handle(event);
         }
         if (hscrollbar->visible() && Fl::event_inside(hscrollbar)) {
            in_scrollbar = true;
            (void)FltkWidgetView::handle(FL_LEAVE);
            return hscrollbar->handle(event);
         }
         if (!in_scrollbar)
            ((OverlayWindow *)window())->mouse_moved(Fl::event_x(), Fl::event_y());
      }
      break;

   case FL_LEAVE:
      if (EXPERIMENTAL_PAGING_MODE)
         return 1;
      mouse_x = mouse_y = -1;
      break;

   case FL_SHOW:
      updateOverlayBounds();
      break;
   }

   return ret ? ret : FltkWidgetView::handle (event);
}

// ----------------------------------------------------------------------

void FltkViewport::setCanvasSize (int width, int ascent, int descent)
{
   FltkWidgetView::setCanvasSize (width, ascent, descent);
   adjustScrollbarValues ();
}

/*
 * This is used to simulate mouse motion (e.g., when scrolling).
 */
void FltkViewport::positionChanged ()
{
   if (dragScrollState != ACTIVE) {
      int xmin = x();
      int xmax = x() + w();
      int ymax = y() + h();

      if (mouse_x >= xmin && mouse_x < xmax && mouse_y >= y() && mouse_y < ymax) {
         (void)theLayout->motionNotify (this,
                                        translateViewXToCanvasX (mouse_x),
                                        translateViewYToCanvasY (mouse_y),
                                        (core::ButtonState)0);
      }
   }
}

/*
 * For scrollbars, this currently sets the same step to both vertical and
 * horizontal. It may be differentiated if necessary.
 */
void FltkViewport::setScrollStep(int step)
{
   vscrollbar->linesize(step);
   hscrollbar->linesize(step);
}

void FltkViewport::setScrollThickness(int w)
{
   SCROLLBAR_THICKNESS = w;
}

bool FltkViewport::usesViewport ()
{
   return true;
}

int FltkViewport::getHScrollbarThickness ()
{
   return SCROLLBAR_THICKNESS;
}

int FltkViewport::getVScrollbarThickness ()
{
   return SCROLLBAR_THICKNESS;
}

void FltkViewport::scrollTo (int x, int y)
{
   x = misc::min (x, canvasWidth - w());
   x = misc::max (x, 0);

   y = misc::min (y, canvasHeight - h());
   y = misc::max (y, 0);

   if (x == scrollX && y == scrollY) {
      return;
   }

   /* multiple calls to scroll can happen before a redraw occurs.
    * scrollDX and scrollDY can therefore be non-zero here.
    */
   updateCanvasWidgets (x - scrollX, y - scrollY);
   scrollDX += x - scrollX;
   scrollDY += y - scrollY;

   scrollX = x;
   scrollY = y;

   adjustScrollbarValues ();

   /* this prints the percentage scrolled */
   _MSG("%.f/%.f\n", 100 * ((Fl_Valuator *)vscrollbar)->value() / ((Fl_Valuator *)vscrollbar)->maximum(), 100 * ((Fl_Valuator *)hscrollbar)->value() / ((Fl_Valuator *)hscrollbar)->maximum());

   damage(FL_DAMAGE_SCROLL);
   theLayout->scrollPosChanged (this, scrollX, scrollY);
   positionChanged();
}

void FltkViewport::scroll (int dx, int dy)
{
   scrollTo (scrollX + dx, scrollY + dy);
}

void FltkViewport::scroll (core::ScrollCommand cmd)
{
   if (cmd == core::SCREEN_UP_CMD) {
      scroll (0, -h () + vscrollbar->linesize ());
   } else if (cmd == core::SCREEN_DOWN_CMD) {
      scroll (0, h () - vscrollbar->linesize ());
   } else if (cmd == core::SCREEN_LEFT_CMD) {
      scroll (-w() + hscrollbar->linesize (), 0);
   } else if (cmd == core::SCREEN_RIGHT_CMD) {
      scroll (w() - hscrollbar->linesize (), 0);
   } else if (cmd == core::LINE_UP_CMD) {
      scroll (0, -vscrollbar->linesize ());
   } else if (cmd == core::LINE_DOWN_CMD) {
      scroll (0, vscrollbar->linesize ());
   } else if (cmd == core::LEFT_CMD) {
      scroll (-hscrollbar->linesize (), 0);
   } else if (cmd == core::RIGHT_CMD) {
      scroll (hscrollbar->linesize (), 0);
   } else if (cmd == core::TOP_CMD) {
      scrollTo (scrollX, 0);
   } else if (cmd == core::BOTTOM_CMD) {
      scrollTo (scrollX, canvasHeight); /* gets adjusted in scrollTo () */
   }

   display_page_position();
}

/*
 * Scrolling in response to selection where the cursor is outside (or near the edge) of the view. (Near the edge added because on mobile there may be no outside possible)
 */
void FltkViewport::selectionScroll ()
{
   int distance;
   int dx = 0, dy = 0;

   if ((distance = x() + vscrollbar->linesize() - mouse_x) > 0)
      dx = -w() / 4;
   else if ((distance = mouse_x - (x() + w() - vscrollbar->linesize())) > 0)
      dx = w() / 4;
   if ((distance = y() + vscrollbar->linesize() - mouse_y) > 0)
      dy = -h() / 4;
   else if ((distance = mouse_y - (y() + h() - vscrollbar->linesize())) > 0)
      dy = h() / 4;

   scroll (dx, dy);
}

void FltkViewport::selectionScroll (void *data)
{
   ((FltkViewport *)data)->selectionScroll ();
   Fl::repeat_timeout(0.15, selectionScroll, data);
}

void FltkViewport::setScrollbarOnLeft (bool enable)
{
   scrollbarOnLeft = enable ? 1 : 0;
   adjustScrollbarsAllocation();
   damage(FL_DAMAGE_ALL);
}

void FltkViewport::setViewportSize (int width, int height,
                                    int hScrollbarThickness,
                                    int vScrollbarThickness)
{
   int adjustReq =
      (hscrollbar->visible() ? !hScrollbarThickness : hScrollbarThickness) ||
      (vscrollbar->visible() ? !vScrollbarThickness : vScrollbarThickness);

   int adjustX =
      scrollbarOnLeft ? (vscrollbar->visible() ? !vScrollbarThickness : vScrollbarThickness) : 0;

   _MSG("FltkViewport::setViewportSize old_w,old_h=%dx%d -> w,h=%dx%d\n"
       "\t hThick=%d hVis=%d, vThick=%d vVis=%d, adjustReq=%d\n",
       w(),h(),width,height,
       hScrollbarThickness,hscrollbar->visible(),
       vScrollbarThickness,vscrollbar->visible(), adjustReq);

   Hscrollable = hScrollbarThickness > 0;
   Vscrollable = vScrollbarThickness > 0;

   if (!preventScrollbars) {
      Hscrollable ? hscrollbar->show () : hscrollbar->hide ();
      Vscrollable ? vscrollbar->show () : vscrollbar->hide ();
   }

   /* If no scrollbar, go to the beginning */
   scroll(hScrollbarThickness ? 0 : -scrollX,
          vScrollbarThickness ? 0 : -scrollY);

   /* Adjust when scrollbar visibility changes */
   if (adjustReq)
      adjustScrollbarsAllocation ();
   if (adjustX)
      updateCanvasWidgets(vScrollbarThickness ? -vScrollbarThickness : SCROLLBAR_THICKNESS, 0);
   updateOverlayBounds();
}

void FltkViewport::updateCanvasWidgets (int dx, int dy)
{
   // scroll all child widgets except scroll bars
   for (int i = children () - 1; i > 0; i--) {
      Fl_Widget *widget = child (i);

      if (widget == hscrollbar || widget == vscrollbar)
         continue;

      widget->position(widget->x () - dx, widget->y () - dy);
   }
}

void FltkViewport::determineContentBounds(int &boundsX, int &boundsY, int &boundsW, int &boundsH)
{
   boundsX = x() + ((scrollbarOnLeft && vscrollbar->visible() && !preventScrollbars) ? SCROLLBAR_THICKNESS : 0);
   boundsY = y();
   boundsW = w() - (vscrollbar->visible()  && !preventScrollbars ? SCROLLBAR_THICKNESS : 0);
   boundsH = h() - (hscrollbar->visible()  && !preventScrollbars ? SCROLLBAR_THICKNESS : 0);
}

/*
 * changed due to resize, scrollbar visibility changed…
 */
void FltkViewport::updateOverlayBounds()
{
   if (visible_r()) {  // this widget and all parents visible
      int boundsX, boundsY, boundsW, boundsH;
      OverlayWindow *win = (OverlayWindow *)(window());

      if (win) {
         determineContentBounds(boundsX, boundsY, boundsW, boundsH);

         win->update_overlay_bounds(boundsX, boundsY, boundsW, boundsH);
      }
   }
}

void FltkViewport::setOverlayLinkDest(const char *s)
{
   if (visible_r()) {  // this widget and all parents visible
      OverlayWindow *win = (OverlayWindow *)(window());

      win->set_overlay_link_dest(s);
   }
}

void FltkViewport::setOverlayProgressMsg(const char *s, double proportion)
{
   if (visible_r()) {  // this widget and all parents visible
      OverlayWindow *win = (OverlayWindow *)(window());

      win->set_overlay_progress_msg(s, proportion);
   }
}

int FltkViewport::translateViewXToCanvasX (int X)
{
   return X - x () + scrollX;
}

int FltkViewport::translateViewYToCanvasY (int Y)
{
   return Y - y () + scrollY;
}

int FltkViewport::translateCanvasXToViewX (int X)
{
   return X + x () - scrollX;
}

int FltkViewport::translateCanvasYToViewY (int Y)
{
   return Y + y () - scrollY;
}


} // namespace fltk
} // namespace dw
