#include <stdlib.h>  // free
#include <math.h>    // isnan, NaN
#include <FL/Fl.H>
#include <FL/fl_draw.H>
#include "overlay_window.hh"

#include "../lout/msg.h"
#include "../lout/misc.hh"

static void OverlayWindow_clear_overlay_progress_msg_handler(void *user_data)
{
   OverlayWindow *win = (OverlayWindow *)user_data;

   win->clear_overlay_progress_msg();
}

OverlayWindow::OverlayWindow (int w, int h) : Fl_Overlay_Window(w,h), boundsX(0), boundsY(0), boundsW(0), boundsH(0) {
   link_dest.str = NULL;
   link_dest.drawX = 0;
   link_dest.drawY = 0;
   link_dest.drawW = 0;
   link_dest.drawH = 0;
   progress_msg.str = NULL;
   progress_msg.drawX = 0;
   progress_msg.drawY = 0;
   progress_msg.drawW = 0;
   progress_msg.drawH = 0;
}

OverlayWindow::~OverlayWindow () {
   Fl::remove_timeout(OverlayWindow_clear_overlay_progress_msg_handler, this);
   if (link_dest.str)
      free(link_dest.str);
   if (progress_msg.str)
      free(progress_msg.str);
}

/*
 * We trigger this by calling redraw_overlay(), but it is also called when whatever it’s overlaying (overlying?) changes, e.g. when scrolling. It also appeared to be called when I was typing in the Location input, don’t know why it wanted to do that.
 *
 * fl_draw can wrap words. Unfortunately, it won’t wrap URLs.
 */
void OverlayWindow::draw_overlay()
{
   /*
    * We don’t draw under the mouse cursor. The link destination gets priority, and wants the bottom left. If the mouse is there, it will take the top left. Then the progress message wants the bottom right, but will be pushed up to get out of the way of link text, and/or the mouse cursor.
    *
    * On an unrealistically tiny screen, that could be imperfect.
    *
    * If we’re on a touch screen, we shouldn’t take the mouse cursor seriously unless it’s being pushed, but we don’t know that we’re on a touch screen. If it were incredibly important, we could ask. If it were fairly important, we could guess from seeing lots of drags and few or no moves.
    */
   bool link_at_top = false;

   if (link_dest.str && *link_dest.str) {
      int total_width, lines, line_height, drawTextX, drawTextY;

      fl_font(FL_HELVETICA, FL_NORMAL_SIZE);

      // How much room do we need for this message? It uses references, not ptrs, and we initialize width to tell it how much room we have.
      total_width = boundsW;
      fl_measure(link_dest.str, total_width, line_height, 0);

      lines = (int)ceil((double)total_width / boundsW);

      link_dest.drawH = lout::misc::min(lines * line_height, boundsH/3); // limit link text to 1/3 of the viewport
      link_dest.drawX = boundsX;
      link_dest.drawY = boundsY + boundsH - link_dest.drawH; // default to the bottom
      link_dest.drawW = lines > 1 ? boundsW : total_width;

      // if the mouse cursor’s there, go to the top instead
      if (mouseX >= link_dest.drawX && mouseX < link_dest.drawX + link_dest.drawW &&
          mouseY >= link_dest.drawY && mouseY < link_dest.drawY + link_dest.drawH) {
         link_at_top = true;
         link_dest.drawY = boundsY;
      }

      fl_push_clip(link_dest.drawX, link_dest.drawY, link_dest.drawW, link_dest.drawH);

      // Fill in the full lines and a partial last line.
      fl_color(FL_BACKGROUND_COLOR);
      if (lines > 1)
         fl_rectf(link_dest.drawX, link_dest.drawY, boundsW, line_height * (lines - 1));
      // I had % to calculate width, which was wrong when the line fit the viewport exactly.
      fl_rectf(link_dest.drawX, link_dest.drawY + line_height * (lines - 1), total_width - (lines - 1) * boundsW, line_height);

      // Since FLTK only wraps words, we’re going to keep going down and to the left, and let clipping do our job for us
      fl_color(FL_FOREGROUND_COLOR);
      drawTextX = link_dest.drawX;
      drawTextY = link_dest.drawY;
      while (lines-- > 0) {
         // when not using ALIGN, get the base with fl_height() - fl_descent()
         fl_draw(link_dest.str, drawTextX, drawTextY, total_width, line_height, FL_ALIGN_TOP_LEFT, 0, 0);
         drawTextX -= boundsW;
         drawTextY += line_height;
      }
      fl_pop_clip();
   }

   if (progress_msg.str && *progress_msg.str) {
      int completedW;

      progress_msg.drawW = boundsW;
      fl_font(FL_HELVETICA, FL_NORMAL_SIZE);

      // How much room do we need for this message? It uses references, not ptrs, and we initialized width to tell it how much room we have.
      fl_measure(progress_msg.str, progress_msg.drawW, progress_msg.drawH, 0);
      completedW = progress * progress_msg.drawW;

      progress_msg.drawW = lout::misc::min(boundsW, progress_msg.drawW);
      progress_msg.drawH = lout::misc::min(boundsH, progress_msg.drawH);
      progress_msg.drawX = boundsX + boundsW - progress_msg.drawW;
      progress_msg.drawY = boundsY + boundsH - progress_msg.drawH;

      if (!link_at_top && link_dest.str && *link_dest.str && link_dest.drawX + link_dest.drawW > progress_msg.drawX) {
         // overlapping with link? Then push progress_msg higher.
         progress_msg.drawY = link_dest.drawY - progress_msg.drawH;
      }

      if (mouseX >= progress_msg.drawX && mouseX < progress_msg.drawX + progress_msg.drawW &&
          mouseY >= progress_msg.drawY && mouseY < progress_msg.drawY + progress_msg.drawH) {
         // overlapping with mouse? Push progress msg still higher.
         progress_msg.drawY = mouseY - progress_msg.drawH;
      }

      // This doesn’t cover every possible case. If I ever feel driven to fill in every case, remember there must be something more important I could be doing.

      fl_color(FL_BACKGROUND_COLOR);
      fl_rectf(progress_msg.drawX, progress_msg.drawY, completedW, progress_msg.drawH);
      fl_color(FL_BACKGROUND2_COLOR);
      fl_rectf(progress_msg.drawX + completedW, progress_msg.drawY, progress_msg.drawW - completedW, progress_msg.drawH);
      fl_color(FL_FOREGROUND_COLOR);
      // when not using ALIGN, get the base with fl_height() - fl_descent()
      fl_draw(progress_msg.str, progress_msg.drawX, progress_msg.drawY, progress_msg.drawW, progress_msg.drawH, FL_ALIGN_TOP_LEFT|FL_ALIGN_WRAP|FL_ALIGN_CLIP, 0, 0);
   }
}

void OverlayWindow::set_overlay_link_dest(const char *s)
{
   if (((!s || !*s) && (!link_dest.str || !*link_dest.str)) ||
       (s && link_dest.str &&!strcmp(s, link_dest.str))) {
      _MSG("re-setting the same link message!\n");
   } else {
      if (link_dest.str)
         free(link_dest.str);
      this->link_dest.str = (s && *s) ? strdup(s) : NULL;
      redraw_overlay();
   }
}

void OverlayWindow::set_overlay_progress_msg(const char *s, double proportion)
{
   if ((((!s || !*s) && (!progress_msg.str || !*progress_msg.str)) ||
        (s && progress_msg.str && !strcmp(s, progress_msg.str))) &&
       this->progress == proportion) {
      _MSG("re-setting the same progress message!\n");
   } else {
      Fl::remove_timeout(OverlayWindow_clear_overlay_progress_msg_handler, this);
      // 3 seconds
      Fl::add_timeout(3.0, OverlayWindow_clear_overlay_progress_msg_handler, this);

      if (this->progress_msg.str)
         free(this->progress_msg.str);
      this->progress_msg.str = (s && *s) ? strdup(s) : NULL;
      this->progress = proportion;
      redraw_overlay();
   }
}

void OverlayWindow::update_overlay_bounds(int boundsX, int boundsY, int boundsW, int boundsH)
{
   if (this->boundsX == boundsX &&
      this->boundsY == boundsY &&
      this->boundsW == boundsW &&
      this->boundsH == boundsH) {
      return;
   }

   this->boundsX = boundsX;
   this->boundsY = boundsY;
   this->boundsW = boundsW;
   this->boundsH = boundsH;

   // We could try to be very clever about avoiding unnecessary redraws, but I don’t want to introduce a bug right now, and I might be able to scrap OverlayWindow anyway if I could make a Fl_Menu_Window/Fl_Tooltip overlay work perfectly.

   redraw_overlay();
}

void OverlayWindow::clear_entire_overlay()
{
   bool changed = false;

   if (link_dest.str) {
      changed = true;
      free(link_dest.str);
      link_dest.str = NULL;
   }
   if (progress_msg.str) {
      changed = true;
      Fl::remove_timeout(OverlayWindow_clear_overlay_progress_msg_handler, this);
      free(progress_msg.str);
      progress_msg.str = NULL;
   }
   if (changed) {
      redraw_overlay();
   }
}

void OverlayWindow::clear_overlay_progress_msg()
{
   if (progress_msg.str) {
      Fl::remove_timeout(OverlayWindow_clear_overlay_progress_msg_handler, this);
      free(progress_msg.str);
      progress_msg.str = NULL;
      redraw_overlay();
   }
}

void OverlayWindow::mouse_moved(int x, int y)
{
   drawn_msg *m = &link_dest;

   mouseX = x;
   mouseY = y;

   if (m->str && *m->str && mouseX >= m->drawX && mouseX < m->drawX + m->drawW &&
       mouseY >= m->drawY && mouseY < m->drawY + m->drawH) {
      _MSG("mouse was trying to get our link_dest\n");
   } else {
      m = &progress_msg;

      if (m->str && *m->str && mouseX >= m->drawX && mouseX < m->drawX + m->drawW &&
          mouseY >= m->drawY && mouseY < m->drawY + m->drawH) {
         _MSG("mouse was trying to get our progress_msg\n");
      } else {
         return;
      }
   }
   redraw_overlay();
}
