/* This was CustTextEditor from fltkui.cc, which has at the top:
 *
 * Dillo Widget
 *
 * Copyright 2005-2007 Sebastian Geerken <sgeerken@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <FL/Fl.H>

#include <stdlib.h>

#include "fltkmisc.hh"
#include "placeholder_editor.hh"

PlaceholderEditor::PlaceholderEditor (int x, int y, int w, int h, const char* l) :
                                      BaseTextEditor(x,y,w,h,l)
{
   placeholder = NULL;
   showing_placeholder = false;
   usual_color = FL_BLACK;      /* just init until widget style is set */
}

PlaceholderEditor::~PlaceholderEditor ()
{
   if (placeholder)
      free(placeholder);
}

/*
 * Show normal text.
 */
void PlaceholderEditor::show_normal(const char *str)
{
   showing_placeholder = false;
   Fl_Text_Editor::textcolor(usual_color);
   buffer()->text(str);
}

/*
 * Show the placeholder text.
 */
void PlaceholderEditor::show_placeholder()
{
   showing_placeholder = true;
   Fl_Text_Editor::textcolor(dw::fltk::misc::dimmedColor(usual_color, color()));
   buffer()->text(placeholder);
}

/*
 * Set the placeholder text.
 */
void PlaceholderEditor::set_placeholder(const char *str)
{
   if (placeholder)
      free(placeholder);
   placeholder = strdup(str);

   if ((Fl::focus() != this) && buffer()->length() == 0) {
      show_placeholder();
   }
}

/*
 * Set the text color.
 */
void PlaceholderEditor::textcolor(Fl_Color c)
{
   usual_color = c;
   if (showing_placeholder)
      c = dw::fltk::misc::dimmedColor(c, color());
   Fl_Text_Editor::textcolor(c);
}

/*
 * Set the value of the input.
 */
void PlaceholderEditor::value(const char *str)
{
   if (placeholder && (!str || !*str) && Fl::focus() != this)
      show_placeholder();
   else
      show_normal(str);
}

/*
 * Return the value (text) of the input.
 */
char* PlaceholderEditor::value()
{
   /* FLTK-1.3 insists upon returning a new copy of the buffer text, so
    * we have to keep track of it.
    */
   if (text_copy)
      free(text_copy);
   text_copy = showing_placeholder ? strdup("") : buffer()->text();
   return text_copy;
}

bool PlaceholderEditor::empty()
{
   return showing_placeholder || !buffer()->char_at(0);
}

int PlaceholderEditor::handle(int e)
{
   int rc;

   if (e == FL_UNFOCUS) {
      if (placeholder && buffer()->length() == 0) {
         show_placeholder();
      }
   }

   rc = BaseTextEditor::handle(e);

   if (rc && e == FL_FOCUS) {
      // Nonzero return from handle() should mean that focus was accepted.
      if (showing_placeholder)
         show_normal("");
   }
   return rc;
}

