/* This was CustInput from fltkui.cc, which has at the top:
 *
 * Dillo Widget
 *
 * Copyright 2005-2007 Sebastian Geerken <sgeerken@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <FL/Fl.H>

#include <stdlib.h>

#include "../lout/msg.h"
#include "fltkmisc.hh"
#include "placeholder_input.hh"

PlaceholderInput::PlaceholderInput (int x, int y, int w, int h, const char* l) :
   BaseInput(x,y,w,h,l)
{
   placeholder = NULL;
   showing_placeholder = false;
   usual_color = FL_BLACK;      /* just init until widget style is set */
   input_type(FL_NORMAL_INPUT);
}

PlaceholderInput::~PlaceholderInput ()
{
   if (placeholder)
      free(placeholder);
}

/*
 * Show normal text.
 */
int PlaceholderInput::show_normal(const char *str)
{
   showing_placeholder = false;
   Fl_Input::textcolor(usual_color);
   Fl_Input::input_type(usual_type);
   return Fl_Input::value(str);
}

/*
 * Show the placeholder text.
 */
int PlaceholderInput::show_placeholder()
{
   int ret;

   showing_placeholder = true;
   Fl_Input::textcolor(dw::fltk::misc::dimmedColor(usual_color, color()));
   Fl_Input::input_type(FL_NORMAL_INPUT);
   ret = Fl_Input::value(placeholder);
   insert_position(0);
   return ret;
}

/*
 * Set the placeholder text.
 */
void PlaceholderInput::set_placeholder(const char *str)
{
   if (placeholder)
      free(placeholder);
   placeholder = strdup(str);

   if ((Fl::focus() != this) && !*value()) {
      show_placeholder();
   }
}

/*
 * Set the text color.
 */
void PlaceholderInput::textcolor(Fl_Color c)
{
   usual_color = c;
   if (showing_placeholder)
      c = dw::fltk::misc::dimmedColor(c, color());
   Fl_Input::textcolor(c);
}

/*
 * Set the input type (normal, password, etc.)
 */
void PlaceholderInput::input_type(int t)
{
   usual_type = t;
   Fl_Input::input_type(t);
}

/*
 * Set the value of the input.
 * NOTE that we're not being very careful with the return value, which is
 * supposed to be nonzero iff the value was changed.
 */
int PlaceholderInput::value(const char *str)
{
   return (placeholder && (!str || !*str) && Fl::focus() != this)
          ? show_placeholder() : show_normal(str);
}

/*
 * Return the value (text) of the input.
 */
const char* PlaceholderInput::value()
{
   return showing_placeholder ? "" : Fl_Input::value();
}

int PlaceholderInput::handle(int e)
{
   int rc, k = Fl::event_key();

   _MSG("PlaceholderInput::handle event=%d\n", e);

   // We're only interested in some flags
   unsigned modifier = Fl::event_state() & (FL_SHIFT | FL_CTRL | FL_ALT);

   if (e == FL_KEYBOARD) {
      if (k == FL_Page_Down || k == FL_Page_Up || k == FL_Up || k == FL_Down) {
         // Let them through for key commands and viewport motion.
         return 0;
      }
      if (modifier == FL_CTRL) {
         if (k == 'h' || k == 'i' || k == 'j' || k == 'l' || k == 'm') {
            // Fl_Input wants to use ^H as backspace, and also "insert a few
            // selected control characters literally", but this gets in the way
            // of key commands.
            return 0;
         }
      }
   } else if (e == FL_UNFOCUS) {
      if (placeholder && !value()[0]) {
         show_placeholder();
      }
   }

   rc = BaseInput::handle(e);

   if (rc && e == FL_FOCUS) {
      // Nonzero return from handle() should mean that focus was accepted.
      if (showing_placeholder)
         show_normal("");
   }
   return rc;
}

