/*
 * File: curl.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

#include <curl/curl.h>
#include <sys/select.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>

#include "../capi.h"
#include "../cookies.h"
#include "../auth.h"
#include "../timeout.hh"
#include "../url.h"
#include "../msg.h"
#include "../prefs.h"
#include "../referrer.h"
#include "http.h"
#include "iowatch.hh"

struct my_dumb_data {
   ulong_t req_id;
   struct curl_slist *headers;
   CURL *handle;
   DilloUrl *url;
   char errbuf[CURL_ERROR_SIZE];
   curl_off_t body_bytes_received;
};

typedef struct {
   char *host;
   char *root_cert_authority;
} host_issuer_t;

/* https://developer.mozilla.org/en-US/docs/Web/HTTP/Guides/Content_negotiation/List_of_default_Accept_values */

static const char *accept_default = "Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8";

/* Beginning with Firefox 92, they put "image/avif" ahead of any of these */
static const char *accept_image = "Accept: "
#ifdef ENABLE_WEBP
"image/webp,"
#endif
"image/png,"
#ifdef ENABLE_SVG
"image/svg+xml,"
#endif
"image/*;q=0.8,*/*;q=0.5";

static const char *accept_stylesheet = "Accept: text/css,*/*;q=0.1";

static curl_version_info_data *curl_version_data = NULL;
static CURLM *multi = NULL;
static Dlist *conns = NULL;
static Dlist *host_issuers; // currently used for informational rather than operational purposes
static long tls_min_version = CURL_SSLVERSION_TLSv1_2;

static int Curl_dumb_data_by_url_cmp(const void *v_data, const void *v_url) {
   const struct my_dumb_data *data = (const struct my_dumb_data *) v_data;
   const DilloUrl *url = (const DilloUrl *) v_url;

   return a_Url_cmp(url, data->url);
}

static int Curl_dumb_data_cmp(const void *v_data1, const void *v_data2) {
   const struct my_dumb_data *data1 = (const struct my_dumb_data *) v_data1,
                             *data2 = (const struct my_dumb_data *) v_data2;

   return a_Url_cmp(data1->url, data2->url);
}


static void Curl_append_certs(CURL *c, Dstr *str)
{
   struct curl_certinfo *ci;
   int i;
   CURLcode res = curl_easy_getinfo(c, CURLINFO_CERTINFO, &ci);

   if (!res && ci->num_of_certs) {
      dStr_append(str, "\nWe got a chain of certificates:\n");

      for(i = 0; i < ci->num_of_certs; i++) {
         struct curl_slist *slist;

         for(slist = ci->certinfo[i]; slist; slist = slist->next) {
            dStr_append(str, slist->data);
            dStr_append_c(str, '\n');
         }
      }
   }
}

static void Curl_print_messages()
{
   struct CURLMsg *m;
   int msgs_queued;

   do {
      m = curl_multi_info_read(multi, &msgs_queued);

      if (m){
         if (m->msg == CURLMSG_DONE) {
            _MSG("curlmsg done\n");
            struct my_dumb_data *data = NULL;
            CURL *c = m->easy_handle;
            CURLcode rc = m->data.result;

            curl_multi_remove_handle(multi, c);

            if (curl_easy_getinfo(c, CURLINFO_PRIVATE, &data)) {
               MSG_ERR("guess we won’t be freeing our private data.\n");
               continue;
            }

            /* Initially I was just making a DilloUrl from CURLINFO_EFFECTIVE_URL,
             * which was a subtle error in the case of POST URLs, where Dillo
             * keeps the body of a request within a DilloUrl and sees that as distinct
             * from a plain, newly-minted DilloUrl made from a URL string.
             */
            _MSG("%s completed\n", URL_STR(data->url));

            if (rc) {
               if (data->body_bytes_received) {
                  // https://airenetworks.es/ successfully sent the page but I also had "Failed writing received data to disk/application" at the bottom of the page. That’s CURLE_WRITE_ERROR, which I believe came from something being wrong in that site’s compression, as the problem went away when I disabled CURLOPT_ACCEPT_ENCODING.

                  MSG_WARN("libcurl reported an error for '%s': %s\n", URL_STR(data->url), data->errbuf[0] ? data->errbuf : curl_easy_strerror(rc));
               } else {
                  // If we didn’t receive a body, let’s print out the error there.
                  _MSG("%s completed, result %d: %s.\n", URL_STR(data->url), rc,
                       curl_easy_strerror(rc));
                  const char *h1 = "HTTP/1.1 200 OK\r\n";
                  const char *h2 = "Content-Type: text/html\r\n";
                  Dstr *body = dStr_new("<p><p><p>\n");

                  if (rc == CURLE_ABORTED_BY_CALLBACK) {
                     // Special message because otherwise we show the libcurl error about the pre-request callback causing an abort, which would not be a useful explanation.
                     dStr_append(body, "Site blocked. Following the lead of Chrome and Firefox, we don’t trust TLS certificates issued by Entrust/AffirmTrust after 20241111.<br>\n");
                  } else {
                     dStr_append(body, "Failed. libcurl reported:<p>\"");

                     if (data->errbuf[0])
                        dStr_append(body, data->errbuf);
                     else
                        dStr_append(body, curl_easy_strerror(rc));
                     dStr_append(body, "\"<br>\n");
                  }

                  dStr_append(body, "<pre>\n");
                  Curl_append_certs(c, body);
                  dStr_append(body, "</pre>\n");

                  a_Capi_reply_header(data->req_id, h1, strlen(h1));
                  a_Capi_reply_header(data->req_id, h2, strlen(h2));
                  a_Capi_reply_body(data->req_id, body->str, body->len);
                  dStr_free(body, 1);
               }
            }
#if 0
            if (dStrAsciiCasecmp(URL_SCHEME(data->url), "file")) {
               char *str;
               curl_easy_getinfo(c, CURLINFO_PRIMARY_IP, &str);
               MSG("%s %s\n", URL_HOST(data->url), str);
            }
#endif
            a_Capi_reply_finished(data->req_id);

            if (data) {
               dList_remove(conns, data);
               if (data->headers)
                  curl_slist_free_all(data->headers);
               a_Url_free(data->url);
               dFree(data);
            }
            curl_easy_cleanup(c);
         } else {
            MSG("curl some other message.\n");
         }
      }
   } while(m);
}

static void Curl_lets_read_cb(int fd, void *data)
{
   int active;
   _MSG("Curl read cb %d ", fd);
   curl_multi_socket_action(multi, fd, CURL_CSELECT_IN, &active);
   Curl_print_messages();
}

static void Curl_lets_write_cb(int fd, void *data)
{
   int active;
   _MSG("Curl write cb%d ", fd);
   curl_multi_socket_action(multi, fd, CURL_CSELECT_OUT, &active);
   Curl_print_messages();
}

static void Curl_lets_rw_cb(int fd, void *data)
{
   int active;
   _MSG("Curl rw cb on %d. ", fd);
   curl_multi_socket_action(multi, fd, CURL_CSELECT_IN|CURL_CSELECT_OUT, &active);
   Curl_print_messages();
}

static int socket_cb(CURL *c, curl_socket_t fd, int wants, void *unused, void *unused2)
{
   if (wants == CURL_POLL_REMOVE) {
      _MSG("remove fd %d\n", fd);
      a_IOwatch_remove_fd(fd, DIO_READ|DIO_WRITE);
   } else {
      if (wants == CURL_POLL_IN) {
         _MSG("watch to read fd %d\n", fd);
         a_IOwatch_remove_fd(fd, DIO_READ|DIO_WRITE);
         a_IOwatch_add_fd(fd, DIO_READ, Curl_lets_read_cb, NULL);
      } else if (wants == CURL_POLL_OUT) {
         _MSG("watch to write fd %d\n", fd);
         a_IOwatch_remove_fd(fd, DIO_READ|DIO_WRITE);
         a_IOwatch_add_fd(fd, DIO_WRITE, Curl_lets_write_cb, NULL);
      } else if (wants == CURL_POLL_INOUT) {
         _MSG("watch to read/write fd %d\n", fd);
         a_IOwatch_remove_fd(fd, DIO_READ|DIO_WRITE);
         a_IOwatch_add_fd(fd, DIO_READ|DIO_WRITE, Curl_lets_rw_cb, NULL);
      } else {
         MSG_ERR("socket_cb wants %d\n", wants);
         /* There’s probably a return value for this; doesn’t seem documented if so, although only looked for a moment. */
      }
   }
   return 0;
}

static void timeout_cb_cb(void *vp)
{
   int active;
   CURLMcode rc;

   struct timeval clock;
   gettimeofday(&clock, NULL);

   _MSG("%ld.%06ld: The timeout time came! Perform the action...\n", clock.tv_sec, clock.tv_usec);
   rc = curl_multi_socket_action(multi, CURL_SOCKET_TIMEOUT, 0, &active);
   if (rc)
      MSG("curl_multi_socket_action got %d. There are now %d active handles.\n", rc, active);
   Curl_print_messages();
}

static int timer_cb(CURLM *m, long timeout_in_ms, void *unused)
{
   if (timeout_in_ms == -1) {
      _MSG("delete our timer.\n");
      a_Timeout_remove(timeout_cb_cb);
   } else {
      _MSG("timeout in %f s\n", ((double)timeout_in_ms) / 1000.0);
      a_Timeout_add(((double)timeout_in_ms) / 1000.0, timeout_cb_cb, NULL);
   }
   return 0;
}

int a_Curl_init(void)
{
   curl_version_data = curl_version_info(CURLVERSION_NOW);

   MSG("curl: %s.\n", curl_version_data->version);
   if (strlen(CA_CERTS_FILE))
      MSG("Root certificate bundle was set at configure time to: " CA_CERTS_FILE);
   if (strlen(CA_CERTS_DIR))
      MSG("Root certificate path was set at configure time to: " CA_CERTS_DIR);

   if (!strcmp(prefs.tls_min_version, "1.3"))
      tls_min_version = CURL_SSLVERSION_TLSv1_3;
   else if (!strcmp(prefs.tls_min_version, "1.2"))
      tls_min_version = CURL_SSLVERSION_TLSv1_2;
   else if (!strcmp(prefs.tls_min_version, "1.1"))
      tls_min_version = CURL_SSLVERSION_TLSv1_1;
   else if (!strcmp(prefs.tls_min_version, "1.0"))
      tls_min_version = CURL_SSLVERSION_TLSv1_0;
   else if (!strcmp(prefs.tls_min_version, "default"))
      tls_min_version = CURL_SSLVERSION_DEFAULT;
   else {
      MSG_ERR("Could not set minimum TLS version '%s'. libcurl uses an enum, and the maximum value when this was written was TLS 1.3\n", prefs.tls_min_version);
      return 0;
   }
   MSG("TLS: configured to use %s or newer.\n", prefs.tls_min_version);

   if (curl_global_init(CURL_GLOBAL_ALL))
      MSG_ERR("libcurl couldn't be initialized for...some unknown reason.\n");
   else {
      multi = curl_multi_init();
      if (curl_multi_setopt(multi, CURLMOPT_MAX_HOST_CONNECTIONS, prefs.http_max_conns))
         MSG_ERR("maxconns failed\n");
      if (curl_multi_setopt(multi, CURLMOPT_PIPELINING, CURLPIPE_NOTHING))
         MSG_ERR("pipelining failed\n");
      if (curl_multi_setopt(multi, CURLMOPT_SOCKETFUNCTION, socket_cb))
         MSG_ERR("socketfunction failed\n");
      if (curl_multi_setopt(multi, CURLMOPT_TIMERFUNCTION, timer_cb))
         MSG_ERR("timerfunction failed\n");
   }

   conns = dList_new(24);
   host_issuers = dList_new(16);
   return 0;
}

static size_t Curl_receiving_header(void *buf, size_t sz, size_t len, void *vp)
{
   ulong_t req_id = (ulong_t)vp;

   _MSG("%zu header bytes.\n", len);
   a_Capi_reply_header(req_id, buf, len);
   return len;
}

static size_t Curl_receiving_data(void *buf, size_t sz, size_t len, void *vp)
{
   int ret;
   ulong_t req_id = (ulong_t)vp;

   _MSG("%zu data bytes.\n", len);
   ret = a_Capi_reply_body(req_id, buf, len);

   /* returning a different len aborts the transfer. Beginning in late 2022, CURL_WRITEFUNC_ERROR was added, but that wasn’t so long ago as I write this. */
   return (ret ? len+1 : len);
}

static int Curl_xferinfo(void *vp, curl_off_t dltotal, curl_off_t dlnow, curl_off_t ultotal, curl_off_t ulnow)
{
   (void)ultotal, (void)ulnow;

   struct my_dumb_data *data = (struct my_dumb_data *)vp;

   if (dlnow > data->body_bytes_received) {
      data->body_bytes_received = dlnow;
      a_Capi_report_progress(data->req_id, dlnow, dltotal);
   }
   return 0;
}

static int Curl_host_issuer_cmp_by_host(const void *v1, const void *v2)
{
   const host_issuer_t *node = v1;
   const char *host = v2;

   return dStrAsciiCasecmp(node->host, host);
}

static int Curl_host_issuer_cmp(const void *v1, const void *v2)
{
   const host_issuer_t *n1 = v1, *n2 = v2;

   return dStrAsciiCasecmp(n1->host, n2->host);
}

static void Curl_host_set_issuer(const char *host, const char *issuer)
{
   host_issuer_t *hi = dList_find_sorted(host_issuers, host, Curl_host_issuer_cmp_by_host);

   if (hi) {
      if (dStrAsciiCasecmp(hi->root_cert_authority, issuer)) {
         MSG_WARN("The authority issuing the TLS certificate for host '%s' changed from '%s' to '%s'.\n", host, hi->root_cert_authority, issuer);
         dFree(hi->root_cert_authority);
         hi->root_cert_authority = dStrdup(issuer);
      }
   } else {
      hi = dNew(host_issuer_t, 1);
      hi->host = dStrdup(host);
      hi->root_cert_authority = dStrdup(issuer);
      dList_insert_sorted(host_issuers, hi, Curl_host_issuer_cmp);
   }
}

/* What are we doing with the prereq callback?
 * Checking out the certificate chain,
 * tracking the certification authorities we are trusting, and
 * deciding whether we accept Entrust’s authority.
 *
 * NOTE If/when Entrust becomes completely irrelevant (end of 2025?), we could remove this function and move the tracking into something called by the CURLMSG_DONE code later on.
 */
static int Curl_prereq(void *vp, char *, char *, int, int)
{
   struct curl_slist *slist;
   CURL *c = (CURL *)vp;

   struct curl_certinfo *ci;
   CURLcode res = curl_easy_getinfo(c, CURLINFO_CERTINFO, &ci);

   if (res == CURLE_OK && ci->num_of_certs) {
      slist = ci->certinfo[ci->num_of_certs - 1];
      while (strncmp(slist->data, "Issuer:", 7))
            slist = slist->next;

      if (!slist) {
         MSG_ERR("Couldn’t find libcurl’s issuer field for TLS certificate\n");
         return CURL_PREREQFUNC_OK; // We weren’t able to demonstrate that they are using Entrust.
      } else {
         // Dillo doesn’t need this to function, but I’m tracking which root certificates are actually used.
         char *root_cert_name;
         const struct my_dumb_data *data = NULL;
         const char *p, *ca_name = strstr(slist->data, "CN = ");

         // With most CAs you can get a detailed name from CN, but GlobalSign has it under OU
         if (ca_name)
            ca_name += 5;
         if (!ca_name || !strcmp(ca_name, "GlobalSign")) {
            ca_name = strstr(slist->data, "OU = ");
            if (ca_name)
               ca_name += 5;
         }
         if (!ca_name)
            ca_name = "???";
         p = ca_name;
         while (*p && *p != ',')
            p++;
         root_cert_name = dStrndup(ca_name, p - ca_name);

         if (curl_easy_getinfo(c, CURLINFO_PRIVATE, &data) == CURLE_OK) {
            Curl_host_set_issuer(URL_HOST(data->url), root_cert_name);
         }
         dFree(root_cert_name);
      }

      if (strstr(slist->data, "Entrust") || strstr(slist->data, "AffirmTrust")) {
         MSG("AN ENTRUST CERTIFICATE\n");
         const char *s;

         slist = ci->certinfo[0];
         while (strncmp(slist->data, "CT Precertificate SCTs:", 23))
            slist = slist->next;

         if (!slist) {
            MSG_ERR("Couldn’t find Entrust CT Precertificate SCTs\n");
            return CURL_PREREQFUNC_ABORT;
         }

         s = slist->data;

         while ((s = strstr(s, "Timestamp : "))) {
            int year = 0, day = 0;
            char month[4] = "xxx";

            s += 12;
            // format arrived at by observation of output, which looks like "Oct  4 05:15:29.787 2024 GMT"
            if (sscanf(s, "%3c %2d %*2d:%*2d:%*2d.%*3d %4d %*3c", month, &day, &year) != 3) {
               MSG_ERR("Could not parse timestamp for TLS certificate\n");
               return CURL_PREREQFUNC_ABORT;
            }
            MSG("Timestamp %d %s %d\n", year, month, day);
            if (year < 2024 ||
                (year == 2024 &&
                 (strcmp(month, "Dec") &&
                  !(!strcmp(month, "Nov") && day >= 12)))) {
               MSG("Certificate timestamp is early enough\n");
               return CURL_PREREQFUNC_OK;
            }
         }
         MSG("Certificate timestamp after 20241111 is rejected!\n");
         return CURL_PREREQFUNC_ABORT;
      }
   }
   return CURL_PREREQFUNC_OK;
}

static int debug_cb(CURL *handle, curl_infotype type, char *data, size_t size, void *userptr)
{
   static size_t data_used = 0; /* at least at the HTTP/TLS level */
#if 0
   switch (type) {
      case CURLINFO_TEXT:
         char *s = dStrndup(data, size);
         MSG("DEBUG: TEXT %s", s);
         dFree(s);
         break;
      case CURLINFO_HEADER_IN:
         MSG("DEBUG: HEADER_IN");
         break;
      case CURLINFO_HEADER_OUT:
         MSG("DEBUG: HEADER_OUT");
         break;
      case CURLINFO_DATA_IN:
         MSG("DEBUG: DATA_IN");
         break;
      case CURLINFO_DATA_OUT:
         MSG("DEBUG: DATA_OUT");
         break;
      case CURLINFO_SSL_DATA_IN:
         MSG("DEBUG: SSL_DATA_IN");
         break;
      case CURLINFO_SSL_DATA_OUT:
         MSG("DEBUG: SSL_DATA_OUT");
         break;
      default:
         MSG("DEBUG: BROKEN");
   }
#endif
   if (type == CURLINFO_HEADER_IN || type == CURLINFO_HEADER_OUT || type == CURLINFO_DATA_IN || type == CURLINFO_DATA_OUT || type == CURLINFO_SSL_DATA_IN || CURLINFO_SSL_DATA_OUT) {
      const size_t megabytes = (data_used + size) / (1024 * 1024);

      if (megabytes > data_used / (1024 * 1024))
         MSG("Total %zu megabyte%s transferred.\n", megabytes, megabytes > 1 ? "s" : "");
      data_used += size;
   }
   return 0;
}

static void Curl_handle_invariant_initialization(CURL *c)
{
   if (curl_easy_setopt(c, CURLOPT_VERBOSE, 1L))
      MSG_ERR("verbose failed\n");
   if (curl_easy_setopt(c, CURLOPT_DEBUGFUNCTION, debug_cb))
      MSG_ERR("debugfunction failed\n");
   if (curl_easy_setopt(c, CURLOPT_WRITEFUNCTION, Curl_receiving_data))
      MSG_ERR("writefunction failed\n");
   if (curl_easy_setopt(c, CURLOPT_HEADERFUNCTION, Curl_receiving_header))
      MSG_ERR("headerfunction failed\n");
   if (curl_easy_setopt(c, CURLOPT_XFERINFOFUNCTION, Curl_xferinfo))
      MSG_ERR("headerfunction failed\n");
   if (curl_easy_setopt(c, CURLOPT_NOPROGRESS, 0L))
      MSG_ERR("noprogress failed\n");

   if (curl_easy_setopt(c, CURLOPT_CONNECTTIMEOUT, prefs.connect_timeout))
      MSG_ERR("connecttimeout failed\n");

#if CURL_AT_LEAST_VERSION(7,65,2)
   /* CURLOPT_MAXAGE_CONN introduced with curl 7.65.0; broken until
    * 7.65.2. By default, curl will reuse a connection up to 118 seconds
    * old, and sometimes it can’t tell when a connection is dead. Let’s
    * see whether a smaller value helps.
    */
   if (curl_easy_setopt(c, CURLOPT_MAXAGE_CONN, 30L))
      MSG_ERR("maxage_conn failed\n");
#else
   /* It seems connections could get arbitrarily old before
    * 7.65.0: https://curl.se//mail/lib-2019-04/0040.html
    */
   if (curl_easy_setopt(c, CURLOPT_FORBID_REUSE, 1L))
      MSG_ERR("forbid_reuse failed\n");
#endif

   if (curl_easy_setopt(c, CURLOPT_FOLLOWLOCATION, 0L))
      MSG_ERR("followlocation failed\n");

   /* DNS-OVER-HTTPS AND THE TLS SETTINGS THAT PROBABLY APPLY */

   if (prefs.dns_over_https_url && prefs.dns_over_https_url[0] &&
       curl_easy_setopt(c, CURLOPT_DOH_URL, prefs.dns_over_https_url))
      MSG_ERR("doh_url failed\n");

   if (curl_easy_setopt(c, CURLOPT_CERTINFO, 1L))
      MSG_ERR("certinfo failed\n");
   if (curl_easy_setopt(c, CURLOPT_SSLVERSION, tls_min_version))
      MSG_ERR("sslversion failed\n");
   if (strlen(CA_CERTS_FILE) && curl_easy_setopt(c, CURLOPT_CAINFO, CA_CERTS_FILE))
         MSG_ERR("cainfo failed\n");
   if (strlen(CA_CERTS_DIR) && curl_easy_setopt(c, CURLOPT_CAPATH, CA_CERTS_DIR))
         MSG_ERR("capath failed\n");

   if (prefs.tls_cipher_list && prefs.tls_cipher_list[0] &&
       curl_easy_setopt(c, CURLOPT_SSL_CIPHER_LIST, prefs.tls_cipher_list))
      MSG_ERR("ssl_cipher_list failed\n");
   if (curl_easy_setopt(c, CURLOPT_SSL_SESSIONID_CACHE, 0L))
      MSG_ERR("ssl_sessionid_cache failed\n");
}

static void Curl_handle_POST_initialization(CURL *c, struct my_dumb_data *data, const DilloUrl *url)
{
   /* Interestingly, you can POST to non-HTTP URLs such as files. Or at least Firefox also allows it. Whatever you POST is ignored, and the file is the response. */
   if (curl_easy_setopt(c, CURLOPT_POST, 1L))
      MSG_ERR("post failed\n");
   if (curl_easy_setopt(c, CURLOPT_POSTFIELDSIZE, URL_DATA(url)->len))
      MSG_ERR("postfieldsize failed\n");
   if (curl_easy_setopt(c, CURLOPT_COPYPOSTFIELDS, URL_DATA(url)->str))
      MSG_ERR("copypostfields failed\n");

   MSG("POST: %s\n", dStr_printable(URL_DATA(url), 350)); /* nothing special about length of 350 for how much data to show */

   /* For multipart/form-data POSTed data, we have to construct our own
    * header. libcurl has a mime mechanism to do this stuff, but that’s
    * only good if we were back at form.cc.
    */
   if (URL_FLAGS(url) & URL_MultipartEnc) {
      Dstr *dstr = dStr_new("Content-Type: multipart/form-data; boundary=\"");
      if (URL_DATA(url)->len > 2) {
         /* boundary lines have "--" prepended. Skip that. */
         const char *start = URL_DATA(url)->str + 2;
         char *eol = strchr(start, '\r');
         if (eol)
            dStr_append_l(dstr, start, eol - start);
      } else {
         /* Zero parts; arbitrary boundary */
         dStr_append_c(dstr, '0');
      }
      dStr_append_c(dstr,'"');

      data->headers = curl_slist_append(data->headers, dstr->str);
      dStr_free(dstr, 1);
   }
}

static void Curl_handle_http_https_initialization(CURL *c, struct my_dumb_data *data, const FetchReq *f)
{
   const char *http_language, *http_accept;
   const char *http_proxy, *http_proxy_auth;
   const char *http_auth_username, *http_auth_password;
   enum AuthParseHTTPAuthType_t authtype;

   char *header, *http_referer, *cookies;
   bool_t is_https = !dStrAsciiCasecmp(URL_SCHEME(f->url), "https");

#if 0
   if (curl_easy_setopt(c, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1))
      MSG_ERR("http_version failed.\n");
#endif

   if (is_https) {
      // For our Entrust test, which we could get rid of roughly when 2025 ends.
      if (curl_easy_setopt(c, CURLOPT_PREREQFUNCTION, Curl_prereq))
         MSG_ERR("prereqfunction failed\n");
      if (curl_easy_setopt(c, CURLOPT_PREREQDATA, c))
         MSG_ERR("prereqdata failed\n");
   }

/* TODO this could use some disentanglement of the this-is-a-proxy-for-this sort */
   http_proxy = a_Http_get_proxy_urlstr();
   if (http_proxy) {
      http_proxy_auth = a_Http_get_proxy_authstr();

      if (curl_easy_setopt(c, CURLOPT_PROXY, http_proxy))
         MSG_ERR("proxy failed\n");

      if (curl_easy_setopt(c, CURLOPT_NOPROXY, prefs.no_proxy))
         MSG_ERR("noproxy failed\n");

      if (http_proxy_auth) {
         if (curl_easy_setopt(c, CURLOPT_PROXYUSERPWD, http_proxy_auth))
            MSG_ERR("proxy_auth failed\n");
      }

      if (is_https)
         if (curl_easy_setopt(c, CURLOPT_HTTPPROXYTUNNEL, 1L))
            MSG_ERR("httpproxytunnel failed\n");
   }

   http_accept = f->sec_fetch_dest == HTTP_Sec_Fetch_Dest_image ? accept_image :
                 f->sec_fetch_dest == HTTP_Sec_Fetch_Dest_style ? accept_stylesheet :
                 accept_default;
   data->headers = curl_slist_append(data->headers, http_accept);

   if (curl_easy_setopt(c, CURLOPT_ACCEPT_ENCODING, "")) /* everything that’s compiled into libcurl */
      MSG_ERR("accept_encoding failed\n");

   http_language = a_Http_get_language();
   if (http_language && *http_language)
      data->headers = curl_slist_append(data->headers, http_language);

   if (!is_https)
      data->headers = curl_slist_append(data->headers, "Upgrade-Insecure-Requests: 1");

   header = dStrconcat("Sec-Fetch-Dest: ", a_Http_sec_fetch_dest_str(f->sec_fetch_dest), NULL);
   data->headers = curl_slist_append(data->headers, header);
   dFree(header);

   header = dStrconcat("Sec-Fetch-Mode: ", a_Http_sec_fetch_mode_str(f->sec_fetch_dest, f->url, f->referrer), NULL);
   data->headers = curl_slist_append(data->headers, header);
   dFree(header);

   header = dStrconcat("Sec-Fetch-Site: ", a_Http_sec_fetch_site_str(f->url, f->referrer), NULL);
   data->headers = curl_slist_append(data->headers, header);
   dFree(header);

   if (f->sec_fetch_dest == HTTP_Sec_Fetch_Dest_document && f->user_saw_url)
      data->headers = curl_slist_append(data->headers, "Sec-Fetch-User: ?1");

   if (curl_easy_setopt(c, CURLOPT_USERAGENT, prefs.http_user_agent))
      MSG_ERR("useragent failed\n");

   http_referer = a_Referrer_make(f->referrerpolicy, f->url, f->referrer);

   if (http_referer)
      if (curl_easy_setopt(c, CURLOPT_REFERER, http_referer))
         MSG_ERR("referer failed\n");
   dFree(http_referer);

   if (!(URL_FLAGS(f->url) & URL_SpamSafe)) {
      cookies = a_Cookies_get_query(f->url, f->referrer);
      if (cookies && *cookies)
         if (curl_easy_setopt(c, CURLOPT_COOKIE, cookies))
            MSG_ERR("cookies failed\n");
      dFree(cookies);
   }

   if (a_Auth_get_credentials(f->url, &authtype, &http_auth_username, &http_auth_password)) {
      long curlauthtype = authtype == BASIC ? CURLAUTH_BASIC :
                          authtype == DIGEST ? CURLAUTH_DIGEST :
                          CURLAUTH_ANY;
      /* If you’re at the point of CURLAUTH_ANY, you’re unlikely to succeed. I don’t start with CURLAUTH_ANY because if libcurl tries one that the server doesn’t like, you get the 401 header for it before you get the 200 (or whatever) for the one that worked, and anyway that is wasted traffic, doing that with every fetch. And, in 2025, what does Apache have? Basic and Digest. */

      if (curl_easy_setopt(c, CURLOPT_USERNAME, http_auth_username))
         MSG_ERR("username failed\n");
      if (curl_easy_setopt(c, CURLOPT_PASSWORD, http_auth_password))
         MSG_ERR("password failed\n");

      if (curl_easy_setopt(c, CURLOPT_HTTPAUTH, curlauthtype))
         MSG_ERR("httpauth failed\n");
   }
}

void a_Curl_request(void *req_id_ptr)
{
   ulong_t req_id;
   const FetchReq *f;

   CURLMcode rc;
   CURL *c;

   dReturn_if_fail(multi);

   req_id = (ulong_t) req_id_ptr;
   f = a_FetchReq_get(req_id);

   dReturn_if_fail(f);

   if ((c = curl_easy_init())) {

      Curl_handle_invariant_initialization(c); /* MAYBE MAKE A PROTOTYPE HANDLE AND curl_easy_duphandle() IT */

      struct my_dumb_data *data = dNew(struct my_dumb_data, 1);
      data->req_id = req_id;
      data->headers = NULL;
      data->url = a_Url_dup(f->url);
      data->errbuf[0] = '\0';
      data->handle = c;
      data->body_bytes_received = 0;

      if (curl_easy_setopt(c, CURLOPT_PRIVATE, data))
         MSG_ERR("private failed\n");
      if (curl_easy_setopt(c, CURLOPT_URL, URL_STR(data->url)))
         MSG_ERR("url failed\n");
      if (curl_easy_setopt(c, CURLOPT_ERRORBUFFER, data->errbuf))
         MSG_ERR("errorbuffer failed\n");
      if (curl_easy_setopt(c, CURLOPT_WRITEDATA, data->req_id))
         MSG_ERR("writedata failed\n");
      if (curl_easy_setopt(c, CURLOPT_HEADERDATA, data->req_id))
         MSG_ERR("headerdata failed\n");
      if (curl_easy_setopt(c, CURLOPT_XFERINFODATA, data))
         MSG_ERR("headerdata failed\n");

      if (URL_FLAGS(f->url) & URL_Post)
         Curl_handle_POST_initialization(c, data, f->url);

      /* HTTP and HTTPS */

      if (!dStrAsciiCasecmp(URL_SCHEME(f->url), "https") ||
          !dStrAsciiCasecmp(URL_SCHEME(f->url), "http"))
         Curl_handle_http_https_initialization(c, data, f);

      if (data->headers) {
         if (curl_easy_setopt(c, CURLOPT_HTTPHEADER, data->headers))
               MSG_ERR("httpheader failed\n");
      }

      rc = curl_multi_add_handle(multi, c);


      if (rc) {
         MSG_ERR("oh, multi_add_handle failed. '%s'\n", curl_multi_strerror(rc));
         if (data->headers)
            curl_slist_free_all(data->headers);
         a_Url_free(data->url);
         curl_easy_cleanup(c);
      } else {
         dList_insert_sorted(conns, (void *)data, Curl_dumb_data_cmp);
      }
   }
}

void a_Curl_abort_reply(const DilloUrl *url){
   struct my_dumb_data *data = dList_find_sorted (conns, url, Curl_dumb_data_by_url_cmp);

   if (data) {
      MSG("Curl: We are aborting %s.\n", URL_STR(url));
      a_Capi_reply_aborted(data->req_id); // you can forget about the url argument beyond this point

      dList_remove(conns, data);
      if (data->headers)
         curl_slist_free_all(data->headers);
      a_Url_free(data->url);
      if (curl_multi_remove_handle(multi, data->handle) != CURLM_OK)
         MSG("…but the remove handle failed.\n");
      curl_easy_cleanup(data->handle);
      dFree(data);
   }
}

static int Curl_host_issuer_cmp_by_CA_then_host(const void *v1, const void *v2)
{
   const host_issuer_t *n1 = v1, *n2 = v2;

   int c = dStrAsciiCasecmp(n1->root_cert_authority, n2->root_cert_authority);
   if (!c)
      c = dStrAsciiCasecmp(n1->host, n2->host);
   return c;
}

void a_Curl_freeall(void)
{
   struct my_dumb_data *data;
   host_issuer_t *hi;
   int i, n;

   n = dList_length(conns);
   for (i = 0; i < n; i++) {
      MSG("remove a dumb data\n");
      data = dList_nth_data(conns, i);
      if (data->headers)
         curl_slist_free_all(data->headers);
      if (multi)
         curl_multi_remove_handle(multi, data->handle);
      curl_easy_cleanup(data->handle);

      a_Url_free(data->url);
      dFree(data);
   }
   dList_free(conns);

   dList_sort(host_issuers, Curl_host_issuer_cmp_by_CA_then_host);

   if ((n = dList_length(host_issuers)))
      MSG("TLS Certification Authorities trusted in this session:\n");
   for (i = 0; i < n; i++) {
      hi = dList_nth_data(host_issuers, i);
      MSG("%s: %s\n", hi->root_cert_authority, hi->host);
      dFree(hi->host);
      dFree(hi->root_cert_authority);
      dFree(hi);
   }
   dList_free(host_issuers);

   if (multi)
      curl_multi_cleanup(multi);
}
