/*
 * Copyright (C) 2006-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include "../msg.h"
#include "../url.h"
#include "../capi.h"

/*
 * Global variables
 */
static void b64strip_illegal_chars(unsigned char* str)
{
   unsigned char *p, *s = str;

   _MSG("len=%zu{%s}\n", strlen((char*)str), str);

   for (p = s; (*p = *s); ++s) {
      if (isascii(*p) && (isalnum(*p) || strchr("+/=", *p)))
         ++p;
   }

   _MSG("len=%zu{%s}\n", strlen((char *)str), str);
}

static int b64decode(unsigned char* str)
{
   unsigned char *cur, *start;
   int d, dlast, phase;
   unsigned char c;
   static int table[256] = {
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 00-0F */
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 10-1F */
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,62,-1,-1,-1,63,  /* 20-2F */
      52,53,54,55,56,57,58,59,60,61,-1,-1,-1,-1,-1,-1,  /* 30-3F */
      -1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,  /* 40-4F */
      15,16,17,18,19,20,21,22,23,24,25,-1,-1,-1,-1,-1,  /* 50-5F */
      -1,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,  /* 60-6F */
      41,42,43,44,45,46,47,48,49,50,51,-1,-1,-1,-1,-1,  /* 70-7F */
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 80-8F */
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* 90-9F */
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* A0-AF */
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* B0-BF */
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* C0-CF */
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* D0-DF */
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,  /* E0-EF */
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1   /* F0-FF */
   };

   d = dlast = phase = 0;
   start = str;
   for (cur = str; *cur != '\0'; ++cur ) {
      // jer: treat line endings as physical breaks.
      //if (*cur == '\n' || *cur == '\r'){phase = dlast = 0; continue;}
      d = table[(int)*cur];
      if (d != -1) {
         switch(phase) {
         case 0:
            ++phase;
            break;
         case 1:
            c = ((dlast << 2) | ((d & 0x30) >> 4));
            *str++ = c;
            ++phase;
            break;
         case 2:
            c = (((dlast & 0xf) << 4) | ((d & 0x3c) >> 2));
            *str++ = c;
            ++phase;
            break;
         case 3:
            c = (((dlast & 0x03 ) << 6) | d);
            *str++ = c;
            phase = 0;
            break;
         }
         dlast = d;
      }
   }
   *str = '\0';
   return str - start;
}

/*
 * Get mime type from the data URI.
 */
static char *datauri_get_mime(const DilloUrl *url)
{
   char buf[256];
   char *mime_type = NULL, *p;
   size_t len = 0;

   if (!dStrAsciiCasecmp(URL_SCHEME(url), "data")) {
      const char *path = URL_PATH(url);

      if ((p = strchr(path, ',')) && p - path < 256) {
         len = p - path;
         strncpy(buf, path, len);
         buf[len] = 0;
         /* strip ";base64" */
         if (len >= 7 && dStrAsciiCasecmp(buf + len - 7, ";base64") == 0) {
            len -= 7;
            buf[len] = 0;
         }
      }

      /* that's it, now handle omitted types */
      if (len == 0) {
         mime_type = dStrdup("text/plain;charset=US-ASCII");
      } else if (!dStrnAsciiCasecmp(buf, "charset", 7)) {
         mime_type = dStrconcat("text/plain;", buf, NULL);
      } else {
         mime_type = dStrdup(buf);
      }
   }

   return mime_type;
}

/*
 * Unescape %XX sequences in a string.
 * Return value: a new unescaped string
 */
static char *Unescape_uri_str(const char *s)
{
   char *p, *buf = dStrdup(s);

   if (strchr(s, '%')) {
      for (p = buf; (*p = *s); ++s, ++p) {
         if (*p == '%' && isxdigit(s[1]) && isxdigit(s[2])) {
            *p = (isdigit(s[1]) ? (s[1] - '0')
                                : D_ASCII_TOUPPER(s[1]) - 'A' + 10) * 16;
            *p += isdigit(s[2]) ? (s[2] - '0')
                                : D_ASCII_TOUPPER(s[2]) - 'A' + 10;
            s += 2;
         }
      }
   }

   return buf;
}

/*
 * Return a decoded data string.
 */
static unsigned char *datauri_get_data(const DilloUrl *url, size_t *p_sz)
{
   char *p;
   int is_base64 = 0;
   const char *path = URL_PATH(url);
   unsigned char *data = NULL;

   if ((p = strchr(path, ',')) && p - path >= 12 &&  /* "data:;base64" */
       dStrnAsciiCasecmp(p - 7, ";base64", 7) == 0) {
      is_base64 = 1;
   }

   if (p) {
      ++p;
      if (is_base64) {
         data = (unsigned char *)Unescape_uri_str(p);
         b64strip_illegal_chars(data);
         *p_sz = (size_t) b64decode(data);
      } else {
         data = (unsigned char *)a_Url_decode_hex_str(p, p_sz);
      }
   } else {
      data = (unsigned char *)dStrdup("");
      *p_sz = 0;
   }
   return data;
}

void a_Datauri_request(void *req_id_ptr)
{
   char *mime_type;
   unsigned char *data;
   size_t data_size = 0;

   ulong_t req_id = (ulong_t) req_id_ptr;
   const DilloUrl *u = a_FetchReq_url(req_id);

   if (u) {
      mime_type = datauri_get_mime(u);
      data = datauri_get_data(u, &data_size);

      if (mime_type && data) {
         char *h2 = dStrconcat("Content-Type: ", mime_type, "\r\n", NULL);
         a_Capi_reply_header(req_id, h2, strlen(h2));
         dFree(h2);
         Dstr *d = dStr_new("Content-Length: ");
         dStr_sprintfa(d, "%zu\r\n\r\n", data_size);
         a_Capi_reply_header(req_id, d->str, d->len);
         dStr_free(d, 1);

         a_Capi_reply_body(req_id, (char *)data, data_size);
         a_Capi_report_progress(req_id, data_size, data_size);
         a_Capi_reply_finished(req_id);
      }

      dFree(data);
      dFree(mime_type);
   }
}
