/*
 * Copyright (C) 2000-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */
#include <sys/stat.h>
#include <sys/time.h>
#include <time.h> /* gmtime */
#include <unistd.h>
#include <errno.h>           /* for errno */
#include <dirent.h>
#include <fcntl.h>

#include "d_size.h"
#include "../../dlib/dlib.h"
#include "../msg.h"
#include "../misc.h"
#include "../capi.h"
#include "filedir.h"

#define MAXNAMESIZE 40

typedef struct {
   char *full_path;
   const char *filename;
   off_t size;
   mode_t mode;
   time_t mtime;
} FileInfo;

typedef struct {
   char *dirname;
   bool_t opened;
   Dlist *flist;       /* List of files and subdirectories (for sorting) */
} DilloDir;

static const char *unsafe_chars = "&<>\"'";
static const char *unsafe_rep[] =
  { "&amp;", "&lt;", "&gt;", "&quot;", "&#39;" };

/*
 * Forward references
 */
#if 0
static const char *File_content_type(const char *filename);
#endif

static char *Escape_uri_str(const char *str, const char *p_esc_set)
{
   static const char *esc_set, *hex = "0123456789ABCDEF";
   char *p;
   Dstr *dstr;
   int i;

   esc_set = (p_esc_set) ? p_esc_set : "%#:' ";
   dstr = dStr_sized_new(64);
   for (i = 0; str[i]; ++i) {
      if (str[i] <= 0x1F || str[i] == 0x7F || strchr(esc_set, str[i])) {
         dStr_append_c(dstr, '%');
         dStr_append_c(dstr, hex[(str[i] >> 4) & 15]);
         dStr_append_c(dstr, hex[str[i] & 15]);
      } else {
         dStr_append_c(dstr, str[i]);
      }
   }
   p = dstr->str;
   dStr_free(dstr, FALSE);

   return p;
}

/*
 * Escape unsafe characters as html entities.
 * Return value: New escaped string.
 */
static char *Escape_html_str(const char *str)
{
   int i;
   char *p;
   Dstr *dstr = dStr_sized_new(64);

   for (i = 0; str[i]; ++i) {
      if ((p = strchr(unsafe_chars, str[i])))
         dStr_append(dstr, unsafe_rep[p - unsafe_chars]);
      else
         dStr_append_c(dstr, str[i]);
   }
   p = dstr->str;
   dStr_free(dstr, FALSE);

   return p;
}


/*
 * Compare two FileInfo pointers
 * This function is used for sorting directories
 */
static int File_comp(const FileInfo *f1, const FileInfo *f2)
{
   return strcmp(f1->filename, f2->filename);
}

/*
 * Allocate a DilloDir structure, set safe values in it and sort the entries.
 */
static DilloDir *File_dillodir_new(char *dirname)
{
   struct stat sb;
   struct dirent *de;
   DilloDir *Ddir;
   FileInfo *finfo;
   char *fname;
   int dirname_len;
   DIR *dir = opendir(dirname);

   Ddir = dNew(DilloDir, 1);
   Ddir->dirname = dStrdup(dirname);
   Ddir->opened = dir != NULL;

   if (dir) {
      Ddir->flist = dList_new(512);

      dirname_len = strlen(Ddir->dirname);

      /* Scan every name and sort them */
      while ((de = readdir(dir)) != 0) {
         if (!strcmp(de->d_name, ".") || !strcmp(de->d_name, ".."))
            continue;              /* skip "." and ".." */

         fname = dStrconcat(Ddir->dirname, de->d_name, NULL);
         if (lstat(fname, &sb) == -1) {
            dFree(fname);
            continue;              /* ignore files we can't stat */
         }

         finfo = dNew(FileInfo, 1);
         finfo->full_path = fname;
         finfo->filename = fname + dirname_len;
         finfo->size = sb.st_size;
         finfo->mode = sb.st_mode;
         finfo->mtime = sb.st_mtime;

         dList_append(Ddir->flist, finfo);
      }
      closedir(dir);
      dList_sort(Ddir->flist, (dCompareFunc)File_comp);
   } else {
      Ddir->flist = NULL;
   }
   return Ddir;
}

/*
 * Deallocate a DilloDir structure.
 */
static void File_dillodir_free(DilloDir *Ddir)
{
   int i;
   FileInfo *finfo;

   dReturn_if (Ddir == NULL);

   if (Ddir->flist) {
      for (i = 0; i < dList_length(Ddir->flist); ++i) {
         finfo = dList_nth_data(Ddir->flist, i);
         dFree(finfo->full_path);
         dFree(finfo);
      }
      dList_free(Ddir->flist);
   }
   dFree(Ddir->dirname);
   dFree(Ddir);
}

/*
 * Output the string for parent directory
 */
static void File_print_parent_dir(Dstr *page, const char *dirname)
{
   if (strcmp(dirname, "/") != 0) {        /* Not the root dir */
      char *p, *parent, *HUparent, *Uparent;

      parent = dStrdup(dirname);
      /* cut trailing slash */
      parent[strlen(parent) - 1] = '\0';
      /* make 'parent' have the parent dir path */
      if ((p = strrchr(parent, '/')))
         *(p + 1) = '\0';

      Uparent = Escape_uri_str(parent, NULL);
      HUparent = Escape_html_str(Uparent);
      dStr_sprintfa(page, "<a href='file://%s'>&nbsp;..&nbsp;</a>", HUparent);
      dFree(HUparent);
      dFree(Uparent);
      dFree(parent);
   }
}

/*
 * Given a timestamp, output an HTML-formatted date string.
 */
static void File_print_mtime(Dstr *page, time_t mtime)
{
   char buf[100];
   struct tm *tmdata = gmtime(&mtime);

   /* 2021: This used to be ctime(), but it was no longer working on Linux. Deprecated, and we’re supposed to use strftime() now. ctime() isn’t threadsafe, but I don’t know what was stomping on its buffer.  */
   if (!strftime(buf, sizeof(buf), "%c", tmdata)) {
      MSG_ERR("file strftime failed\n");
      return;
   }

   dStr_sprintfa(page, "<td align=right>%.3s&nbsp;", buf + 4); /* month */

   /* day */
   if (buf[8] == ' ')
      dStr_sprintfa(page, "&nbsp;%c", buf[9]);
   else
      dStr_sprintfa(page, "%.2s", buf + 8);

   /* hour or year */
   dStr_sprintfa(page, "&nbsp;%.5s",
      /* (more than 6 months old) ? year : hour; */
      (time(NULL) - mtime > 15811200) ? buf + 20 : buf + 11);
}

/*
 * Return a HTML-line from file info.
 */
static void File_info2html(Dstr *page, FileInfo *finfo, int n)
{
   double magnitude;
   char *suffix;
   char namebuf[MAXNAMESIZE + 1];
   char *Uref, *HUref, *Hname;
   const char *sizeformat, *ref, *filecont, *name = finfo->filename;

   if (S_ISDIR (finfo->mode)) {
      filecont = "dir";
   } else if (S_ISCHR( finfo->mode)) {
      filecont = "chardev";
   } else if (S_ISBLK( finfo->mode)) {
      filecont = "blockdev";
   } else if (S_ISFIFO( finfo->mode)) {
      filecont = "pipe";
   } else if (S_ISLNK( finfo->mode)) {
      filecont = "symlink";
   } else if (S_ISSOCK( finfo->mode)) {
      filecont = "socket";
   } else if (finfo->mode & (S_IXUSR | S_IXGRP | S_IXOTH)) {
      filecont = "executable";
   } else {
#if 1
      filecont = "file";
#else
      filecont = File_content_type(finfo->full_path);
      if (!filecont || !strcmp(filecont, "application/octet-stream"))
         filecont = "unknown";
#endif
   }

   ref = name;

   if (strlen(name) > MAXNAMESIZE) {
      memcpy(namebuf, name, MAXNAMESIZE - 3);
      strcpy(namebuf + (MAXNAMESIZE - 3), "...");
      name = namebuf;
   }

   /* escape problematic filenames */
   Uref = Escape_uri_str(ref, NULL);
   HUref = Escape_html_str(Uref);

   Hname = Escape_html_str(name);

   dStr_sprintfa(page, "<tr align=center%s><td align=left><a href='%s'",
      (n & 1) ? " bgcolor=#d0d0d0" : "",
      HUref);

   if (name == namebuf) {
      char *escaped_fullname = Escape_html_str(ref);
      dStr_sprintfa(page, " title='%s'", escaped_fullname);
      dFree(escaped_fullname);
   }

   dStr_sprintfa(page,
      ">%s</a>%c<td align=left>%s<td align=right>",
      Hname, S_ISDIR (finfo->mode) ? '/' : ' ',
      filecont);

   dFree(Hname);
   dFree(HUref);
   dFree(Uref);

   if (finfo->size >= 1024 * 1024 * 1024) {
      suffix = "G";
      magnitude = finfo->size / (1024 * 1024 * 1024.0);
   } else if (finfo->size >= 1024 * 1024) {
      suffix = "M";
      magnitude = finfo->size / (1024 * 1024.0);
   } else if (finfo->size >= 1024) {
      suffix = "K";
      magnitude = finfo->size / 1024.0;
   } else {
      suffix = "";
      magnitude = finfo->size;
   }
   // The 3 significant digits of %.3g is generally nice until you give it a value above 999 and then it uses scientific notation. Let’s avoid that.
   sizeformat = magnitude > 999 ? "%.0f%s" : "%.3g%s";
   dStr_sprintfa(page, sizeformat, magnitude, suffix);

   File_print_mtime(page, finfo->mtime);
   dStr_append_c(page, '\n');
}

/*
 * Send the HTML directory page in HTTP.
 */
static char *File_send_dir(DilloDir *Ddir)
{
   int n;
   char *str, *Hdirname, *Udirname, *HUdirname;

   Dstr *page = dStr_new("");

   /* Send page title */
   Udirname = Escape_uri_str(Ddir->dirname, NULL);
   HUdirname = Escape_html_str(Udirname);
   Hdirname = Escape_html_str(Ddir->dirname);

   dStr_sprintf(page,
      "<!DOCTYPE HTML PUBLIC '-//W3C//DTD HTML 4.01 Transitional//EN'>\n"
      "<HTML>\n<HEAD>\n <BASE href='file://%s'>\n"
      " <TITLE>file://%s</TITLE>\n</HEAD>\n"
      "<BODY>\n",
      HUdirname, Hdirname, Hdirname);
   dFree(Hdirname);
   dFree(HUdirname);
   dFree(Udirname);

   /* Output the parent directory */
   File_print_parent_dir(page, Ddir->dirname);

   if (!Ddir->opened) {
      dStr_append(page, "<br><br>Cannot open directory.");
   } else if (dList_length(Ddir->flist) == 0) {
      dStr_append(page, "<br><br>Directory is empty.");
   } else {
      dStr_append(page,
         "<br><br>\n"
         "<table border=0 cellpadding=1 cellspacing=0"
         " bgcolor=#E0E0E0 width=100%>\n"
         "<tr align=center>\n"
         "<td width=60%><b>Filename</b>"
         "<td><b>Type</b>"
         "<td><b>Size</b>"
         "<td><b>Modified</b>\n");
      for (n = 0; n < dList_length(Ddir->flist); ++n) {
         File_info2html(page, dList_nth_data(Ddir->flist,n), n+1);
      }
      dStr_append(page, "</table>\n");
   }

   dStr_append(page, "</BODY></HTML>\n");

   str = page->str;
   dStr_free(page, 0);
   return str;
}

#if 0
/*
 * Close a file descriptor, but handling EINTR
 */
static void File_close(int fd)
{
   while (fd >= 0 && close(fd) < 0 && errno == EINTR)
      ;
}

static const char *File_content_type(const char *filename)
{
   int fd;
   struct stat sb;
   const char *ct = "unknown";
   char buf[256];
   ssize_t buf_size;

   /* everything failed, let's analyze the data... */
   if ((fd = open(filename, O_RDONLY | O_NONBLOCK)) != -1) {
      if ((buf_size = read(fd, buf, 256)) == 256 ) {
         (void)a_Misc_get_content_type_from_data(buf, (size_t)buf_size, &ct);

      } else if (stat(filename, &sb) != -1) {
         if (!sb.st_size) {
            ct = "empty";
         } else if (buf_size == sb.st_size) {
            (void)a_Misc_get_content_type_from_data(buf, (size_t)buf_size, &ct);
         }
      }
      File_close(fd);
   } else {
      ct = "unreadable";
   }
   _MSG("File_content_type: name=%s ct=%s\n", filename, ct);
   return ct;
}
#endif

/*
 * Scan the directory, sort and prepare to send it enclosed in HTTP.
 */
static DilloDir *File_prepare_send_dir(const char *DirName)
{
   Dstr *ds_dirname;
   DilloDir *Ddir;

   /* Let's make sure this directory url has a trailing slash */
   ds_dirname = dStr_new(DirName);
   if (ds_dirname->str[ds_dirname->len - 1] != '/')
      dStr_append(ds_dirname, "/");

   /* Let's get a structure ready for transfer */
   Ddir = File_dillodir_new(ds_dirname->str);
   dStr_free(ds_dirname, TRUE);
   return Ddir;
}

void a_Filedir_request(void *req_id_ptr)
{
   size_t ignored;
   ulong_t req_id = (ulong_t) req_id_ptr;
   const DilloUrl *u = a_FetchReq_url(req_id); // I don’t dare expose FetchReq again just yet. I jumped in yesterday and was quickly in cyclic dependencies.

   if (u) {
      char *path = a_Url_decode_hex_str(URL_PATH(u), &ignored);
      char *body = NULL;
      DilloDir *ddir = File_prepare_send_dir(path);

      dFree(path);

      if (ddir) {
         size_t body_len;
         const char *h2 = "Content-Type: text/html\r\n\r\n";
         a_Capi_reply_header(req_id, h2, strlen(h2));
         body = File_send_dir(ddir);
         File_dillodir_free(ddir);
         body_len = strlen(body);
         a_Capi_reply_body(req_id, body, body_len);
         a_Capi_report_progress(req_id, body_len, body_len);
         dFree(body);
         a_Capi_reply_finished(req_id);
      }
   }
}
