/*
 * File: http.c
 *
 * Copyright (C) 2000-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

#include <config.h>
#include <assert.h>
#include <stdlib.h>

#include "http.h"
#include "../../dlib/dlib.h"
#include "../url.h"
#include "../prefs.h"

static char *HTTP_Proxy = NULL;
static char *HTTP_Proxy_Auth = NULL;
static char *HTTP_Language_hdr = NULL;


const char *a_Http_get_proxy_urlstr()
{
   return HTTP_Proxy;
}

const char *a_Http_get_proxy_authstr()
{
   return HTTP_Proxy_Auth;
}

const char *a_Http_get_language()
{
   return HTTP_Language_hdr;
}

/*
 * Activate entered proxy password for HTTP.
 *
 * CURLOPT_PROXYUSERPWD(3) “Both the name and the password will be URL decoded before use, so to include for example a colon in the user name you should encode it as %3A.”
 */
void a_Http_set_proxy_passwd(const char *str)
{
   char *u = a_Url_encode_hex_str(prefs.http_proxyuser),
        *p = a_Url_encode_hex_str(str);
   HTTP_Proxy_Auth = dStrconcat(u, ":", p, NULL);
}

const char *a_Http_sec_fetch_dest_str(HttpSecFetchDest dest)
{
   const char *ret;

   switch (dest) {
   case HTTP_Sec_Fetch_Dest_audio:
      ret = "audio";
      break;
   case HTTP_Sec_Fetch_Dest_document:
      ret = "document";
      break;
   case HTTP_Sec_Fetch_Dest_embed:
      ret = "embed";
      break;
   case HTTP_Sec_Fetch_Dest_frame:
      ret = "frame";
      break;
   case HTTP_Sec_Fetch_Dest_iframe:
      ret = "iframe";
      break;
   case HTTP_Sec_Fetch_Dest_image:
      ret = "image";
      break;
   case HTTP_Sec_Fetch_Dest_object:
      ret = "object";
      break;
   case HTTP_Sec_Fetch_Dest_style:
      ret = "style";
      break;
   case HTTP_Sec_Fetch_Dest_video:
      ret = "video";
      break;
   default:
      assert (0);
      ret = "crash!";
   }
   return ret;
}

const char *a_Http_sec_fetch_mode_str(HttpSecFetchDest dest, const DilloUrl *url, const DilloUrl *referrer)
{
   const char *ret;

   if (dest == HTTP_Sec_Fetch_Dest_document)
      ret = "navigate";
   else if (referrer && a_Url_same_origin(url, referrer))
      ret = "same-origin";
   else
      ret = "no-cors";

   return ret;
}

const char *a_Http_sec_fetch_site_str(const DilloUrl *url, const DilloUrl *referrer)
{
   const char *ret;

   if (!referrer)
      ret = "none";
   else if (a_Url_same_origin(url, referrer))
      ret = "same-origin";
   else if (!dStrAsciiCasecmp(URL_SCHEME(url), URL_SCHEME(referrer)) && a_Url_same_organization(url, referrer))
      ret = "same-site";
   else
      ret = "cross-site";

   return ret;
}

/*
 * Initialize proxy vars and Accept-Language header
 */
int a_Http_init(void)
{
   char *env_proxy = getenv("http_proxy");

   HTTP_Language_hdr = prefs.http_language ?
      dStrconcat("Accept-Language: ", prefs.http_language, NULL) :
      dStrdup("Accept-Language:");

   if (env_proxy && strlen(env_proxy))
      HTTP_Proxy = dStrdup(env_proxy);
   if (!HTTP_Proxy && prefs.http_proxy)
      HTTP_Proxy = dStrdup(prefs.http_proxy);

   return 0;
}

/*
 * Deallocate memory used by http module
 * (Call this one at exit time)
 */
void a_Http_freeall(void)
{
   dFree(HTTP_Proxy);
   dStrshred(HTTP_Proxy_Auth);
   dFree(HTTP_Proxy_Auth);
   dFree(HTTP_Language_hdr);
}
