/*
 * File: auth.c
 *
 * Copyright 2008 Jeremy Henty   <onepoint@starurchin.org>
 * Copyright 2009 Justus Winter  <4winter@informatik.uni-hamburg.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

/* Handling of HTTP AUTH takes place here.
 * This implementation aims to follow RFC 2617:
 * http://www.ietf.org/rfc/rfc2617.txt
 */

/* 2025: In old Dillo, this was more complicated. Now, we supply libcurl with a username and password, and tell it which auth method to use, and it takes it from there. */

#include <ctype.h> /* iscntrl */
#include "auth.h"
#include "msg.h"
#include "uicmd.hh"
#include "../dlib/dlib.h"

typedef struct {
   char *scheme;
   char *authority;
   Dlist *realms;
} AuthHost_t;

typedef struct {
   enum AuthParseHTTPAuthType_t type;
   char *name;
   Dlist *paths; /* stripped of any trailing '/', so the root path is "" */
   char *username;
   char *password;
} AuthRealm_t;

typedef struct {
   enum AuthParseHTTPAuthType_t type;
   char *realm_name;
   DilloUrl *url;
} AuthDialogData_t;

/*
 *  Local data
 */
static Dlist *auth_hosts;

/*
 * Initialize the auth module.
 */
void a_Auth_init(void)
{
   auth_hosts = dList_new(1);
}

static int Auth_path_is_inside(const char *path1, const char *path2, int len)
{
   /*
    * path2 is effectively truncated to length len.  Typically len will be
    * strlen(path2), or 1 less when we want to ignore a trailing '/'.
    */
   return
      strncmp(path1, path2, len) == 0 &&
      (path1[len] == '\0' || path1[len] == '/');
}

/*
 * Check valid chars.
 * Return: 0 if invalid, 1 otherwise.
 */
static int Auth_is_token_char(char c)
{
   const char *invalid = "\"()<>@,;:\\[]?=/{} \t";
   return (!isascii(c) || strchr(invalid, c) || iscntrl((uchar_t)c)) ? 0 : 1;
}

/*
 * Unquote the content of a (potentially) quoted string.
 * Return: newly allocated unquoted content.
 *
 * Arguments:
 * valuep: pointer to a pointer to the first char.
 *
 * Preconditions:
 * *valuep points to a correctly quoted and escaped string.
 *
 * Postconditions:
 * *valuep points to the first not processed char.
 *
 */
static Dstr *Auth_unquote_value(char **valuep)
{
   char c, quoted;
   char *value = *valuep;
   Dstr *result;

   while (*value == ' ' || *value == '\t')
      value++;

   if ((quoted = *value == '"'))
      value++;

   result = dStr_new(NULL);
   while ((c = *value) &&
          (( quoted && c != '"') ||
           (!quoted && Auth_is_token_char(c)))) {
      dStr_append_c(result, (c == '\\' && value[1]) ? *++value : c);
      value++;
   }

   if (quoted && *value == '\"')
      value++;
   *valuep = value;
   return result;
}

/*
 * Parse authentication challenge into token-value pairs
 *
 * With libcurl, our only real goal is to find the realm name and put it in the supplied argument.
 *
 * Return: 1 if the parse succeeds, 0 otherwise.
 */
static int Auth_parse_token_value(char **auth, char **realm_name)
{
   char keep_going, expect_quoted;
   char *token, *beyond_token;
   Dstr *value;
   size_t *token_size;

   while (**auth) {
      _MSG("Auth_parse_token_value: remaining: %s\n", *auth);

     /* parse a token */
      token = *auth;

      token_size = 0;
      while (Auth_is_token_char(**auth)) {
         (*auth)++;
         token_size++;
      }
      if (token_size == 0) {
         MSG_WARN("Auth_parse_token_value: missing auth token\n");
         return 0;
      }
      beyond_token = *auth;
      /* skip linear whitespace characters */
      while (**auth == ' ' || **auth == '\t')
         (*auth)++;

      /* parse the '=' */
      switch (*(*auth)++) {
      case '=':
         *beyond_token = '\0';
         break;
      case '\0':
      case ',':
         MSG_WARN("Auth_parse_token_value: missing auth token value\n");
         return 0;
         break;
      default:
         MSG_WARN("Auth_parse_token_value: garbage after auth token\n");
         return 0;
         break;
      }

      value = Auth_unquote_value(auth);
      expect_quoted = !(strcmp(token, "stale") == 0 ||
                        strcmp(token, "algorithm") == 0);

      if (((*auth)[-1] == '"') != expect_quoted)
         MSG_WARN("Auth_parse_token_value: "
                  "Values for key %s should%s be quoted.\n",
                  token, expect_quoted ? "" : " not");

      keep_going = TRUE;
      if (dStrAsciiCasecmp("realm", token) == 0) {
         *realm_name = strdup(value->str);
         keep_going = FALSE; /* end parsing */
      }

      dStr_free(value, 1);
      if (!keep_going)
         break;

      /* skip ' ' and ',' */
      while ((**auth == ' ') || (**auth == ','))
         (*auth)++;
   }
   return 1;
}

/* returns the realm name if found */
static char * Auth_parse_challenge_args(char **challenge)
{
   char *realm = NULL;

   /* parse comma-separated token-value pairs */
   while (1) {
      /* skip space and comma characters */
      while (**challenge == ' ' || **challenge == ',')
         (*challenge)++;
      /* end of string? */
      if (!**challenge)
         break;
      /* parse token-value pair */
      if (!Auth_parse_token_value(challenge, &realm))
         break;
   }

   if (!realm) {
      MSG_WARN("Auth_parse_challenge_args: missing auth realm\n");
   }
   return realm;
}

/* returns the realm name if found */
static char *Auth_parse_challenge(char *challenge)
{
   MSG("auth.c: Auth_parse_challenge: challenge = '%s'\n", challenge);
   challenge = strchr(challenge, ' ');
   if (challenge) {
      challenge++;
      return Auth_parse_challenge_args(&challenge);
   }
   return NULL;
}

/*
 * Return the host that contains a URL, or NULL if there is no such host.
 */
static AuthHost_t *Auth_host_by_url(const DilloUrl *url)
{
   AuthHost_t *host;
   int i;

   for (i = 0; (host = dList_nth_data(auth_hosts, i)); i++)
      if (((dStrAsciiCasecmp(URL_SCHEME(url), host->scheme) == 0) &&
           (dStrAsciiCasecmp(URL_AUTHORITY(url), host->authority) == 0)))
         return host;

   return NULL;
}

/*
 * Search all realms for the one with the given name.
 */
static AuthRealm_t *Auth_realm_by_name(const AuthHost_t *host,
                                           const char *name)
{
   AuthRealm_t *realm;
   int i;

   for (i = 0; (realm = dList_nth_data(host->realms, i)); i++)
      if (strcmp(realm->name, name) == 0)
         return realm;

   return NULL;
}

/*
 * Search all realms for the one with the best-matching path.
 */
static AuthRealm_t *Auth_realm_by_path(const AuthHost_t *host,
                                       const char *path)
{
   AuthRealm_t *realm_best, *realm;
   int i, j;
   int match_length = 0;

   realm_best = NULL;
   for (i = 0; (realm = dList_nth_data(host->realms, i)); i++) {
      char *realm_path;

      for (j = 0; (realm_path = dList_nth_data(realm->paths, j)); j++) {
         int realm_path_length = strlen(realm_path);
         if (Auth_path_is_inside(path, realm_path, realm_path_length) &&
             !(realm_best && match_length >= realm_path_length)) {
            realm_best = realm;
            match_length = realm_path_length;
         }
      }
   }

   return realm_best;
}

static int Auth_realm_includes_path(const AuthRealm_t *realm, const char *path)
{
   int i;
   char *realm_path;

   for (i = 0; (realm_path = dList_nth_data(realm->paths, i)); i++)
      if (Auth_path_is_inside(path, realm_path, strlen(realm_path)))
         return 1;

   return 0;
}

static void Auth_realm_add_path(AuthRealm_t *realm, const char *path)
{
   int len, i;
   char *realm_path, *n_path;

   n_path = dStrdup(path);
   len = strlen(n_path);

   /* remove trailing '/' */
   if (len && n_path[len - 1] == '/')
      n_path[--len] = 0;

   /* delete existing paths that are inside the new one */
   for (i = 0; (realm_path = dList_nth_data(realm->paths, i)); i++) {
      if (Auth_path_is_inside(realm_path, path, len)) {
         dList_remove_fast(realm->paths, realm_path);
         dFree(realm_path);
         i--; /* reconsider this slot */
      }
   }

   dList_append(realm->paths, n_path);
}

/*
 * Get the user:pass string and the authentication type to feed to libcurl.
 */
bool_t a_Auth_get_credentials(const DilloUrl *url, enum AuthParseHTTPAuthType_t *type, const char **username, const char **password)
{
   bool_t ret = FALSE;
   AuthHost_t *host;
   AuthRealm_t *realm;

   if ((host = Auth_host_by_url(url)) &&
       (realm = Auth_realm_by_path(host, URL_PATH(url)))) {
      ret = TRUE;
      *username = realm->username;
      *password = realm->password;
      *type = realm->type;
   }
   return ret;
}

/*
 * Determine whether the user needs to authenticate.
 */
static int Auth_do_auth_required(const char *realm_name, const DilloUrl *url)
{
   /*
    * TO DO: I dislike the way that this code must decide whether we
    * sent authentication during the request and trust us to resend it
    * after the reload.  Could it be more robust if every DilloUrl
    * recorded its authentication, and whether it was accepted?  (JCH)
    */

   AuthHost_t *host;
   AuthRealm_t *realm;

   /*
    * The size of the following comments reflects the concerns in the
    * TO DO at the top of this function.  It should not be so hard to
    * explain why code is correct! (JCH)
    */

   /*
    * If we have authentication but did not send it (because we did
    * not know this path was in the realm) then we update the realm.
    * We do not re-authenticate because our authentication is probably
    * OK.  Thanks to the updated realm the forthcoming reload will
    * make us send the authentication.  If our authentication is not
    * OK the server will challenge us again after the reload and then
    * we will re-authenticate.
    */
   if ((host = Auth_host_by_url(url)) &&
       (realm = Auth_realm_by_name(host, realm_name))) {
      if (!Auth_realm_includes_path(realm, URL_PATH(url))) {
         _MSG("Auth_do_auth_required: updating realm '%s' with URL '%s'\n",
              realm_name, URL_STR(url));
         Auth_realm_add_path(realm, URL_PATH(url));
         return 0;
      }
   }

   /*
    * Either we had no authentication or we sent it and the server
    * rejected it, so we must re-authenticate.
    */
   return 1;
}

static void Auth_do_auth_dialog_cb(bool_t ok, const char *user, const char *password,
                                   void *vData)
{
   AuthDialogData_t *data = (AuthDialogData_t *)vData;

   if (ok) {
      AuthHost_t *host;
      AuthRealm_t *realm;

      /* find or create the host */
      if (!(host = Auth_host_by_url(data->url))) {
         /* create a new host */
         host = dNew(AuthHost_t, 1);
         host->scheme = dStrdup(URL_SCHEME(data->url));
         host->authority = dStrdup(URL_AUTHORITY(data->url));
         host->realms = dList_new(1);
         dList_append(auth_hosts, host);
      }

      /* find or create the realm */
      if (!(realm = Auth_realm_by_name(host, data->realm_name))) {
         realm = dNew0(AuthRealm_t, 1);
         realm->name = dStrdup(data->realm_name);
         realm->type = data->type;
         realm->paths = dList_new(1);
         dList_append(host->realms, realm);
      }
      dStrshred(realm->username);
      dFree(realm->username);
      dStrshred(realm->password);
      dFree(realm->password);

      Auth_realm_add_path(realm, URL_PATH(data->url));

      realm->username = dStrdup(user);
      realm->password = dStrdup(password);
   }

   dFree(data->realm_name);

   a_UIcmd_authenticated(data->url);
   a_Url_free((void *)data->url);
   dFree(data);
}

static void Auth_do_auth_dialog(void *vbw,
                               enum AuthParseHTTPAuthType_t type,
                               const char *realm_name,
                               const DilloUrl *url)
{
   char *msg;
   AuthDialogData_t *data;
   _MSG("auth.c: Auth_do_auth_dialog: realm = '%s'\n", auth_parse->realm_name);

   msg = dStrconcat("The server at ", URL_HOST(url), " requires a username"
                    " and password for  \"", realm_name, "\".", NULL);

   data = dNew(AuthDialogData_t, 1);
   data->type = type;
   data->realm_name = dStrdup(realm_name);
   data->url = a_Url_dup(url);
   a_UIcmd_username_password(vbw, msg, Auth_do_auth_dialog_cb, data);
   dFree(msg);
}

/*
 * Do authorization for an auth string.
 */
static int Auth_do_auth(void *vbw, char *challenge, enum AuthParseHTTPAuthType_t type,
                        const DilloUrl *url)
{
   int reload = 0;

   _MSG("auth.c: Auth_do_auth: challenge={%s}\n", challenge);
   char *realm_name = Auth_parse_challenge(challenge);
   if (realm_name) {
      if (Auth_do_auth_required(realm_name, url)) {
         Auth_do_auth_dialog(vbw, type, realm_name, url);
      } else {
         reload = 1;
      }
   }
   dFree(realm_name);

   return reload;
}

/*
 * Given authentication challenge(s), prepare authorization.
 * Return: 0 on failure
 *         nonzero on success. A new query will be sent to the server.
 */
int a_Auth_do_auth(void *vbw, Dlist *challenges, const DilloUrl *url)
{
   int i;
   char *chal;

   for (i = 0; (chal = dList_nth_data(challenges, i)); ++i)
      if (!dStrnAsciiCasecmp(chal, "Digest ", 7))
         if (Auth_do_auth(vbw, chal, DIGEST, url))
            return 1;
   for (i = 0; (chal = dList_nth_data(challenges, i)); ++i)
      if (!dStrnAsciiCasecmp(chal, "Basic ", 6))
         if (Auth_do_auth(vbw, chal, BASIC, url))
            return 1;

   return 0;
}

