/*
 * File: bookmark.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

#include <errno.h>
#include <fcntl.h>  /* open */
#include <sys/stat.h> /* S_* */
#include <unistd.h> /* write */

#include "msg.h"
#include "bw.h"
#include "history.h"

static const char *unsafe_chars = "&<>\"'";
static const char *unsafe_rep[] =
  { "&amp;", "&lt;", "&gt;", "&quot;", "&#39;" };

/*
 * Escape unsafe characters as html entities.
 * Return value: New escaped string.
 */
static char *Escape_html_str(const char *str)
{
   int i;
   char *p;
   Dstr *dstr = dStr_sized_new(64);

   for (i = 0; str[i]; ++i) {
      if ((p = strchr(unsafe_chars, str[i])))
         dStr_append(dstr, unsafe_rep[p - unsafe_chars]);
      else
         dStr_append_c(dstr, str[i]);
   }
   p = dstr->str;
   dStr_free(dstr, FALSE);

   return p;
}

void a_Bookmark_add(BrowserWindow *bw, const DilloUrl *url)
{
   char *title = Escape_html_str(a_History_get_title_by_url(url, 1));
   char *line = dStrconcat("<a href=\"", URL_STR(url), "\">", title, "</a><br>\n", NULL);
   size_t len = strlen(line);
   char *filename = dStrconcat(dGethomedir(), "/.dillo/bm.html", NULL);

   int fd = open(filename, O_WRONLY | O_CREAT | O_APPEND, S_IRUSR | S_IWUSR);

   if (fd == -1) {
      MSG_WARN("Could not open %s: %s.\n", filename, strerror(errno));
   } else {
      ssize_t just_written = 0;
      size_t total_written = 0;

      while (total_written < len) {
         just_written = write(fd, line + total_written, len - total_written);
         if (just_written == -1) {
            MSG_WARN("Write failed with %s.\n", strerror(errno));
            break;
         } else {
            total_written += (size_t)just_written;
         }
      }
      MSG("Wrote %zu bytes to %s.\n", total_written, filename);
      close(fd);
   }
   dFree(title);
   dFree(line);
   dFree(filename);
}
