/*
 * File: bw.c
 *
 * Copyright (C) 2006-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

/* Data structures for each browser window */


#include "bw.h"
#include "msg.h"
#include "list.h"
#include "capi.h"
#include "uicmd.hh"

#include "html.hh"

/*
 * Local Data
 */
/* A list of working browser windows */
static BrowserWindow **bws;
static int num_bws, num_bws_max;

/*
 * Initialize global data
 */
void a_Bw_init(void)
{
   num_bws = 0;
   num_bws_max = 16;
   bws = NULL;
}

/*
 * Create a new browser window and return it.
 * (the new window is stored in browser_window[])
 */
BrowserWindow *a_Bw_new()
{
   BrowserWindow *bw;

   /* We use dNew0() to zero the memory */
   bw = dNew0(BrowserWindow, 1);
   a_List_add(bws, num_bws, num_bws_max);
   bws[num_bws++] = bw;

   /* Initialize nav_stack */
   bw->nav_stack = dList_new(8);
   bw->nav_stack_idx = -1;

   /* Init expect */
   bw->nav_expect_url = NULL;

   bw->RootClients = dList_new(8);
   bw->ImageClients = dList_new(8);
   bw->NumPendingStyleSheets = 0;
   bw->Docs = dList_new(8);

   bw->resources = NULL;
   bw->num_resources = 0;
   bw->num_resources_max = 16;
   bw->total_document_bytes_received = 0;
   bw->total_document_bytes_still_expected = 0;

   bw->num_page_bugs = 0;
   bw->page_bugs = dStr_new("");

   bw->num_cookies = 0;
#ifndef DISABLE_COOKIES
   bw->cookie_strings = dList_new(8);
#endif

   return bw;
}

/*
 * Free resources associated to a bw.
 */
void a_Bw_free(BrowserWindow *bw)
{
   int i, j;

   for (i = 0; i < num_bws; i++) {
      if (bws[i] == bw) {
         a_List_remove(bws, i, num_bws);

         dList_free(bw->RootClients);
         dList_free(bw->ImageClients);
         dList_free(bw->Docs);

         a_Url_free(bw->nav_expect_url);

         for (j = 0; j < bw->num_resources; j++)
            a_Url_free(bw->resources[j].url);
         dFree(bw->resources);

         for (j = 0; j < dList_length(bw->nav_stack); ++j)
            dFree(dList_nth_data(bw->nav_stack, j));
         dList_free(bw->nav_stack);

         dStr_free(bw->page_bugs, 1);

#ifndef DISABLE_COOKIES
         for (j = 0; j < dList_length(bw->cookie_strings); j++)
            dFree(dList_nth_data(bw->cookie_strings, j));
         dList_free(bw->cookie_strings);
#endif
         dFree(bw);
         break;
      }
   }
}

/*- Clients ----------------------------------------------------------------*/
/*
 * Add a reference to a cache-client. It is kept int this bw's list.
 * This helps us keep track of which are active in the window so that it's
 * possible to abort/stop them.
 * (Root: Flag, whether a Root URL or not)
 */
void a_Bw_add_client(BrowserWindow *bw, int Key, int Root)
{
   dReturn_if_fail ( bw != NULL );

   if (Root) {
      dList_append(bw->RootClients, INT2VOIDP(Key));
   } else {
      dList_append(bw->ImageClients, INT2VOIDP(Key));
   }
   if (dList_length(bw->RootClients) + dList_length(bw->ImageClients) == 1)
      a_UIcmd_set_buttons_sens(bw);
}

/*
 * Close a cache-client upon successful retrieval.
 * Remove the cache-client from the bw list and update the meters.
 * (client can be a image or a html page)
 */
void a_Bw_close_client(BrowserWindow *bw, int ClientKey)
{
   void *data;

   if ((data = dList_find(bw->RootClients, INT2VOIDP(ClientKey)))) {
      dList_remove_fast(bw->RootClients, data);
      if (dList_length(bw->RootClients) == 0) {
         // Can a new root client come before the old one has ended? Perhaps. If not now, maybe in future code.
         if (bw->total_document_bytes_received == 0) {
            // Help to show that something happened with us getting this empty page.
            a_UIcmd_set_page_prog(bw, bw->total_document_bytes_received, 1.0f, bw->num_page_bugs, bw->num_cookies);
         }
      }
   } else if ((data = dList_find(bw->ImageClients, INT2VOIDP(ClientKey)))) {
      dList_remove_fast(bw->ImageClients, data);
   }
   if (!dList_length(bw->RootClients) && !dList_length(bw->ImageClients))
      a_UIcmd_set_buttons_sens(bw);
}

/*
 * Stop the active clients of this bw's top page.
 * Note: rendering stops, but the cache continues to be fed.
 */
void a_Bw_stop_clients(BrowserWindow *bw, int flags)
{
   void *data;

   if (flags & BW_Root) {
      /* Remove root clients */
      while ((data = dList_nth_data(bw->RootClients, 0))) {
         a_Capi_stop_client(VOIDP2INT(data), (flags & BW_Force));
         dList_remove_fast(bw->RootClients, data);
      }
   }

   if (flags & BW_Img) {
      /* Remove image clients */
      while ((data = dList_nth_data(bw->ImageClients, 0))) {
         a_Capi_stop_client(VOIDP2INT(data), (flags & BW_Force));
         dList_remove_fast(bw->ImageClients, data);
      }
   }
}

/*- Page -------------------------------------------------------------------*/

bool_t a_Bw_has_url(BrowserWindow *bw, const DilloUrl *url)
{
   int i;

   if (bw && url) {
      for (i = 0; i < bw->num_resources; i++) {
         if (!a_Url_cmp(url, bw->resources[i].url))
            return TRUE;
      }
   }
   return FALSE;
}

/*
 * Add an URL to the browser window's list.
 * This helps us keep track of page-requested URLs so that it's
 * possible to stop, abort and reload them.
 */
void a_Bw_add_url(BrowserWindow *bw, const DilloUrl *Url)
{
   dReturn_if_fail ( bw != NULL && Url != NULL );

   if (!a_Bw_has_url(bw, Url)) {
      a_List_add(bw->resources, bw->num_resources, bw->num_resources_max);
      bw->resources[bw->num_resources].url = a_Url_dup(Url);
      bw->resources[bw->num_resources].bytes_received =
         bw->resources[bw->num_resources].bytes_still_expected = 0;
      bw->num_resources++;
   }
}

/*
 * Add a document to the browser window's list.
 */
void a_Bw_add_doc(BrowserWindow *bw, void *vdoc)
{
   dReturn_if_fail ( bw != NULL && vdoc != NULL);

   dList_append(bw->Docs, vdoc);
}

/*
 * Get current document.
 */
void *a_Bw_get_current_doc(BrowserWindow *bw)
{
   void *doc = NULL;
   int len = dList_length(bw->Docs);

   if (len == 1)
      doc = dList_nth_data(bw->Docs, 0);
   else if (len > 1)
      MSG("a_Bw_get_current_doc() multiple docs not implemented\n");

   return doc;
}

/*
 * Get document by URL.
 *
 * This is currently used by popup menus that need to ensure that the
 * page has not changed while the menu was popped up.
 */
void *a_Bw_get_url_doc(BrowserWindow *bw, const DilloUrl *url)
{
   void *doc = NULL;

   if (url) {
      int i;

      for (i = 0; i < bw->num_resources; i++) {
         if (!a_Url_cmp(url, bw->resources[i].url)) {
            doc = a_Bw_get_current_doc(bw);
            break;
         }
      }
   }
   return doc;
}

/*
 * Remove a document from the bw's list
 */
void a_Bw_remove_doc(BrowserWindow *bw, void *vdoc)
{
   void *data;

   if ((data = dList_find(bw->Docs, vdoc))) {
      dList_remove_fast(bw->Docs, data);
   }
}

void a_Bw_load_images(BrowserWindow *bw, void *pattern, bool_t user_saw_url, bool_t background)
{
   void *doc = a_Bw_get_current_doc(bw);

   if (doc) {
      a_Html_load_images(doc, pattern, user_saw_url, background);
   }
}

/*- Cleanup ----------------------------------------------------------------*/
/*
 * Empty RootClients, ImageClients and resources lists and
 * reset progress bar data.
 */
void a_Bw_cleanup(BrowserWindow *bw)
{
   void *data;
   int i;

   /* Remove root clients */
   while ((data = dList_nth_data(bw->RootClients, 0))) {
      dList_remove_fast(bw->RootClients, data);
   }
   /* Remove image clients */
   while ((data = dList_nth_data(bw->ImageClients, 0))) {
      dList_remove_fast(bw->ImageClients, data);
   }
   /* Remove resources */
   for (i = 0; i < bw->num_resources; i++) {
      a_Url_free(bw->resources[i].url);
   }
   bw->num_resources = 0;
   bw->total_document_bytes_received = 0;
   bw->total_document_bytes_still_expected = 0;

   bw->num_cookies = 0;

   /* Zero stylesheet counter */
   bw->NumPendingStyleSheets = 0;
}

/*--------------------------------------------------------------------------*/

int a_Bw_num()
{
   return num_bws;
}

/*
 * Return a bw by index
 */
BrowserWindow *a_Bw_get(int i)
{
   if (i >= 0 && i < num_bws)
      return bws[i];
   return NULL;
}

/* expect API ------------------------------------------------------------- */

void a_Bw_expect(BrowserWindow *bw, const DilloUrl *url)
{
   a_Url_free(bw->nav_expect_url);
   bw->nav_expect_url = a_Url_dup(url);
}

void a_Bw_cancel_expect(BrowserWindow *bw)
{
   a_Url_free(bw->nav_expect_url);
   bw->nav_expect_url = NULL;
}

bool_t a_Bw_expecting(BrowserWindow *bw)
{
   return (bw->nav_expect_url != NULL);
}

const DilloUrl *a_Bw_expected_url(BrowserWindow *bw)
{
   return bw->nav_expect_url;
}

/*
 * As it is, this will be called per client rather than per url because the cache loops to inform clients of new data. If the current page has 20 of the same image, we will be called 20 times when we have received some more of that image.
 *
 * For bytes_still_expected, if there is no Content-Length header, we have no way to know at all, and if the reply body in compressed, we can only estimate the uncompressed amount remaining.
 */
void a_Bw_data_received(BrowserWindow *bw, const DilloUrl *Url, size_t bytes_received, size_t bytes_still_expected) {
   int i;

   for (i = 0; i < bw->num_resources; i++) {
      if (!a_Url_cmp(Url, bw->resources[i].url)) {
         if (bytes_received != bw->resources[i].bytes_received) {
            /* Update totals for the document. Do not double-count data: children. */
            if (!i || dStrAsciiCasecmp(URL_SCHEME(Url), "data")) {
               size_t denominator;
               double total_document_proportion;

               bw->total_document_bytes_received +=
                  bytes_received - bw->resources[i].bytes_received;
               bw->total_document_bytes_still_expected +=
                  bytes_still_expected - bw->resources[i].bytes_still_expected;
               denominator = bw->total_document_bytes_received + bw->total_document_bytes_still_expected;
               total_document_proportion = denominator ? ((double)bw->total_document_bytes_received) / denominator : 1.0f;
               a_UIcmd_set_page_prog(bw, bw->total_document_bytes_received, total_document_proportion, bw->num_page_bugs, bw->num_cookies);

            bw->resources[i].bytes_received = bytes_received;
            bw->resources[i].bytes_still_expected = bytes_still_expected;
            }
            _MSG("data_received, we have %zu received and expect %zu more.\n", bw->total_document_bytes_received, bw->total_document_bytes_still_expected);
         }
         break;
      }
   }
}

#ifndef DISABLE_COOKIES

static int list_string_cmp(const void *a, const void *b)
{
   return strcmp((const char *)a, (const char *)b);
}

/*
 * This is not sophisticated. If a page from a.example.com and an image on that page from b.example.com set cookies with Dillo=a_browser for their respective hosts, we would only see it as one cookie. And if they both set cookies for example.com where the first had time=1 and the second had time=2, you might argue those are one cookie. But that degree of detail simply does not matter here.
 */
void a_Bw_cookie_accepted(BrowserWindow *bw, const char *cookie) {
   if (!dList_find_sorted(bw->cookie_strings, cookie, list_string_cmp)) {
      dList_insert_sorted(bw->cookie_strings, dStrdup(cookie), list_string_cmp);
      bw->num_cookies++;
   }
}
#endif

void a_Bw_clear_bugs(BrowserWindow *bw)
{
   bw->num_page_bugs = 0;
   dStr_truncate(bw->page_bugs, 0);
}
