/*
 * File: cache.c
 *
 * Copyright 2000-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

#include <sys/types.h>

#include <limits.h>
#include <stdlib.h>
#include <string.h>

#include "msg.h"
#include "cache.h"
#include "dicache.h"
#include "nav.h"
#include "cookies.h"
#include "hsts.h"
#include "misc.h"
#include "capi.h"
#include "auth.h"
#include "domain.h"
#include "timeout.hh"
#include "uicmd.hh"
#include "client.hh"

/*
 *  Local data types
 */

typedef struct {
   DilloUrl *Url;      /* Cached Url. Url is used as a primary Key */
   FetchReq *req;         /* data needed for request */

   /* response-related */
   char *TypeDet;            /* MIME type string (detected from data) */
   char *TypeHdr;            /* MIME type string as from the HTTP Headers */
   Dstr *Headers;            /* HTTP headers */
   const DilloUrl *Location; /* New URI for redirects */
   Dlist *Auth;              /* Authentication fields */
   Dstr *Body;               /* Pointer to raw data */
   size_t bytes_received;    /* How much of the reply body received. */
   size_t ContentLength;
   uint_t Flags;             /* See Flag Defines in cache.h */
} CacheEntry_t;


/*
 *  Local data
 */
/* A sorted list for cached data. Holds pointers to CacheEntry_t structs */
static Dlist *Cache_entries;

/* A list for cache clients. */
static Dlist *Clients;

/* A list for already-cached data (which has to be sent behind a timeout so the
 * client can be ready). */
static Dlist *ReawakenedCacheEntries;
static bool_t ReawakenedCacheEntriesTimeoutSet = FALSE;

/*
 *  Forward declarations
 */
static int Cache_main_feed_clients(CacheEntry_t *entry);
static void Cache_feed_clients_from_cache(CacheEntry_t *entry);
static void Cache_auth_entry(CacheEntry_t *entry, BrowserWindow *bw);

/*
 * Determine if two cache entries are equal
 */
static int Cache_entry_cmp(const void *v1, const void *v2)
{
   const CacheEntry_t *d1 = v1, *d2 = v2;

   return a_Url_cmp(d1->Url, d2->Url);
}

/*
 * Determine if two cache entries are equal, using a URL as key.
 */
static int Cache_entry_by_url_cmp(const void *v1, const void *v2)
{
   const DilloUrl *u1 = ((CacheEntry_t*)v1)->Url;
   const DilloUrl *u2 = v2;

   return a_Url_cmp(u1, u2);
}

/*
 * Initialize cache data
 */
void a_Cache_init(void)
{
   Clients = dList_new(32);
   ReawakenedCacheEntries = dList_new(32);
   Cache_entries = dList_new(256);
}

/* Client operations ------------------------------------------------------ */

/*
 * Add a client to Clients.
 *  - Every client-field is just a reference.
 *  - Return a unique number for identifying the client.
 */
static int Cache_client_add(const DilloUrl *Url, ClientData *client,
                            CA_Callback_t Callback, void *CbData)
{
   static int ClientKey = 0; /* Provide a primary key for each client */

   if (ClientKey < INT_MAX) /* check for integer overflow */
      ClientKey++;
   else
      ClientKey = 1;

   client->Key = ClientKey;
   client->RedirectLevel = 0;
   client->Callback = Callback;
   client->CbData = CbData;

   dList_append(Clients, client);

   return ClientKey;
}

/*
 * Compare function for searching a Client by its key
 */
static int Cache_client_by_key_cmp(const void *client, const void *key)
{
   return ((ClientData *)client)->Key - VOIDP2INT(key);
}

/*
 * Remove a client from the list
 */
static void Cache_client_remove(ClientData *Client)
{
   if (Client) {
      a_Image_unref(Client->Image);
      a_Bw_close_client(Client->bw, Client->Key);
      dList_remove(Clients, Client);
      dFree(Client);
   }
}


/* Entry operations ------------------------------------------------------- */

/*
 * Set safe values for a new cache entry
 */
static void Cache_entry_init(CacheEntry_t *NewEntry, FetchReq *req)
{
   NewEntry->Url = a_Url_dup(req->url);    // TODO not have this duplicated for no reason (but not worth the changes as we’re experimenting right now)
   NewEntry->req = req;
   NewEntry->TypeDet = NULL;
   NewEntry->TypeHdr = NULL;
   NewEntry->Headers = dStr_new("");
   NewEntry->Location = NULL;
   NewEntry->Auth = NULL;
   NewEntry->Body = dStr_sized_new(8*1024);
   NewEntry->bytes_received = 0;
   NewEntry->ContentLength = 0;
   NewEntry->Flags = CA_InProgress;
}

/*
 * Get the data structure for a cached URL (using 'Url' as the search key)
 * If 'Url' isn't cached, return NULL
 */
static CacheEntry_t *Cache_entry_search(const DilloUrl *Url)
{
   return dList_find_sorted(Cache_entries, Url, Cache_entry_by_url_cmp);
}

/*
 * Given a URL, find its cache entry, following redirections.
 */
static CacheEntry_t *Cache_entry_search_with_redirect(const DilloUrl *Url)
{
   int i;
   CacheEntry_t *entry;

   for (i = 0; (entry = Cache_entry_search(Url)); ++i) {

      /* Test for a redirection loop */
      if (entry->Flags & CA_RedirectLoop || i == 3) {
         _MSG_WARN("Redirect loop for URL: >%s<\n", URL_STR_(Url));
         break;
      }
      /* Test for a working redirection */
      if (entry->Flags & CA_Redirect && entry->Location) {
         Url = entry->Location;
      } else
         break;
   }
   return entry;
}

/*
 * Allocate and set a new entry in the cache list
 */
static CacheEntry_t *Cache_entry_add(FetchReq *req)
{
   CacheEntry_t *old_entry, *new_entry;

   if ((old_entry = Cache_entry_search(req->url))) {
      MSG_WARN("Cache_entry_add, leaking an entry.\n");
      dList_remove(Cache_entries, old_entry);
   }

   new_entry = dNew(CacheEntry_t, 1);
   Cache_entry_init(new_entry, req);  /* Set safe values */
   dList_insert_sorted(Cache_entries, new_entry, Cache_entry_cmp);
   return new_entry;
}

/*
 *  Free Authentication fields.
 */
static void Cache_auth_free(Dlist *auth)
{
   int i;
   void *auth_field;
   for (i = 0; (auth_field = dList_nth_data(auth, i)); ++i)
      dFree(auth_field);
   dList_free(auth);
}

/*
 *  Free the components of a CacheEntry_t struct.
 */
static void Cache_entry_free(CacheEntry_t *entry)
{
   a_Url_free((DilloUrl *)entry->Url);
   a_FetchReq_free(entry->req);
   dFree(entry->TypeDet);
   dFree(entry->TypeHdr);
   dStr_free(entry->Headers, TRUE);
   a_Url_free((DilloUrl *)entry->Location);
   Cache_auth_free(entry->Auth);
   dStr_free(entry->Body, 1);
   dFree(entry);
}

/*
 * Remove an entry from the cache.
 * All the entry clients are removed too! (it may stop rendering of this
 * same resource on other windows, but nothing more).
 */
static void Cache_entry_remove(CacheEntry_t *entry, const DilloUrl *url)
{
   int i;
   ClientData *Client;

   if (!entry && !(entry = Cache_entry_search(url)))
      return;

   /* remove all clients for this entry */
   for (i = 0; (Client = dList_nth_data(Clients, i)); ++i) {
      if (Client->url == entry->Url) {
         a_Cache_stop_client(Client->Key);
         --i;
      }
   }

   /* remove from ReawakenedCacheEntries */
   dList_remove(ReawakenedCacheEntries, entry);

   /* remove from dicache */
   a_Dicache_invalidate_entry(entry->Url);

   /* remove from cache */
   dList_remove(Cache_entries, entry);
   Cache_entry_free(entry);
}

/* Misc. operations ------------------------------------------------------- */

void a_Cache_new_fetch(FetchReq *req)
{
   Cache_entry_add(req);
}

FetchReq *a_Cache_get_req(const DilloUrl *url)
{
   CacheEntry_t *entry = Cache_entry_search(url);

   return entry->req;
}

/*
 * Currently duplicating much of Cache_entry_remove(). But the important point is that we want to keep our old FetchReq so the reload will use the same information.
 */
static void Cache_entry_reinit_for_reload(CacheEntry_t *entry)
{
   int i;
   ClientData *client;

   /* remove all clients for this entry */
   for (i = 0; (client = dList_nth_data(Clients, i)); ++i) {
      if (client->url == entry->Url) {
         a_Cache_stop_client(client->Key);
         --i;
      }
   }

   /* remove from ReawakenedCacheEntries */
   dList_remove(ReawakenedCacheEntries, entry);

   /* remove from dicache */
   a_Dicache_invalidate_entry(entry->Url);

   /* hollow out the entry to refill */
   entry->TypeHdr = NULL;
   entry->TypeDet = NULL;
   dStr_truncate(entry->Headers, 0);
   a_Url_free((DilloUrl *)entry->Location);
   entry->Location = NULL;
   Cache_auth_free(entry->Auth);
   entry->Auth = NULL;
   dStr_truncate(entry->Body, 0);
   entry->bytes_received = 0;
   entry->ContentLength = 0;
   entry->Flags = CA_InProgress;
}

/*
 * Try finding the url in the cache. If it hits, send the cache contents
 * from there. If it misses, set up a new connection.
 *
 * - 'client' contains client-specific information needed when supplying the response data
 * - 'Call' is the callback that receives the data
 * - 'CbData' is custom data passed to 'Call'
 *   Note: 'Call' and/or 'CbData' can be NULL, in that case they get set
 *   later by a_Client_dispatch, based on content/type and client data.
 *
 * Return value: A primary key for identifying the client,
 */
int a_Cache_open_url(ClientData *client, CA_Callback_t Call, void *CbData)
{
   int ClientKey;
   CacheEntry_t *entry = Cache_entry_search(client->url);

   if (URL_FLAGS(client->url) & URL_E2EQuery)
      Cache_entry_reinit_for_reload(entry);

   // client no longer needs its own copy of the URL
   a_Url_free(client->url);
   client->url = entry->Url;

   ClientKey = Cache_client_add(entry->Url, client, Call, CbData);
   if (entry->Flags & CA_GotHeaders)
      Cache_feed_clients_from_cache(entry);

   return ClientKey;
}

/*
 * Get cache entry status
 */
uint_t a_Cache_get_flags(const DilloUrl *url)
{
   CacheEntry_t *entry = Cache_entry_search(url);
   return (entry ? entry->Flags : 0);
}

/*
 * Get cache entry status (following redirections).
 */
uint_t a_Cache_get_flags_with_redirection(const DilloUrl *url)
{
   CacheEntry_t *entry = Cache_entry_search_with_redirect(url);
   return (entry ? entry->Flags : 0);
}

/*
 * Get current content type.
 */
static const char *Cache_current_content_type(CacheEntry_t *entry)
{
   return entry->TypeHdr ? entry->TypeHdr : entry->TypeDet;
}

/*
 * Get the pointer to the URL document, and its size, from the cache entry.
 * Return: 1 cached, 0 not cached.
 */
int a_Cache_get_buf(const DilloUrl *Url, char **PBuf, int *BufSize)
{
   CacheEntry_t *entry = Cache_entry_search_with_redirect(Url);
   if (entry) {
      *PBuf = entry->Body->str;
      *BufSize = entry->Body->len;
   } else {
      *PBuf = NULL;
      *BufSize = 0;
   }
   return (entry ? 1 : 0);
}

#ifndef DISABLE_COOKIES
static void Cache_report_cookie_accepted(CacheEntry_t *entry, const char *cookie)
{
   int i;
   ClientData *Client;

   dReturn_if_fail(entry && cookie);

   for (i = 0; (Client = dList_nth_data(Clients, i)); ++i)
      if (Client->url == entry->Url)
         a_Bw_cookie_accepted(Client->bw, cookie);
}
#endif

/*
 * Extract a single field from the headers, allocating and storing the value
 * in 'field'. ('fieldname' must not include the trailing ':')
 * Return a new string with the field-content if found (NULL on error)
 * (This function expects a '\r'-stripped header, with one-line header fields)
 */
static char *Cache_parse_field(const char *headers, const char *fieldname)
{
   char *field;
   uint_t i, j;

   for (i = 0; headers[i]; i++) {
      /* Search fieldname */
      for (j = 0; fieldname[j]; j++)
        if (D_ASCII_TOLOWER(fieldname[j]) != D_ASCII_TOLOWER(headers[i + j]))
           break;
      if (fieldname[j]) {
         /* skip to next line */
         for ( i += j; headers[i] != '\n'; i++);
         continue;
      }

      i += j;
      if (headers[i] == ':') {
        /* Field found! */
        while (headers[++i] == ' ' || headers[i] == '\t');
        for (j = 0; headers[i + j] != '\n'; j++);
        while (j && (headers[i + j - 1] == ' ' || headers[i + j - 1] == '\t'))
           j--;
        field = dStrndup(headers + i, j);
        return field;
      }
      while (headers[i] != '\n') i++;
   }
   return NULL;
}

/*
 * Extract multiple fields from the headers.
 */
static Dlist *Cache_parse_multiple_fields(const char *headers,
                                          const char *fieldname)
{
   uint_t i, j;
   Dlist *fields = dList_new(8);
   char *field;

   for (i = 0; headers[i]; i++) {
      /* Search fieldname */
      for (j = 0; fieldname[j]; j++)
         if (D_ASCII_TOLOWER(fieldname[j]) != D_ASCII_TOLOWER(headers[i + j]))
            break;
      if (fieldname[j]) {
         /* skip to next line */
         for (i += j; headers[i] != '\n'; i++);
         continue;
      }

      i += j;
      if (headers[i] == ':') {
         /* Field found! */
         while (headers[++i] == ' ' || headers[i] == '\t');
         for (j = 0; headers[i + j] != '\n'; j++);
         while (j && (headers[i + j - 1] == ' ' || headers[i + j - 1] == '\t'))
            j--;
         field = dStrndup(headers + i, j);
         dList_append(fields, field);
      } else {
         while (headers[i] != '\n') i++;
      }
   }

   if (dList_length(fields) == 0) {
      dList_free(fields);
      fields = NULL;
   }
   return fields;
}

/*
 * Scan, allocate, and set things according to header info.
 * (This function needs the whole header to work)
 */
static void Cache_parse_headers(CacheEntry_t *entry)
{
   char *headers = entry->Headers->str;
   size_t headerslen = entry->Headers->len;
   char *Type, *location_str, *hsts, *retry_after;
#ifndef DISABLE_COOKIES
   Dlist *Cookies;
#endif
   Dlist *warnings;
   void *data;
   int i;

   _MSG("Cache_parse_headers:\n%s", headers);

   char *s;

   /* How fundamentally hacky the header parsing is. Really we should build something like a dictionary, with the wrinkles that you can get multiple responses and more than one of the same header in a response (e.g., cookies). */

   while ((s = strstr(headers, "\n\n")) && s+2-headers < (int)headerslen) {
      MSG("We may have a preliminary HTTP response to skip over for %s.\n", URL_STR(entry->Url));
      // 1xx from server telling us to continue (proxy on server side?), or 200 Connection established when tunneling HTTPS through a proxy.
      if (!dStrnAsciiCasecmp(headers, "HTTP/", 5)) {
         size_t linelen = strcspn(headers, "\n");
         size_t protolen = strcspn(headers, " ");

         if (protolen && (linelen > protolen) && (headers[protolen+1] == '1' || headers[protolen+1] == '2')) {
            MSG("I think we do.\n");
            headerslen -= s + 2 - headers;
            headers = s+2;
         } else {
            break;
         }
      } else {
         break;
      }
   }

   if (headerslen && !dStrnAsciiCasecmp(headers, "HTTP/", 5)) {
      size_t linelen = strcspn(headers, "\n");
      size_t protolen = strcspn(headers, " ");

      if (protolen && (linelen > protolen)) {
         const char *code = headers + protolen + 1;

         if (code[0] == '3' && code[1] == '0' &&
             ((code[2] >= '0' && code[2] <= '3') || code[2] == '7' || code[2] == '8') &&
             (location_str = Cache_parse_field(headers, "Location"))) {
            /* 30x: URL redirection */
            entry->Location = a_Url_new(location_str, URL_STR_(entry->Url));

            if (!a_Domain_permit(entry->Url, entry->Location)) {
               MSG_WARN("Redirection not followed from %s to %s\n",
                   URL_HOST(entry->Url), URL_STR(entry->Location));
            } else {
               entry->Flags |= CA_Redirect;
            }
            dFree(location_str);
         } else if (strncmp(code, "401", 3) == 0) {
            entry->Auth = Cache_parse_multiple_fields(headers, "WWW-Authenticate");
            if (entry->Auth &&
                entry->req->sec_fetch_dest != HTTP_Sec_Fetch_Dest_document &&
                !a_Url_same_origin(entry->req->url, entry->req->referrer)) {
               MSG_WARN("Ignoring authentication demand made when fetching cross-origin resource at %s\n", URL_STR(entry->Url));
               Cache_auth_free(entry->Auth);
               entry->Auth = NULL;
            }
         } else if (strncmp(code, "404", 3) == 0) {
            entry->Flags |= CA_NotFound;
         }
      }
   }

   if ((warnings = Cache_parse_multiple_fields(headers, "Warning"))) {
      for (i = 0; (data = dList_nth_data(warnings, i)); ++i) {
         MSG_WARN("Warning header sent for %s: %s\n", URL_STR(entry->Url), (char *)data);
         dFree(data);
      }
      dList_free(warnings);
   }

   if ((retry_after = Cache_parse_field(headers, "Retry-After"))) {
      MSG("Server asks that you wait: %s seconds.\n", retry_after);
      dFree(retry_after);
   }
#if 0
   if ((CSP = Cache_parse_field(headers, "Content-Security-Policy"))) {
      MSG("Content-Security-Policy '%s' for %s\n", CSP, URL_STR(entry->Url));
      dFree(CSP);
   }
#endif
   if (prefs.http_insecure) {
      if (!dStrAsciiCasecmp(URL_SCHEME(entry->Url), "https") &&
          a_Url_host_type(URL_HOST(entry->Url)) == URL_HOST_NAME &&
          (hsts = Cache_parse_field(headers, "Strict-Transport-Security"))) {
         a_Hsts_set(hsts, entry->Url);
         dFree(hsts);
      }
   }

#ifndef DISABLE_COOKIES
   if (!(URL_FLAGS(entry->Url) & URL_SpamSafe) &&
      (Cookies = Cache_parse_multiple_fields(headers, "Set-Cookie"))) {
      if (!entry->req->referrer || a_Url_same_organization(entry->Url, entry->req->referrer)) {
         /* If cookies are third party, don't even consider them. */
         char *cookie, *server_date = Cache_parse_field(headers, "Date");
         for (i = 0; (cookie = dList_nth_data(Cookies, i)); ++i) {
            if (a_Cookies_set(cookie, entry->Url, server_date))
               Cache_report_cookie_accepted(entry, cookie);
         }
         dFree(server_date);
      }
      for (i = 0; (data = dList_nth_data(Cookies, i)); ++i)
         dFree(data);
      dList_free(Cookies);
   }
#endif /* !DISABLE_COOKIES */

   /* Get Content-Type */
   if ((Type = Cache_parse_field(headers, "Content-Type"))) {
      entry->TypeHdr = Type;
   }
}

/*
 * libcurl sends "\r\n" as a separate final header for HTTP, but for a file, it sends the last real header (currently Last-Modified) with the extra "\r\n" at the end.
 */
void a_Cache_reply_header(const DilloUrl *url, const char *buf, size_t buf_size)
{
   CacheEntry_t *entry = Cache_entry_search(url);
   Dstr *headers;
   const size_t terminator_len = (buf_size >=4 && !strncmp(buf+buf_size-4, "\r\n\r\n", 4)) ? 4 : 2;

   if (buf_size < 2) {
      MSG_ERR("libcurl provides a header without CRLF? That’s news to me.\n");
      return;
   }

   dReturn_if_fail (entry != NULL);

   headers = entry->Headers;

   dStr_append_l(headers, buf, buf_size - terminator_len);
   dStr_append_c(headers, '\n');
}

static void Cache_finish_msg(CacheEntry_t *entry)
{
   entry->Flags &= ~CA_InProgress;

   dStr_fit(entry->Body);                /* fit buffer size! */

   Cache_main_feed_clients(entry);
}

static char *Cache_get_referrer_policy(CacheEntry_t *entry)
{
   return entry ? Cache_parse_field(entry->Headers->str, "Referrer-Policy") : NULL;
}

char *a_Cache_get_referrer_policy_with_redirection(const DilloUrl *url)
{
   CacheEntry_t *entry = Cache_entry_search_with_redirect(url);

   return entry ? Cache_parse_field(entry->Headers->str, "Referrer-Policy") : NULL;
}

/*
 * Process redirections (HTTP 30x answers)
 *
 * TO CONSIDER: Maybe it would be better to have lists of clients hanging from the individual cache entries. Although then we’d have an additional list stuck on every cache entry.
 *
 * It’s also tempting to make this one for-loop instead of lots of for-loops, but then I’d be testing repeatedly for URL_SpamSafe and so on. There’s no pressing reason to prevent the not-common case of redirection from being relatively straightforward.
 */
static void Cache_redirect(CacheEntry_t *entry)
{
   uint_t i;
   ClientData *Client;

   /* Don't allow redirection for SpamSafe/local URLs */
   if (URL_FLAGS(entry->Url) & URL_SpamSafe) {
      for (i = 0; (Client = dList_nth_data(Clients, i)); ++i) {
         if (Client->url == entry->Url) {
            a_UIcmd_set_msg(Client->bw, "WARNING: disallowing local URL’s attempt to redirect.");
            a_Cache_stop_client(Client->Key);
         }
      }
      return;
   }

   for (i = 0; (Client = dList_nth_data(Clients, i)); ++i) {
      if (Client->url == entry->Url) {
         if (Client->RedirectLevel >= 5)
            entry->Flags |= CA_RedirectLoop;
      }
   }

   if (entry->Flags & CA_RedirectLoop) {
      for (i = 0; (Client = dList_nth_data(Clients, i)); ++i) {
         if (Client->url == entry->Url) {
            a_UIcmd_set_msg(Client->bw, "ERROR: redirect loop for: %s", URL_STR_(entry->Url));
            a_Cache_stop_client(Client->Key);
         }
      }
      return;
   }

   if (entry->Flags & CA_Redirect && entry->Location) {
      char *pol = Cache_get_referrer_policy(entry);
      ReferrerPolicyType referrerpolicy = a_Referrer_read(pol);
      dFree(pol);

      DilloUrl *NewUrl = a_Url_new(URL_STR_(entry->Location), URL_STR_(entry->Url));

      for (i = 0; (Client = dList_nth_data(Clients, i)); ++i) {
         if (Client->url == entry->Url) {
            int NewClientKey;
            int NewRedirectLevel = Client->RedirectLevel + 1;
            const bool_t user_saw_url = FALSE;
            HttpSecFetchDest dest = entry->req->sec_fetch_dest;
            FetchReq NewReq = {0, NewUrl, entry->Url, referrerpolicy, dest, user_saw_url};
            _MSG("Client key %d %s\n", Client->Key, Client->flags & CLIENT_RootUrl ? "Root" : "Non-root");
            _MSG("entry url '%s'\n new url  '%s'\n", URL_STR(entry->Url), URL_STR(NewUrl));
            if (Client->flags & CLIENT_RootUrl) {
               /* Redirection of the main page */
               NewClientKey = a_Nav_push(Client->bw, &NewReq);
               // the Nav_open_url() will stop the clients for that bw.
            } else {
               _MSG("non-root redirection for %s\n", URL_STR(entry->Url));
               ClientData *NewClient = a_Client_new(Client->bw, NewUrl);
               NewClient->flags = Client->flags;
               if ((NewClient->Image = Client->Image))
                  Client->Image = NULL;
               NewClient->bgColor = Client->bgColor;
               NewClient->bw = Client->bw;
               NewClientKey = a_Capi_open_url(NewClient, &NewReq, Client->Callback, Client->CbData);
               a_Cache_stop_client(Client->Key);
               // For completeness, it would be nice to do a_Bw_remove_url(entry->Url); here, but sometimes javascripted links cause image URLs to have the same URL as the page, and if we’re also redirecting the page right now, that would surely interfere with that. If bw held the root URL separately, we could do it.
            }
            if (NewClientKey) {
               ClientData *NewClient = dList_find_custom(Clients, INT2VOIDP(NewClientKey), Cache_client_by_key_cmp);

               if (!NewClient) {
                  MSG_ERR("Cache_redirect: client key but couldn’t find client\n");
               } else {
                  NewClient->RedirectLevel = NewRedirectLevel;
               }
            }
         }
      }
      a_Url_free(NewUrl);
   }
}

/*
 * Receive new data, update the reception buffer (for next read), update the
 * cache, and feed the clients.
 *
 * This function gets called whenever the IO has new data.
 *
 * Return -1 to stop, 0 to continue.
 */
int a_Cache_reply_body(const DilloUrl *url, int Op, const char *buf, size_t buf_size)
{
   CacheEntry_t *entry = Cache_entry_search(url);

   dReturn_val_if_fail (entry != NULL, -1);

   _MSG("__a_Cache_reply_body__\n");

   int ret = 0;

   if (!(entry->Flags & CA_GotHeaders)) {
      entry->Flags |= CA_GotHeaders;
      Cache_parse_headers(entry);
   }

   if (Op == IORead) {
      const char *str = buf;
      int len = buf_size;

      dStr_append_l(entry->Body, str, len);

      ret = Cache_main_feed_clients(entry);
   } else if (Op == IOClose) {
      Cache_finish_msg(entry);
   } else if (Op == IOAbort) {
      if (entry->Body->len) {
         MSG_WARN("Premature close for %s\n", URL_STR(entry->Url));
         Cache_finish_msg(entry);
      } else {

         /* REMOVED */
         Cache_entry_remove(entry, entry->Url);
      }
   }
   return ret;
}

typedef struct {
   Dlist *authChallengeHeaders;
   DilloUrl *url;
   BrowserWindow *bw;
} CacheAuthData_t;

/*
 * Ask for user/password and reload the page.
 */
static void Cache_auth_callback(void *vdata)
{
   CacheAuthData_t *data = (CacheAuthData_t *)vdata;
   if (a_Auth_do_auth(data->bw, data->authChallengeHeaders, data->url))
      a_Nav_reload(data->bw);
   Cache_auth_free(data->authChallengeHeaders);
   a_Url_free(data->url);
   dFree(data);
   Cache_auth_entry(NULL, NULL);
}

/*
 * Set a timeout function to ask for user/password.
 */
static void Cache_auth_entry(CacheEntry_t *entry, BrowserWindow *bw)
{
   static int busy = 0;
   CacheAuthData_t *data;

   if (!entry) {
      busy = 0;
   } else if (busy) {
      MSG_WARN("Cache_auth_entry: caught busy!\n");
   } else if (entry->Auth) {
      busy = 1;
      data = dNew(CacheAuthData_t, 1);
      data->url = a_Url_dup(entry->Url);
      data->bw = bw;
      data->authChallengeHeaders = entry->Auth;
      entry->Auth = NULL;
      a_Timeout_add(0.0, Cache_auth_callback, data);
   }
}

/*
 * Don't process data any further, but let the cache fill the entry.
 * (Currently used to handle CLIENT_RootUrl redirects,
 *  and to ignore image redirects --Jcid)
 */
static void Cache_null_client(int Op, void *CbData, void *Buf, uint_t BufSize)
{
}

/*
 * Let the client know that we're not following a redirection.
 */
static void Cache_provide_redirection_blocked_page(CacheEntry_t *entry,
                                                   ClientData *client)
{
   CA_Callback_t Callback;
   void *CbData;
   char *msg;

   a_Client_dispatch("text/html", client, &Callback, &CbData);
   msg = dStrconcat("<!doctype html><html><body>"
                    "Browser blocked a redirection from <a href=\"",
                    URL_STR(entry->Url), "\">", URL_STR(entry->Url),
                    "</a> to <a href=\"", URL_STR(entry->Location), "\">",
                    URL_STR(entry->Location), "</a> based on your domainrc "
                    "settings.</body></html>", NULL);
   (Callback)(CA_Send, CbData, msg, strlen(msg));
   dFree(msg);
}

/*
 * Update cache clients for a single cache-entry
 * Tasks:
 *   - Set the client function (if not already set)
 *   - Look if new data is available and pass it to client functions
 *   - Remove clients when done
 *   - Call redirect handler
 *
 * TODO: Implement CA_Abort Op in client callback
 */
static int Cache_main_feed_clients(CacheEntry_t *entry)
{
   uint_t i;
   int st;
   const char *Type;
   ClientData *Client;
   BrowserWindow *Client_bw = NULL;
   static bool_t Busy = FALSE;

   if (Busy)
      MSG_ERR("FATAL!: >>>> Cache_main_feed_clients Caught busy!!! <<<<\n");

   if (entry->Flags & CA_Redirect) {
      int i;
      ClientData *Client;

      for (i = 0; (Client = dList_nth_data(Clients, i)); ++i) {
         if (Client->url == entry->Url) {
            /* if there are any clients waiting for us */
            Cache_redirect(entry);
            break;
         }
      }
      return 0; // TODO I wanted to abort in case the redirect chooses to have infinite body that we have no need for, but the rest of the code takes Abort more seriously than we would want at present, and let’s see how things settle before complicating that.
   }

   if (entry->Body->len && !(entry->Flags & CA_GotContentType)) {
      st = a_Misc_get_content_type_from_data(
              entry->Body->str, entry->Body->len, &Type);
      _MSG("Cache: detected Content-Type '%s'\n", Type);
      if (!st) {
         entry->TypeDet = dStrdup(Type);
         entry->Flags |= CA_GotContentType;
      }
   }
   if (!(entry->Flags & CA_GotContentType) && (entry->Flags & CA_InProgress)) {
      return 0;  /* wait for more data */
   }

   Busy = TRUE;
   for (i = 0; (Client = dList_nth_data(Clients, i)); ++i) {
      if (Client->url == entry->Url) {
         Client_bw = Client->bw;  /* 'bw' in a local var */

         if (!(Client->flags & CLIENT_RootUrl) && entry->Flags & CA_NotFound) {
            /* For non root URLs, ignore 404 answers */
            Client->Callback = Cache_null_client;
         }

         /* Set the client function */
         if (!Client->Callback) {
            const char *curr_type = Cache_current_content_type(entry);

            Client->Callback = Cache_null_client;

            if (entry->Location && !(entry->Flags & CA_Redirect)) {
               /* Not following redirection, so don't display page body. */
            } else if (curr_type || !(entry->Flags & CA_InProgress)) {
               st = a_Client_dispatch(curr_type, Client,
                                   &Client->Callback,
                                   &Client->CbData);
            }
         }

         /* Send data to our client */
         if (Client->Callback && entry->Body->len > 0) {
            (Client->Callback)(CA_Send, Client->CbData, entry->Body->str, entry->Body->len);
         }

         /* Remove client when done */
         if (!(entry->Flags & CA_InProgress)) {
            if (Client->flags & CLIENT_RootUrl && entry->Location &&
                !(entry->Flags & CA_Redirect)) {
               Cache_provide_redirection_blocked_page(entry, Client);
            }

            a_Cache_report_progress(entry->Url, entry->bytes_received, entry->ContentLength); // if we’re cached, curl isn’t there to do this for us.

            /* We finished sending data, let the client know */
            (Client->Callback)(CA_Close, Client->CbData, entry->Body->str, entry->Body->len);
            Cache_client_remove(Client);
            --i; /* Keep the index value in the next iteration */
         }
      }
   } /* for */

   if (entry->Auth && !(entry->Flags & CA_InProgress)) {
      Cache_auth_entry(entry, Client_bw);
   }

   /* Trigger cleanup when there are no cache clients */
   if (dList_length(Clients) == 0) {
      a_Dicache_cleanup();
   }

   Busy = FALSE;
   _MSG("Clients ====> %d\n", dList_length(Clients));

   return 0;
}

/*
 * Callback function for Cache_feed_clients_from_cache.
 */
static void Cache_feed_clients_from_cache_cb()
{
   CacheEntry_t *entry;

   while ((entry = (CacheEntry_t *)dList_nth_data(ReawakenedCacheEntries, 0))) {
      Cache_main_feed_clients(entry);
      dList_remove(ReawakenedCacheEntries, entry);
   }
   ReawakenedCacheEntriesTimeoutSet = FALSE;
}

/*
 * Set a timeout for Cache_main_feed_clients, giving us time to unwind first.
 */
static void Cache_feed_clients_from_cache(CacheEntry_t *entry)
{
   /* there's no need to repeat entries */
   if (!dList_find(ReawakenedCacheEntries, entry))
      dList_append(ReawakenedCacheEntries, entry);

   if (!ReawakenedCacheEntriesTimeoutSet) {
      _MSG("  Setting timeout callback\n");
      a_Timeout_add(0.0, Cache_feed_clients_from_cache_cb, NULL);
      ReawakenedCacheEntriesTimeoutSet = TRUE;
   }
}

/*
 * It would have been convenient to get this earlier from CURLINFO_SIZE_DOWNLOAD_T and send it with reply_body(), but the value wasn’t set yet for file: urls. It worked for https:.
 *
 * bytes_received is of the actual reply body, which may be compressed.
 */
void a_Cache_report_progress(const DilloUrl *url, size_t bytes_received, size_t content_length) {
   int i;
   ClientData *Client;
   CacheEntry_t *entry = Cache_entry_search(url);

   /* Assert a valid entry (not aborted) */
   dReturn_if_fail (entry != NULL);

   entry->bytes_received = bytes_received;
   entry->ContentLength = content_length;

   for (i = 0; (Client = dList_nth_data(Clients, i)); ++i) {
      if (Client->url == entry->Url) {
         const double proportion = content_length > 0 ? ((double)bytes_received / content_length) : 1.0f;
         const size_t estimated_total = proportion ? entry->Body->len / proportion : entry->Body->len;
         const size_t bytes_still_expected = estimated_total - entry->Body->len;

         _MSG("%zu bytes_received, %ld content-length, %f proportion, %zu estimated total, %zd bytes_still_expected\n", bytes_received, content_length, proportion, estimated_total, bytes_still_expected);

         a_Bw_data_received(Client->bw, Client->url, entry->Body->len, bytes_still_expected);
         break;
      }
   }
}

/*
 * Last Client for this entry?
 * Return: Client if true, NULL otherwise
 * (cache.c has only one call to a capi function. This avoids a second one)
 */
ClientData *a_Cache_client_get_if_unique(int Key)
{
   int i, n = 0;
   ClientData *Client, *iClient;

   if ((Client = dList_find_custom(Clients, INT2VOIDP(Key),
                                   Cache_client_by_key_cmp))) {
      for (i = 0; (iClient = dList_nth_data(Clients, i)); ++i) {
         if (iClient->url == Client->url) {
            ++n;
         }
      }
   }
   return (n == 1) ? Client : NULL;
}

/*
 * Stop a client
 * TODO: notify the dicache and upper layers    [ <-- very old comment. Who knows how relevant it is today]
 */
void a_Cache_stop_client(int Key)
{
   ClientData *Client;
   CacheEntry_t *entry;

   if ((Client = dList_find_custom(Clients, INT2VOIDP(Key),
                                   Cache_client_by_key_cmp))) {

      /* 2025: This removal has been in the code since 2009 (revision 812), but doesn’t it seem strange that any client stopping completely removes an entry that might feed other clients? */
      if ((entry = Cache_entry_search(Client->url)))
         dList_remove(ReawakenedCacheEntries, entry);

      Cache_client_remove(Client);

   } else {
      _MSG("WARNING: Cache_stop_client, nonexistent client\n");
   }
}


/*
 * Memory deallocator (only called at exit time)
 */
void a_Cache_freeall(void)
{
   ClientData *Client;
   void *data;

   /* free the client list */
   while ((Client = dList_nth_data(Clients, 0)))
      Cache_client_remove(Client);

   /* Remove every cache entry */
   while ((data = dList_nth_data(Cache_entries, 0))) {
      dList_remove_fast(Cache_entries, data);
      Cache_entry_free(data);
   }
   /* Remove the cache list */
   dList_free(Cache_entries);
}
