/*
 * File: capi.c
 *
 * Copyright 2002-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

/*
 * Cache API. This is the module that manages the cache.
 */

#include <string.h>
#include <errno.h>

#include <sys/stat.h> /* for now for filedir stupidity */

#include "config.h"
#include "msg.h"
#include "capi.h"
#include "IO/curl.h"
#include "IO/filedir.h"
#include "IO/datauri.h"
#include "history.h"
#include "nav.h"
#include "timeout.hh"
#include "uicmd.hh"
#include "domain.h"
#include "client.hh"

/* ------------------------------------------------------------------------- */

/* In a way, these passthroughs are like the old Dillo, but I brought this in, thinking that the cache clients might come up here and perhaps their feeding, and that it might make redirection and/or auth cleaner. But so far I haven’t actually seen the way to do that and accomplish anything. If the clients and their feeding come up here, for one thing, there really won’t be much left of cache. Is that conceptually good for anything? */

void a_Capi_reply_header(ulong_t req_id, const char *buf, size_t buf_size)
{
   const FetchReq *f = a_FetchReq_get(req_id);

   if (f) {
      a_Cache_reply_header(f->url, buf, buf_size);
   }
}

int a_Capi_reply_body(ulong_t req_id, const char *buf, size_t buf_size)
{
   int ret;
   const FetchReq *f = a_FetchReq_get(req_id);

   if (f) {
      ret = a_Cache_reply_body(f->url, IORead, buf, buf_size);
   } else {
      ret = -1;
   }
   return ret;
}

void a_Capi_reply_finished(ulong_t req_id)
{
   const FetchReq *f = a_FetchReq_get(req_id);

   if (f) {
      a_Cache_reply_body(f->url, IOClose, NULL, 0);
      a_FetchReq_deactivate(f);
   }
}

void a_Capi_reply_aborted(ulong_t req_id)
{
   const FetchReq *f = a_FetchReq_get(req_id);

   if (f) {
      a_Cache_reply_body(f->url, IOAbort, NULL, 0);
      a_FetchReq_deactivate(f);
   }
}

void a_Capi_report_progress(ulong_t req_id, size_t bytes_received, size_t content_length) {
   const FetchReq *f = a_FetchReq_get(req_id);

   if (f) {
      a_Cache_report_progress(f->url, bytes_received, content_length);
   }
}

/*
 * Shall we permit this request to open a URL?
 */
static bool_t Capi_request_permitted(ClientData *client, const DilloUrl *referrer, bool_t user_saw_url)
{
   bool_t permit = FALSE;

   /* For user_saw_url, the user knows what URL we’re going to (listed on command line, entered into Location bar, saw link text when clicking a link or a form submit button, saw the URL of a style sheet in menu list, and maybe more cases. If they know what they’re getting into, let them do it.
    *
    * There are a few cases where that can be FALSE but the referrer is NULL (using forward or backward buttons, repush)
    */
   if (user_saw_url || !referrer) {
      return TRUE;
   }

   if (!dStrAsciiCasecmp(URL_SCHEME(referrer), "https")) {
      const char *s = URL_SCHEME(client->url);

      /* As of 2015, blocking of "active" mixed content is widespread
       * (style sheets, javascript, fonts, etc.), but the big browsers aren't
       * quite in a position to block "passive" mixed content (images) yet.
       * (Not clear whether there's consensus on which category to place
       * background images in.)
       */
      if (dStrAsciiCasecmp(s, "https") && dStrAsciiCasecmp(s, "data")) {
         MSG("capi: Blocked mixed content: %s -> %s\n",
             URL_STR(referrer), URL_STR(client->url));
         return FALSE;
      }
   }

   if (a_Capi_get_flags(client->url) & CAPI_IsCached ||
       a_Domain_permit(referrer, client->url)) {
      permit = TRUE;
   }
   return permit;
}

/*
 * Most used function for requesting a URL.
 *
 * Return value: A primary key for identifying the client,
 *               0 if the client is aborted in the process.
 */
int a_Capi_open_url(ClientData *client, FetchReq *callerreq, CA_Callback_t Call, void *CbData)
{
   int key = 0;

   // TODO See how a client is made and passed to us, and freed if the request isn’t permitted, whereas a good FetchReq isn’t made until we know we are making a new request of the outside world? The thinking is that most requests are probably permitted and clients therefore needed, although that depends on domainrc settings, and most FetchReqs probably not needed because something is cached (or not permitted). But that hasn’t been tested.

   if (Capi_request_permitted(client, callerreq->referrer, callerreq->user_saw_url)) {
      FetchReq *req = NULL;

      if (!(a_Capi_get_flags(client->url) & CAPI_IsCached)) {
         req = a_FetchReq_official(callerreq); // Successfully got through! Make a permanent one.
         a_FetchReq_activate(req);
         a_Cache_new_fetch(req);
      } else if (URL_FLAGS(client->url) & URL_E2EQuery) {
         req = a_Cache_get_req(client->url); // 2025: Apparently you are supposed to reuse the stuff you did to fetch the first time. Same referrer and such.
         a_FetchReq_activate(req);
      }

      if (req) {
         a_Curl_abort_reply(client->url);
      }
      key = a_Cache_open_url(client, Call, CbData);
      if (key) {
         a_Bw_add_client(client->bw, key, client->flags & CLIENT_RootUrl);
         a_Bw_add_url(client->bw, client->url);
      }
      if (req) {
         struct stat sb;
         size_t ignored;
         TimeoutCb_t request_fn;
         char *path = NULL;

         void *id_as_ptr = (void*) req->id;

         request_fn = (!dStrAsciiCasecmp(URL_SCHEME(client->url), "data")) ?
                      a_Datauri_request :
                      (!dStrAsciiCasecmp(URL_SCHEME(client->url), "file") &&
                       (path = a_Url_decode_hex_str(URL_PATH(client->url), &ignored)) &&
                       !stat(path, &sb) &&
                       S_ISDIR(sb.st_mode)) ?
                      a_Filedir_request :
                      a_Curl_request;
         /* My memory of why this needed to be placed behind a callback when getting curl integrated a few years ago is that curl would feed us a file: url straight through without there being any callbacks in the way to let us unwind first in preparation here on the request-sending side.
          */
         a_Timeout_add(0, request_fn, id_as_ptr);
         dFree(path);
      }
   } else {
      a_Client_free(client);
   }
   return key;
}

/*
 * Convert cache-defined flags to Capi ones.
 */
static int Capi_map_cache_flags(uint_t flags)
{
   int status = 0;

   if (flags) {
      status |= CAPI_IsCached;
      if (!(flags & CA_InProgress))
         status |= CAPI_Completed;
   }
   return status;
}

/*
 * Return status information of an URL's content-transfer process.
 */
int a_Capi_get_flags(const DilloUrl *Url)
{
   uint_t flags = a_Cache_get_flags(Url);
   int status = flags ? Capi_map_cache_flags(flags) : 0;
   return status;
}

/*
 * Same as a_Capi_get_flags() but following redirections.
 */
int a_Capi_get_flags_with_redirection(const DilloUrl *Url)
{
   uint_t flags = a_Cache_get_flags_with_redirection(Url);
   int status = flags ? Capi_map_cache_flags(flags) : 0;
   return status;
}

/*
 * Get the cache's buffer for the URL, and its size.
 * Return: 1 cached, 0 not cached.
 */
int a_Capi_get_buf(const DilloUrl *Url, char **PBuf, int *BufSize)
{
   return a_Cache_get_buf(Url, PBuf, BufSize);
}

char *a_Capi_get_referrer_policy_with_redirection(const DilloUrl *url)
{
   return a_Cache_get_referrer_policy_with_redirection(url);
}

/*
 * Remove a client from the cache client queue.
 * force = also abort the CCC if this is the last client.
 */
void a_Capi_stop_client(int Key, int force)
{
   ClientData *Client;

   _MSG("a_Capi_stop_client:  force=%d\n", force);

   Client = a_Cache_client_get_if_unique(Key);
//   if (Client && (force || Client->BufSize == 0)) {
//      /* remove empty entries too */
   if (Client && force) {
      a_Curl_abort_reply(Client->url);
   }
   a_Cache_stop_client(Key);
}
