/*
 * File: cookies.c
 * Cookies server.
 *
 * Copyright 2001 Lars Clausen   <lrclause@cs.uiuc.edu>
 *                Jörgen Viksell <jorgen.viksell@telia.com>
 * Copyright 2002-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 */

#include "msg.h"

#ifdef DISABLE_COOKIES

/*
 * Initialize the cookies module
 */
void a_Cookies_init(void)
{
   MSG_WARN("Cookies: absolutely disabled at compilation time.\n");
}

#else

/* The current standard for cookies is RFC 6265.
 *
 * Info from 2009 on cookies in the wild:
 *  http://www.ietf.org/mail-archive/web/http-state/current/msg00078.html
 * And dates specifically:
 *  http://www.ietf.org/mail-archive/web/http-state/current/msg00128.html
 */

#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stddef.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>       /* for time() and time_t */
#include <ctype.h>
#include <limits.h>

#include "list.h"
#include "cookies.h"

/* The maximum length of a line in the cookie file */
#define LINE_MAXLEN 4096

#define MAX_DOMAIN_COOKIES 20
#define MAX_TOTAL_COOKIES 1200

typedef enum {
   COOKIE_ACCEPT,
   COOKIE_ACCEPT_SESSION,
   COOKIE_DENY
} CookieControlAction;

typedef enum {
   COOKIE_SAMESITE_NOTSET,
   COOKIE_SAMESITE_STRICT,
   COOKIE_SAMESITE_LAX,
   COOKIE_SAMESITE_NONE
} CookieSameSiteValue;

typedef struct {
   char *domain;
   CookieControlAction action;
} CookieControl;

typedef struct {
   char *domain;
   Dlist *cookies;
} DomainNode;

typedef struct {
   char *name;
   char *value;
   char *domain;
   char *path;
   time_t expires_at;
   bool_t host_only;
   bool_t secure;
   bool_t session_only;
   CookieSameSiteValue same_site;
   long last_used;
} CookieData_t;

/*
 * Local data
 */

static Dlist *all_cookies;

/* List of DomainNode. Each node holds a domain and its list of cookies */
static Dlist *domains;

/* Variables for access control */
static CookieControl *ccontrol = NULL;
static int num_ccontrol = 0;
static int num_ccontrol_max = 1;
static CookieControlAction default_action = COOKIE_DENY;

static long cookies_use_counter = 0;
static bool_t disabled;
static FILE *file_stream;
static const char *const cookies_txt_header_str =
"# HTTP Cookie File\n"
"# This is a generated file!  Do not edit.\n"
"# [domain  subdomains  path  secure  expiry_time  name  value]\n\n";

/* The epoch is Jan 1, 1970. When there is difficulty in representing future
 * dates, use the (by far) most likely last representable time in Jan 19, 2038.
 */
static struct tm cookies_epoch_tm = {0, 0, 0, 1, 0, 70, 0, 0, 0, 0, 0};
static time_t cookies_epoch_time, cookies_future_time;

/*
 * Forward declarations
 */

static CookieControlAction Cookies_control_check_domain(const char *domain);
static int Cookie_control_init(void);
static void Cookies_add_cookie(CookieData_t *cookie);
static int Cookies_cmp(const void *a, const void *b);

/*
 * Compare function for searching a domain node
 */
static int Domain_node_cmp(const void *v1, const void *v2)
{
   const DomainNode *n1 = v1, *n2 = v2;

   return dStrAsciiCasecmp(n1->domain, n2->domain);
}

/*
 * Compare function for searching a domain node by domain
 */
static int Domain_node_by_domain_cmp(const void *v1, const void *v2)
{
   const DomainNode *node = v1;
   const char *domain = v2;

   return dStrAsciiCasecmp(node->domain, domain);
}

/*
 * Delete node. This will not free any cookies that might be in node->cookies.
 */
static void Cookies_delete_node(DomainNode *node)
{
   dList_remove(domains, node);
   dFree(node->domain);
   dList_free(node->cookies);
   dFree(node);
}

/*
 * Return a file pointer. If the file doesn't exist, try to create it,
 * with the optional 'init_str' as its content.
 */
static FILE *Cookies_fopen(const char *filename, const char *mode,
                           const char *init_str)
{
   FILE *F_in;
   int fd, rc;

   if ((F_in = fopen(filename, mode)) == NULL) {
      /* Create the file */
      fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
      if (fd != -1) {
         if (init_str) {
            rc = write(fd, init_str, strlen(init_str));
            if (rc == -1) {
               MSG("Cookies: Could not write initial string to file %s: %s\n",
                  filename, dStrerror(errno));
            }
         }
         dClose(fd);

         MSG("Created file: %s\n", filename);
         F_in = fopen(filename, mode);
      } else {
         MSG("Could not create file: %s!\n", filename);
      }
   }

   if (F_in) {
      /* set close on exec */
      fcntl(fileno(F_in), F_SETFD, FD_CLOEXEC | fcntl(fileno(F_in), F_GETFD));
   }

   return F_in;
}

static void Cookies_free_cookie(CookieData_t *cookie)
{
   dFree(cookie->name);
   dFree(cookie->value);
   dFree(cookie->domain);
   dFree(cookie->path);
   dFree(cookie);
}

static void Cookies_tm_init(struct tm *tm)
{
   tm->tm_sec = cookies_epoch_tm.tm_sec;
   tm->tm_min = cookies_epoch_tm.tm_min;
   tm->tm_hour = cookies_epoch_tm.tm_hour;
   tm->tm_mday = cookies_epoch_tm.tm_mday;
   tm->tm_mon = cookies_epoch_tm.tm_mon;
   tm->tm_year = cookies_epoch_tm.tm_year;
   tm->tm_isdst = cookies_epoch_tm.tm_isdst;
}

/*
 * Read in cookies from 'stream' (cookies.txt)
 */
static void Cookies_load_cookies(FILE *stream)
{
   char line[LINE_MAXLEN];

   all_cookies = dList_new(32);
   domains = dList_new(32);

   /* Get all lines in the file */
   while (!feof(stream)) {
      line[0] = '\0';
      if ((fgets(line, LINE_MAXLEN, stream) == NULL) && ferror(stream)) {
         MSG("Error while reading from cookies.txt: %s\n", dStrerror(errno));
         break; /* bail out */
      }

      /* Remove leading and trailing whitespaces */
      dStrstrip(line);

      if ((line[0] != '\0') && (line[0] != '#')) {
         /*
          * Split the row into pieces using a tab as the delimiter.
          * pieces[0] The domain name
          * pieces[1] TRUE/FALSE: is the domain a suffix, or a full domain?
          * pieces[2] The path
          * pieces[3] TRUE/FALSE: is the cookie for secure use only?
          * pieces[4] Timestamp of expire date
          * pieces[5] Name of the cookie
          * pieces[6] Value of the cookie
          */
         CookieControlAction action;
         char *piece;
         char *line_marker = line;
         CookieData_t *cookie = dNew0(CookieData_t, 1);

         cookie->session_only = FALSE;
         cookie->same_site = COOKIE_SAMESITE_NOTSET;
         cookie->domain = dStrdup(dStrsep(&line_marker, "\t"));
         piece = dStrsep(&line_marker, "\t");
         if (piece != NULL && piece[0] == 'F')
            cookie->host_only = TRUE;
         cookie->path = dStrdup(dStrsep(&line_marker, "\t"));
         piece = dStrsep(&line_marker, "\t");
         if (piece != NULL && piece[0] == 'T')
            cookie->secure = TRUE;
         piece = dStrsep(&line_marker, "\t");
         if (piece != NULL) {
            /* There is some problem with simply putting the maximum value
             * into tm.tm_sec (although a value close to it works).
             */
            long seconds = strtol(piece, NULL, 10);
            struct tm tm;
            Cookies_tm_init(&tm);
            tm.tm_min += seconds / 60;
            tm.tm_sec += seconds % 60;
            cookie->expires_at = mktime(&tm);
         } else {
            cookie->expires_at = (time_t) -1;
         }
         cookie->name = dStrdup(dStrsep(&line_marker, "\t"));
         cookie->value = dStrdup(line_marker ? line_marker : "");

         if (!cookie->domain || cookie->domain[0] == '\0' ||
             !cookie->path || cookie->path[0] != '/' ||
             !cookie->name || !cookie->value) {
            MSG("Malformed line in cookies.txt file!\n");
            Cookies_free_cookie(cookie);
            continue;
         }

         action = Cookies_control_check_domain(cookie->domain);
         if (action == COOKIE_DENY) {
            Cookies_free_cookie(cookie);
            continue;
         } else if (action == COOKIE_ACCEPT_SESSION) {
            cookie->session_only = TRUE;
         }

         /* Save cookie in memory */
         Cookies_add_cookie(cookie);
      }
   }
   MSG("Cookies loaded: %d.\n", dList_length(all_cookies));
}

/*
 * Initialize the cookies module
 * (The 'disabled' variable is writeable only within Cookies_init)
 */
void a_Cookies_init()
{
   char *filename;
#if 0
#ifndef HAVE_LOCKF
   struct flock lck;
#endif
#endif
   struct tm future_tm = {7, 14, 3, 19, 0, 138, 0, 0, 0, 0, 0};

   /* Default setting */
   disabled = TRUE;

   cookies_epoch_time = mktime(&cookies_epoch_tm);
   cookies_future_time = mktime(&future_tm);

   /* Read and parse the cookie control file (cookiesrc) */
   if (Cookie_control_init() != 0) {
      MSG("Cookies disabled.\n");
      return;
   }

   /* Get a stream for the cookies file */
   filename = dStrconcat(dGethomedir(), "/.dillo/cookies.txt", NULL);
   file_stream = Cookies_fopen(filename, "r+", cookies_txt_header_str);

   dFree(filename);

   if (!file_stream) {
      MSG("ERROR: Can't open ~/.dillo/cookies.txt; disabling cookies\n");
      return;
   }

   /* disable the file locking. A small price to pay to get rid of the Dillo
    * DPIs.
    */
#if 1
   disabled = FALSE;
#else
   /* Try to get a lock from the file descriptor */
#ifdef HAVE_LOCKF
   disabled = (lockf(fileno(file_stream), F_TLOCK, 0) == -1);
#else /* POSIX lock */
   lck.l_start = 0; /* start at beginning of file */
   lck.l_len = 0;  /* lock entire file */
   lck.l_type = F_WRLCK;
   lck.l_whence = SEEK_SET;  /* absolute offset */

   disabled = (fcntl(fileno(file_stream), F_SETLK, &lck) == -1);
#endif
   if (disabled) {
      MSG("The cookies file has a file lock; disabling cookies!\n");
      fclose(file_stream);
      return;
   }
#endif
   MSG("Enabling cookies as per cookiesrc...\n");

   Cookies_load_cookies(file_stream);
}

/*
 * Flush cookies to disk and free all the memory allocated.
 */
static void Cookies_save_and_free()
{
   int i, fd, saved = 0;
   DomainNode *node;
   CookieData_t *cookie;
   time_t now;

#if 0
#ifndef HAVE_LOCKF
   struct flock lck;
#endif
#endif
   if (disabled)
      return;

   now = time(NULL);

   rewind(file_stream);
   fd = fileno(file_stream);
   if (ftruncate(fd, 0) == -1)
      MSG("Cookies: Truncate file stream failed: %s\n", dStrerror(errno));
   fprintf(file_stream, "%s", cookies_txt_header_str);

   /* Iterate cookies per domain, saving and freeing */
   while ((node = dList_nth_data(domains, 0))) {
      for (i = 0; (cookie = dList_nth_data(node->cookies, i)); ++i) {
         if (!cookie->session_only && cookie->same_site != COOKIE_SAMESITE_STRICT && difftime(cookie->expires_at, now) > 0) {
            /* We don’t have to throw out strict cookies when saving, but we would need to make an incompatible change with a new field in the cookies file. */
            int len;
            char buf[LINE_MAXLEN];

            len = snprintf(buf, LINE_MAXLEN, "%s\t%s\t%s\t%s\t%ld\t%s\t%s\n",
                           cookie->domain,
                           cookie->host_only ? "FALSE" : "TRUE",
                           cookie->path,
                           cookie->secure ? "TRUE" : "FALSE",
                           (long) difftime(cookie->expires_at,
                                           cookies_epoch_time),
                           cookie->name,
                           cookie->value);
            if (len < LINE_MAXLEN) {
               fprintf(file_stream, "%s", buf);
               saved++;
            } else {
               MSG("Not saving overly long cookie for %s.\n", cookie->domain);
            }
         }
         Cookies_free_cookie(cookie);
      }
      Cookies_delete_node(node);
   }
   dList_free(domains);
   dList_free(all_cookies);

#if 0
#ifdef HAVE_LOCKF
   lockf(fd, F_ULOCK, 0);
#else  /* POSIX file lock */
   lck.l_start = 0; /* start at beginning of file */
   lck.l_len = 0;  /* lock entire file */
   lck.l_type = F_UNLCK;
   lck.l_whence = SEEK_SET;  /* absolute offset */

   fcntl(fileno(file_stream), F_SETLKW, &lck);
#endif
#endif
   fclose(file_stream);

   MSG("Cookies saved: %d.\n", saved);
}

/*
 * Month parsing
 */
static bool_t Cookies_get_month(struct tm *tm, const char **str)
{
   static const char *const months[] =
   { "Jan", "Feb", "Mar",
     "Apr", "May", "Jun",
     "Jul", "Aug", "Sep",
     "Oct", "Nov", "Dec"
   };
   int i;

   for (i = 0; i < 12; i++) {
      if (!dStrnAsciiCasecmp(months[i], *str, 3)) {
         _MSG("Found month: %s\n", months[i]);
         tm->tm_mon = i;
         *str += 3;
         return TRUE;
      }
   }
   return FALSE;
}

/*
 * As seen in the production below, it's just one digit or two.
 * Return the value, or -1 if no proper value found.
 */
static int Cookies_get_timefield(const char **str)
{
   int n;
   const char *s = *str;

   if (!isdigit(*s))
      return -1;

   n = *(s++) - '0';
   if (isdigit(*s)) {
      n *= 10;
      n += *(s++) - '0';
      if (isdigit(*s))
         return -1;
   }
   *str = s;
   return n;
}

/*
 * Time parsing: 'time-field ":" time-field ":" time-field'
 *               'time-field = 1*2DIGIT'
 */
static bool_t Cookies_get_time(struct tm *tm, const char **str)
{
   const char *s = *str;

   if ((tm->tm_hour = Cookies_get_timefield(&s)) == -1)
      return FALSE;

   if (*(s++) != ':')
      return FALSE;

   if ((tm->tm_min = Cookies_get_timefield(&s)) == -1)
      return FALSE;

   if (*(s++) != ':')
      return FALSE;

   if ((tm->tm_sec = Cookies_get_timefield(&s)) == -1)
      return FALSE;

   *str = s;
   return TRUE;
}

/*
 * Day parsing: "day-of-month    = 1*2DIGIT"
 */
static bool_t Cookies_get_day(struct tm *tm, const char **str)
{
   const char *s = *str;

   if ((tm->tm_mday = Cookies_get_timefield(&s)) == -1)
      return FALSE;

   *str = s;
   return TRUE;
}

/*
 * Date parsing: "year = 2*4DIGIT"
 */
static bool_t Cookies_get_year(struct tm *tm, const char **str)
{
   int n;
   const char *s = *str;

   if (isdigit(*s))
      n = *(s++) - '0';
   else
      return FALSE;
   if (isdigit(*s)) {
      n *= 10;
      n += *(s++) - '0';
   } else
      return FALSE;
   if (isdigit(*s)) {
      n *= 10;
      n += *(s++) - '0';
   }
   if (isdigit(*s)) {
      n *= 10;
      n += *(s++) - '0';
   }
   if (isdigit(*s)) {
      /* Sorry, users of prehistoric software in the year 10000! */
      return FALSE;
   }
   if (n >= 70 && n <= 99)
      n += 1900;
   else if (n <= 69)
      n += 2000;

   tm->tm_year = n - 1900;

   *str = s;
   return TRUE;
}

/*
 * As given in RFC 6265.
 */
static bool_t Cookies_date_delim(char c)
{
   return (c == '\x09' ||
           (c >= '\x20' && c <= '\x2F') ||
           (c >= '\x3B' && c <= '\x40') ||
           (c >= '\x5B' && c <= '\x60') ||
           (c >= '\x7B' && c <= '\x7E'));
}

/*
 * Parse date string.
 *
 * A true nightmare of date formats appear in cookies, so one basically
 * has to paw through the soup and look for anything that looks sufficiently
 * like any of the date fields.
 *
 * Return a pointer to a struct tm, or NULL on error.
 */
static struct tm *Cookies_parse_date(const char *date)
{
   bool_t found_time = FALSE, found_day = FALSE, found_month = FALSE,
          found_year = FALSE, matched;
   struct tm *tm = dNew0(struct tm, 1);
   const char *s = date;

   while (*s) {
      matched = FALSE;

      if (!found_time)
         matched = found_time = Cookies_get_time(tm, &s);
      if (!matched && !found_day)
         matched = found_day = Cookies_get_day(tm, &s);
      if (!matched && !found_month)
         matched = found_month = Cookies_get_month(tm, &s);
      if (!matched && !found_year)
         matched = found_year = Cookies_get_year(tm, &s);
      while (*s && !Cookies_date_delim(*s))
         s++;
      while (*s && Cookies_date_delim(*s))
         s++;
   }
   if (!found_time || !found_day || !found_month || !found_year) {
      dFree(tm);
      tm = NULL;
      MSG("In date \"%s\", format not understood.\n", date);
   }

   /* Error checks. This may be overkill.
    *
    * RFC 6265: "Note that leap seconds cannot be represented in this
    * syntax." I'm not sure whether that's good, but that's what it says.
    */
   if (tm &&
       !(tm->tm_mday > 0 && tm->tm_mday < 32 && tm->tm_mon >= 0 &&
         tm->tm_mon < 12 && tm->tm_year >= 0 && tm->tm_hour >= 0 &&
         tm->tm_hour < 24 && tm->tm_min >= 0 && tm->tm_min < 60 &&
         tm->tm_sec >= 0 && tm->tm_sec < 60)) {
      MSG("Date \"%s\" values not in range.\n", date);
      dFree(tm);
      tm = NULL;
   }

   return tm;
}

/*
 * Find the least recently used cookie among those in the provided list.
 */
static CookieData_t *Cookies_get_LRU(Dlist *cookies)
{
   int i, n = dList_length(cookies);
   CookieData_t *lru = dList_nth_data(cookies, 0);

   for (i = 1; i < n; i++) {
      CookieData_t *curr = dList_nth_data(cookies, i);

      if (curr->last_used < lru->last_used)
         lru = curr;
   }
   return lru;
}

/*
 * Delete expired cookies.
 * If node is given, only check those cookies.
 * Note that nodes can disappear if all of their cookies were expired.
 *
 * Return the number of cookies that were expired.
 */
static int Cookies_rm_expired_cookies(DomainNode *node)
{
   Dlist *cookies = node ? node->cookies : all_cookies;
   int removed = 0;
   int i = 0, n = dList_length(cookies);
   time_t now = time(NULL);

   while (i < n) {
      CookieData_t *c = dList_nth_data(cookies, i);

      if (difftime(c->expires_at, now) < 0) {
         DomainNode *currnode = node ? node :
              dList_find_sorted(domains, c->domain, Domain_node_by_domain_cmp);
         dList_remove(currnode->cookies, c);
         if (dList_length(currnode->cookies) == 0)
            Cookies_delete_node(currnode);
         dList_remove_fast(all_cookies, c);
         Cookies_free_cookie(c);
         n--;
         removed++;
      } else {
         i++;
      }
   }
   return removed;
}

/*
 * There are too many cookies. Choose one to remove and delete.
 * If node is given, select from among its cookies only.
 */
static void Cookies_too_many(DomainNode *node)
{
   CookieData_t *lru = Cookies_get_LRU(node ? node->cookies : all_cookies);

   MSG("Too many cookies! "
       "Removing LRU cookie for \'%s\': \'%s=%s\'\n", lru->domain,
       lru->name, lru->value);
   if (!node)
      node = dList_find_sorted(domains, lru->domain,Domain_node_by_domain_cmp);

   dList_remove(node->cookies, lru);
   dList_remove_fast(all_cookies, lru);
   Cookies_free_cookie(lru);
   if (dList_length(node->cookies) == 0)
      Cookies_delete_node(node);
}

static void Cookies_add_cookie(CookieData_t *cookie)
{
   Dlist *domain_cookies;
   CookieData_t *c;
   DomainNode *node;

   node = dList_find_sorted(domains, cookie->domain,Domain_node_by_domain_cmp);
   domain_cookies = (node) ? node->cookies : NULL;

   if (domain_cookies) {
      /* Remove any cookies with the same name, path, and host-only values. */
      while ((c = dList_find_custom(domain_cookies, cookie, Cookies_cmp))) {
         dList_remove(domain_cookies, c);
         dList_remove_fast(all_cookies, c);
         Cookies_free_cookie(c);
      }
   }

   if ((cookie->expires_at == (time_t) -1) ||
       (difftime(cookie->expires_at, time(NULL)) <= 0)) {
      /*
       * Don't add an expired cookie. Whether expiring now == expired, exactly,
       * is arguable, but we definitely do not want to add a Max-Age=0 cookie.
       */
      _MSG("Goodbye, cookie %s=%s d:%s p:%s\n", cookie->name,
           cookie->value, cookie->domain, cookie->path);
      Cookies_free_cookie(cookie);
   } else {
      if (domain_cookies && dList_length(domain_cookies) >=MAX_DOMAIN_COOKIES){
         int removed = Cookies_rm_expired_cookies(node);

         if (removed == 0) {
            Cookies_too_many(node);
         } else if (removed >= MAX_DOMAIN_COOKIES) {
            /* So many were removed that the node might have been deleted. */
            node = dList_find_sorted(domains, cookie->domain,
                                                    Domain_node_by_domain_cmp);
            domain_cookies = (node) ? node->cookies : NULL;
         }
      }
      if (dList_length(all_cookies) >= MAX_TOTAL_COOKIES) {
         if (Cookies_rm_expired_cookies(NULL) == 0) {
            Cookies_too_many(NULL);
         } else if (domain_cookies) {
            /* Our own node might have just been deleted. */
            node = dList_find_sorted(domains, cookie->domain,
                                                    Domain_node_by_domain_cmp);
            domain_cookies = (node) ? node->cookies : NULL;
         }
      }

      cookie->last_used = cookies_use_counter++;

      /* Actually add the cookie! */
      dList_append(all_cookies, cookie);

      if (!domain_cookies) {
         domain_cookies = dList_new(5);
         dList_append(domain_cookies, cookie);
         node = dNew(DomainNode, 1);
         node->domain = dStrdup(cookie->domain);
         node->cookies = domain_cookies;
         dList_insert_sorted(domains, node, Domain_node_cmp);
      } else {
         dList_append(domain_cookies, cookie);
      }
   }
   if (domain_cookies && (dList_length(domain_cookies) == 0))
      Cookies_delete_node(node);
}

/*
 * Return the attribute that is present at *cookie_str.
 */
static char *Cookies_parse_attr(const char **cookie_str)
{
   const char *str;
   uint_t len;

   while (dIsspace(**cookie_str))
      (*cookie_str)++;

   str = *cookie_str;
   /* find '=' at end of attr, ';' after attr/val pair, '\0' end of string */
   len = strcspn(str, "=;");
   *cookie_str += len;

   while (len && (str[len - 1] == ' ' || str[len - 1] == '\t'))
      len--;
   return dStrndup(str, len);
}

/*
 * Get the value in *cookie_str.
 */
static char *Cookies_parse_value(const char **cookie_str)
{
   uint_t len;
   const char *str;

   if (**cookie_str == '=') {
      (*cookie_str)++;
      while (dIsspace(**cookie_str))
         (*cookie_str)++;

      str = *cookie_str;
      /* finds ';' after attr/val pair or '\0' at end of string */
      len = strcspn(str, ";");
      *cookie_str += len;

      while (len && (str[len - 1] == ' ' || str[len - 1] == '\t'))
         len--;
   } else {
      str = *cookie_str;
      len = 0;
   }
   return dStrndup(str, len);
}

/*
 * Advance past any value
 */
static void Cookies_eat_value(const char **cookie_str)
{
   if (**cookie_str == '=')
      *cookie_str += strcspn(*cookie_str, ";");
}

/*
 * Return the number of seconds by which our clock is ahead of the server's
 * clock.
 */
static double Cookies_server_timediff(const char *server_date)
{
   double ret = 0;

   if (server_date) {
      struct tm *server_tm = Cookies_parse_date(server_date);

      if (server_tm) {
         time_t server_time = mktime(server_tm);

         if (server_time != (time_t) -1)
            ret = difftime(time(NULL), server_time);
         dFree(server_tm);
      }
   }
   return ret;
}

static void Cookies_unquote_string(char *str)
{
   if (str && str[0] == '\"') {
      uint_t len = strlen(str);

      if (len > 1 && str[len - 1] == '\"') {
         str[len - 1] = '\0';
         while ((*str = str[1]))
            str++;
      }
   }
}

/*
 * Parse cookie. A cookie might look something like:
 * "Name=Val; Domain=example.com; Max-Age=3600; HttpOnly"
 */
static CookieData_t *Cookies_parse(const char *cookie_str, const char *server_date)
{
   time_t now, limit;
   int thirty_days = 60 * 60 * 24 * 30;
   CookieData_t *cookie = NULL;
   const char *str = cookie_str;
   bool_t first_attr = TRUE;
   bool_t max_age = FALSE;
   bool_t expires = FALSE;

   /* Iterate until there is nothing left of the string */
   while (*str) {
      char *attr;
      char *value;

      /* Get attribute */
      attr = Cookies_parse_attr(&str);

      /* Get the value for the attribute and store it */
      if (first_attr) {
         if (*str != '=' || *attr == '\0') {
            /* disregard nameless cookie */
            dFree(attr);
            return NULL;
         }
         cookie = dNew0(CookieData_t, 1);
         cookie->name = attr;
         cookie->value = Cookies_parse_value(&str);

      } else if (dStrAsciiCasecmp(attr, "Path") == 0) {
         value = Cookies_parse_value(&str);
         dFree(cookie->path);
         cookie->path = value;
      } else if (dStrAsciiCasecmp(attr, "Domain") == 0) {
         value = Cookies_parse_value(&str);
         dFree(cookie->domain);
         cookie->domain = value;
      } else if (dStrAsciiCasecmp(attr, "Max-Age") == 0) {
         value = Cookies_parse_value(&str);
         if (isdigit(*value) || *value == '-') {
            long age;
            time_t now = time(NULL);
            struct tm *tm = gmtime(&now);

            errno = 0;
            age = (*value == '-') ? 0 : strtol(value, NULL, 10);

            if (errno == ERANGE ||
                (age > 0 && (age > INT_MAX - tm->tm_sec))) {
               /* let's not overflow */
               tm->tm_sec = INT_MAX;
            } else {
               tm->tm_sec += age;
            }
            cookie->expires_at = mktime(tm);
            if (age > 0 && cookie->expires_at == (time_t) -1) {
               cookie->expires_at = cookies_future_time;
            }
            _MSG("Cookie to expire at %s", ctime(&cookie->expires_at));
            expires = max_age = TRUE;
         }
         dFree(value);
      } else if (dStrAsciiCasecmp(attr, "Expires") == 0) {
         if (!max_age) {
            struct tm *tm;

            value = Cookies_parse_value(&str);
            Cookies_unquote_string(value);
            _MSG("Expires attribute gives %s\n", value);
            tm = Cookies_parse_date(value);
            if (tm) {
               tm->tm_sec += Cookies_server_timediff(server_date);
               cookie->expires_at = mktime(tm);
               if (cookie->expires_at == (time_t) -1 && tm->tm_year >= 138) {
                  /* Just checking tm_year does not ensure that the problem was
                   * inability to represent a distant date...
                   */
                  cookie->expires_at = cookies_future_time;
               }
               _MSG("Cookie to expire at %s", ctime(&cookie->expires_at));
               dFree(tm);
            } else {
               cookie->expires_at = (time_t) -1;
            }
            expires = TRUE;
            dFree(value);
         } else {
            Cookies_eat_value(&str);
         }
      } else if (dStrAsciiCasecmp(attr, "SameSite") == 0) {
         value = Cookies_parse_value(&str);
         if (!dStrAsciiCasecmp(value, "Lax")) {
            cookie->same_site = COOKIE_SAMESITE_LAX;
         } else if (!dStrAsciiCasecmp(value, "Strict")) {
            cookie->same_site = COOKIE_SAMESITE_STRICT;
            MSG("A STRICT COOKIE!!!\n");
         } else if (!dStrAsciiCasecmp(value, "None")) {
            cookie->same_site = COOKIE_SAMESITE_NONE;
         } else {
            MSG_WARN("Cookie SameSite value \"%s\" not recognized.\n", value);
         }
         dFree(value);
      } else if (dStrAsciiCasecmp(attr, "Secure") == 0) {
         cookie->secure = TRUE;
         Cookies_eat_value(&str);
      } else if (dStrAsciiCasecmp(attr, "HttpOnly") == 0) {
         Cookies_eat_value(&str);
      } else {
         MSG("Cookie contains unknown attribute: '%s'\n", attr);
         Cookies_eat_value(&str);
      }

      if (first_attr)
         first_attr = FALSE;
      else
         dFree(attr);

      if (*str == ';')
         str++;
   }

   /* Limit cookie expiration. RFC 6265bis suggests 400 days. */
   now = time(NULL);

   if (thirty_days > INT_MAX - now)  // wrapped
      limit = cookies_future_time;
   else
      limit = now + thirty_days;
   if (!expires || cookie->expires_at > limit)
      cookie->expires_at = limit;

   cookie->session_only = expires == FALSE;
   return cookie;
}

/*
 * Compare cookies by host_only, name, and path. Return 0 if equal.
 */
static int Cookies_cmp(const void *a, const void *b)
{
   const CookieData_t *ca = a, *cb = b;

   return (ca->host_only != cb->host_only) ||
          (strcmp(ca->name, cb->name) != 0) ||
          (strcmp(ca->path, cb->path) != 0);
}

/*
 * Is the domain an IP address?
 */
static bool_t Cookies_domain_is_ip(const char *domain)
{
   uint_t len;

   if (!domain)
      return FALSE;

   len = strlen(domain);

   if (len == strspn(domain, "0123456789.")) {
      _MSG("an IPv4 address\n");
      return TRUE;
   }
   if (strchr(domain, ':') &&
       (len == strspn(domain, "0123456789abcdefABCDEF:."))) {
      /* The precise format is shown in section 3.2.2 of rfc 3986 */
      MSG("an IPv6 address\n");
      return TRUE;
   }
   return FALSE;
}

/*
 * Check whether url_path path-matches cookie_path
 *
 * Note different user agents apparently vary in path-matching behaviour,
 * but this is the recommended method at the moment.
 */
static bool_t Cookies_path_matches(const char *url_path,
                                   const char *cookie_path)
{
   bool_t ret = TRUE;

   if (!url_path || !cookie_path) {
      ret = FALSE;
   } else {
      uint_t c_len = strlen(cookie_path);
      uint_t u_len = strlen(url_path);

      ret = (!strncmp(cookie_path, url_path, c_len) &&
             ((c_len == u_len) ||
              (c_len > 0 && cookie_path[c_len - 1] == '/') ||
              (url_path[c_len] == '/')));
   }
   return ret;
}

/*
 * If cookie path is not properly set, remedy that.
 */
static void Cookies_validate_path(CookieData_t *cookie, const char *url_path)
{
   if (!cookie->path || cookie->path[0] != '/') {
      dFree(cookie->path);

      if (url_path) {
         uint_t len = strlen(url_path);

         while (len && url_path[len] != '/')
            len--;
         cookie->path = dStrndup(url_path, len ? len : 1);
      } else {
         cookie->path = dStrdup("/");
      }
   }
}

/*
 * Check whether host name A domain-matches host name B.
 */
static bool_t Cookies_domain_matches(const char *A, const char *B)
{
   int diff;

   if (!A || !*A || !B || !*B)
      return FALSE;

   if (*B == '.')
      B++;

   /* Should we concern ourselves with trailing dots in matching (here or
    * elsewhere)? The HTTP State people have found that most user agents
    * don't, so: No.
    */

   if (!dStrAsciiCasecmp(A, B))
      return TRUE;

   if (Cookies_domain_is_ip(B))
      return FALSE;

   diff = strlen(A) - strlen(B);

   if (diff > 0) {
      /* B is the tail of A, and the match is preceded by a '.' */
      return (dStrAsciiCasecmp(A + diff, B) == 0 && A[diff - 1] == '.');
   } else {
      return FALSE;
   }
}

/*
 * Check whether domain is within organization.
 *
 * This is so similar to Cookies_domain_matches(), but I don’t want to move the moving-past-the-initial-dot to the calling function, because it seems like it should be part of the argument, and we don’t need the IP test here, so let’s just have two separate very similar functions.
 */
static bool_t Cookies_domain_in_organization(const char *domain, const char *org)
{
   int diff;

   if (!domain || !*domain || !org || !*org)
      return FALSE;

   if (*domain == '.')
      domain++;

   /* Should we concern ourselves with trailing dots in matching (here or
    * elsewhere)? The HTTP State people have found that most user agents
    * don't, so: No.
    */

   if (!dStrAsciiCasecmp(domain, org))
      return TRUE;

   diff = strlen(domain) - strlen(org);

   if (diff > 0) {
      /* org is the tail of domain, and the match is preceded by a '.' */
      return (dStrAsciiCasecmp(domain + diff, org) == 0 && domain[diff - 1] == '.');
   } else {
      return FALSE;
   }
}

/*
 * Validate cookies domain against some security checks.
 */
static bool_t Cookies_validate_domain(CookieData_t *cookie, const char *host)
{
   if (!cookie->domain) {
      cookie->domain = dStrdup(host);
      cookie->host_only = TRUE;
      return TRUE;
   }

   if (!Cookies_domain_matches(host, cookie->domain))
      return FALSE;

   if (!Cookies_domain_in_organization(cookie->domain, a_Url_host_get_organization(host)))
      return FALSE;

   _MSG("host %s and domain %s is all right\n", host, cookie->domain);
   return TRUE;
}

/*
 * Set the value corresponding to the cookie string
 * Return value: 0 accepted
 *              -1 rejected by domain checks
 *              -2 rejected for being malformed or trying to do something improper
 */
static int Cookies_set(const char *cookie_string, const DilloUrl *url,
                       const char *server_date)
{
   CookieControlAction action;
   CookieData_t *cookie;
   const char *url_host = URL_HOST(url),
              *url_path = URL_PATH(url);
   int ret = -1;

   action = Cookies_control_check_domain(url_host);
   if (action == COOKIE_DENY) {
      MSG("REJECTED COOKIE for %s\n", url_host);
   } else {
      MSG("Received cookie: %s: %s\n", url_host, cookie_string);
      ret = -2;
      if ((cookie = Cookies_parse(cookie_string, server_date))) {
         if (cookie->secure && dStrAsciiCasecmp(URL_SCHEME(url), "https")) {
            MSG("Cannot set Secure cookie from insecure url %s\n", URL_STR(url));
         } else if (!dStrnAsciiCasecmp(cookie->name, "__Secure-", 9) && !cookie->secure) {
            MSG("Attempt to set \"__Secure-\" cookie insecurely.\n");
         } else if (!dStrnAsciiCasecmp(cookie->name, "__Host-", 7) &&
                    !(cookie->secure &&
                      !dStrAsciiCasecmp(URL_SCHEME(url), "https") &&
                      !cookie->domain &&
                      (cookie->path && !strcmp(cookie->path, "/")))) {
            MSG("Attempt to set \"__Host-\" cookie improperly.\n");
         } else if (!Cookies_validate_domain(cookie, url_host)) {
            MSG("Rejecting cookie for domain %s from host %s path %s\n",
                cookie->domain, url_host, url_path);
         } else {
            Cookies_validate_path(cookie, url_path);
            if (action == COOKIE_ACCEPT_SESSION)
               cookie->session_only = TRUE;
            Cookies_add_cookie(cookie);
            ret = 0;
         }
         if (ret)
            Cookies_free_cookie(cookie);
      }
   }
   return ret;
}

/*
 * Compare the cookie with the supplied data to see whether it matches
 */
static bool_t Cookies_match(CookieData_t *cookie, const char *url_path,
                            bool_t host_only_val, bool_t is_tls, bool_t is_strict)
{
   if (cookie->host_only != host_only_val)
      return FALSE;

   /* Insecure cookies match both secure and insecure urls, secure
      cookies match only secure urls */
   if (cookie->secure && !is_tls)
      return FALSE;

   if (cookie->same_site == COOKIE_SAMESITE_STRICT && !is_strict)
      return FALSE;

   if (!Cookies_path_matches(url_path, cookie->path))
      return FALSE;

   /* It's a match */
   return TRUE;
}

static void Cookies_add_matching_cookies(const char *domain,
                                         const char *url_path,
                                         bool_t host_only_val,
                                         Dlist *matching_cookies,
                                         bool_t is_tls,
                                         bool_t is_strict)
{
   DomainNode *node = dList_find_sorted(domains, domain,
                                        Domain_node_by_domain_cmp);
   if (node) {
      int i;
      CookieData_t *cookie;
      Dlist *domain_cookies = node->cookies;

      for (i = 0; (cookie = dList_nth_data(domain_cookies, i)); ++i) {
         /* Remove expired cookie. */
         if (difftime(cookie->expires_at, time(NULL)) < 0) {
            _MSG("Goodbye, expired cookie %s=%s d:%s p:%s\n", cookie->name,
                 cookie->value, cookie->domain, cookie->path);
            dList_remove(domain_cookies, cookie);
            dList_remove_fast(all_cookies, cookie);
            Cookies_free_cookie(cookie);
            --i; continue;
         }
         /* Check if the cookie matches the requesting URL */
         if (Cookies_match(cookie, url_path, host_only_val, is_tls, is_strict)) {
            int j;
            CookieData_t *curr;
            uint_t path_length = strlen(cookie->path);

            cookie->last_used = cookies_use_counter;

            /* Longest cookies go first */
            for (j = 0;
                 (curr = dList_nth_data(matching_cookies, j)) &&
                  strlen(curr->path) >= path_length;
                 j++) ;
            dList_insert_pos(matching_cookies, cookie, j);
         }
      }

      if (dList_length(domain_cookies) == 0)
         Cookies_delete_node(node);
   }
}

/*
 * Return a string that contains all relevant cookies as headers.
 */
static char *Cookies_get(const DilloUrl *url, bool_t is_strict)
{
   char *domain_str, *str;
   CookieData_t *cookie;
   Dlist *matching_cookies;
   bool_t is_tls, is_ip_addr, host_only_val;
   const char *url_scheme = URL_SCHEME(url),
              *url_host = URL_HOST(url),
              *url_path = URL_PATH(url);

   Dstr *cookie_dstring;
   int i;

   matching_cookies = dList_new(8);

   /* Check if the protocol is secure or not */
   is_tls = (!dStrAsciiCasecmp(url_scheme, "https"));

   is_ip_addr = Cookies_domain_is_ip(url_host);

   /* If a cookie is set that lacks a Domain attribute, its domain is set to
    * the server's host and the host_only flag is set for that cookie. Such a
    * cookie can only be sent back to that host. Cookies with Domain attrs do
    * not have the host_only flag set, and may be sent to subdomains. Domain
    * attrs can have leading dots, which should be ignored for matching
    * purposes.
    */
   host_only_val = FALSE;
   if (!is_ip_addr) {
      /* e.g., sub.example.com set a cookie with domain ".sub.example.com". */
      domain_str = dStrconcat(".", url_host, NULL);
      Cookies_add_matching_cookies(domain_str, url_path, host_only_val,
                                   matching_cookies, is_tls, is_strict);
      dFree(domain_str);
   }
   host_only_val = TRUE;
   /* e.g., sub.example.com set a cookie with no domain attribute. */
   Cookies_add_matching_cookies(url_host, url_path, host_only_val,
                                matching_cookies, is_tls, is_strict);
   host_only_val = FALSE;
   /* e.g., sub.example.com set a cookie with domain "sub.example.com". */
   Cookies_add_matching_cookies(url_host, url_path, host_only_val,
                                matching_cookies, is_tls, is_strict);

   if (!is_ip_addr) {
      for (domain_str = strchr(url_host+1, '.');
           domain_str != NULL && *domain_str;
           domain_str = strchr(domain_str+1, '.')) {
         /* e.g., sub.example.com set a cookie with domain ".example.com". */
         Cookies_add_matching_cookies(domain_str, url_path, host_only_val,
                                      matching_cookies, is_tls, is_strict);
         if (domain_str[1]) {
            domain_str++;
            /* e.g., sub.example.com set a cookie with domain "example.com".*/
            Cookies_add_matching_cookies(domain_str, url_path, host_only_val,
                                         matching_cookies, is_tls, is_strict);
         }
      }
   }

   /* Found the cookies, now make the string */
   cookie_dstring = dStr_new("");
   if (dList_length(matching_cookies) > 0) {

// libcurl just wants the guts of the header
//    dStr_sprintfa(cookie_dstring, "Cookie: ");


      for (i = 0; (cookie = dList_nth_data(matching_cookies, i)); ++i) {
         dStr_sprintfa(cookie_dstring, "%s=%s", cookie->name, cookie->value);
         dStr_append(cookie_dstring,
//                     dList_length(matching_cookies) > i + 1 ? "; " : "\r\n");
                     dList_length(matching_cookies) > i + 1 ? "; " : "");
      }
   }

   dList_free(matching_cookies);
   str = cookie_dstring->str;
   dStr_free(cookie_dstring, FALSE);

   if (*str) {
      MSG("SEND COOKIE: %s: %s\n", url_host, str);
      cookies_use_counter++;
   }
   return str;
}

/* -------------------------------------------------------------
 *                    Access control routines
 * ------------------------------------------------------------- */


/*
 * Get the cookie control rules (from cookiesrc).
 * Return value:
 *   0 = Parsed OK, with cookies enabled
 *   1 = Parsed OK, with cookies disabled
 *   2 = Can't open the control file
 */
static int Cookie_control_init(void)
{
   CookieControl cc;
   FILE *stream;
   char *filename, *rc;
   char line[LINE_MAXLEN];
   char domain[LINE_MAXLEN];
   char rule[LINE_MAXLEN];
   bool_t enabled = FALSE;

   /* Get a file pointer */
   filename = dStrconcat(dGethomedir(), "/.dillo/cookiesrc", NULL);
   stream = Cookies_fopen(filename, "r", "DEFAULT DENY\n");
   dFree(filename);

   if (!stream)
      return 2;

   /* Get all lines in the file */
   while (!feof(stream)) {
      line[0] = '\0';
      rc = fgets(line, LINE_MAXLEN, stream);
      if (!rc && ferror(stream)) {
         MSG("Error while reading rule from cookiesrc: %s\n",
             dStrerror(errno));
         break; /* bail out */
      }

      /* Remove leading and trailing whitespaces */
      dStrstrip(line);

      if (line[0] != '\0' && line[0] != '#') {
         int i = 0, j = 0;

         /* Get the domain */
         while (line[i] != '\0' && !dIsspace(line[i]))
            domain[j++] = line[i++];
         domain[j] = '\0';

         /* Skip past whitespaces */
         while (dIsspace(line[i]))
            i++;

         /* Get the rule */
         j = 0;
         while (line[i] != '\0' && !dIsspace(line[i]))
            rule[j++] = line[i++];
         rule[j] = '\0';

         if (dStrAsciiCasecmp(rule, "ACCEPT") == 0)
            cc.action = COOKIE_ACCEPT;
         else if (dStrAsciiCasecmp(rule, "ACCEPT_SESSION") == 0)
            cc.action = COOKIE_ACCEPT_SESSION;
         else if (dStrAsciiCasecmp(rule, "DENY") == 0)
            cc.action = COOKIE_DENY;
         else {
            MSG("Cookies: rule '%s' for domain '%s' is not recognised.\n",
                rule, domain);
            continue;
         }

         cc.domain = dStrdup(domain);
         if (dStrAsciiCasecmp(cc.domain, "DEFAULT") == 0) {
            /* Set the default action */
            default_action = cc.action;
            dFree(cc.domain);
         } else {
            int i;
            uint_t len = strlen(cc.domain);

            /* Insert into list such that longest rules come first. */
            a_List_add(ccontrol, num_ccontrol, num_ccontrol_max);
            for (i = num_ccontrol++;
                 i > 0 && (len > strlen(ccontrol[i-1].domain));
                 i--) {
               ccontrol[i] = ccontrol[i-1];
            }
            ccontrol[i] = cc;
         }

         if (cc.action != COOKIE_DENY)
            enabled = TRUE;
      }
   }

   fclose(stream);

   return (enabled ? 0 : 1);
}

/*
 * Check the rules for an appropriate action for this domain.
 * The rules are ordered by domain length, with longest first, so the
 * first match is the most specific.
 */
static CookieControlAction Cookies_control_check_domain(const char *domain)
{
   int i, diff;

   for (i = 0; i < num_ccontrol; i++) {
      if (ccontrol[i].domain[0] == '.') {
         diff = strlen(domain) - strlen(ccontrol[i].domain);
         if (diff >= 0) {
            if (dStrAsciiCasecmp(domain + diff, ccontrol[i].domain) != 0)
               continue;
         } else {
            continue;
         }
      } else {
         if (dStrAsciiCasecmp(domain, ccontrol[i].domain) != 0)
            continue;
      }

      /* If we got here we have a match */
      return( ccontrol[i].action );
   }

   return default_action;
}

/*
 * Set the value corresponding to the cookie string
 *
 * Return TRUE if accepted, FALSE if not.
 */
bool_t a_Cookies_set(const char *cookie_string, const DilloUrl *set_url,
                  const char *date)
{
   CookieControlAction action;

   if (disabled)
      return FALSE;

   action = Cookies_control_check_domain(URL_HOST(set_url));
   if (action == COOKIE_DENY) {
      _MSG("Cookies: denied SET for %s\n", URL_HOST_(set_url));
      return FALSE;
   }

   _MSG("Cookies.c: a_Cookies_set \n\t \"%s\" \n",cmd );
   return (Cookies_set(cookie_string, set_url, date) == 0) ? TRUE : FALSE;
}

/*
 * Return a string containing cookie data for an HTTP query.
 */
char *a_Cookies_get_query(const DilloUrl *query_url, const DilloUrl *requester)
{
   CookieControlAction action;
   bool_t strict_request;

   if (disabled)
      return dStrdup("");

   action = Cookies_control_check_domain(URL_HOST(query_url));
   if (action == COOKIE_DENY) {
      _MSG("Cookies: denied GET for %s\n", URL_HOST_(query_url));
      return dStrdup("");
   }

   strict_request = FALSE;

   if (requester == NULL) {
      /* request made by user */
   } else if (!a_Url_same_organization(query_url, requester)) {
      MSG_WARN("Cookies: not sent for request by '%s' for '%s'\n",
          URL_HOST(requester), URL_HOST(query_url));
      return dStrdup("");
   } else if (!dStrAsciiCasecmp(URL_SCHEME(query_url), URL_SCHEME(requester))) {
      strict_request = TRUE;
   }

   return Cookies_get(query_url, strict_request);
}

void a_Cookies_freeall() {
   Cookies_save_and_free();
}
#endif /* !DISABLE_COOKIES */
