/*
 * Dillo web browser
 *
 * Copyright 1999-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <errno.h>
#include <math.h> /* rint */

#include <FL/Fl.H>
#include <FL/Fl_Window.H>
#include <FL/fl_ask.H>
#include <FL/fl_draw.H>

#include "msg.h"
#include "paths.hh"
#include "referrer.h"
#include "uicmd.hh"
#include "tabs.hh"

#include "prefs.h"
#include "prefsparser.hh"
#include "keys.hh"
#include "bw.h"
#include "misc.h"
#include "history.h"

#include "IO/curl.h"
#include "IO/mime.h"
#include "capi.h"
#include "dicache.h"
#include "cookies.h"
#include "hsts.h"
#include "domain.h"
#include "auth.h"
#include "styleengine.hh"

#include "dw/fltkcore.hh"
#include "dw/widget.hh"
#include "dw/textblock.hh"
#include "dw/table.hh"

/*
 * Command line options structure
 */
typedef enum {
   DILLO_CLI_NONE          = 0,
   DILLO_CLI_XID           = 1 << 0,
   DILLO_CLI_HELP          = 1 << 1,
   DILLO_CLI_VERSION       = 1 << 2,
   DILLO_CLI_LOCAL         = 1 << 3,
   DILLO_CLI_GEOMETRY      = 1 << 4,
   DILLO_CLI_HTTP          = 1 << 5,
   DILLO_CLI_ERROR         = 1 << 15,
} OptID;

typedef struct {
   const char *shortopt;
   const char *longopt;
   int opt_argc; /* positive: mandatory, negative: optional */
   OptID id;
   const char *help;
} CLI_options;

static const CLI_options Options[] = {
   {"-g", "-geometry",    1, DILLO_CLI_GEOMETRY,
    "  -g, -geometry GEO      Set initial window position where GEO is\n"
    "                         WxH[{+-}X{+-}Y]"},
   {"-h", "--help",       0, DILLO_CLI_HELP,
    "  -h, --help             Display this help text and exit."},
   {"-i", "--http",       0, DILLO_CLI_HTTP,
    "  -i, --http             Don't upgrade all HTTP to HTTPS."},
   {"-l", "--local",      0, DILLO_CLI_LOCAL,
    "  -l, --local            Don't load images or stylesheets, send or"
    " receive\n                         cookies, or follow redirections,"
    " for these FILEs or\n                         URLs."},
   {"-v", "--version",    0, DILLO_CLI_VERSION,
    "  -v, --version          Display version info and exit."},
   {NULL, NULL, 0, DILLO_CLI_NONE, NULL}
};

#if FL_API_VERSION >= 10400
   FL_EXPORT bool fl_disable_wayland = true;
#endif

/*
 * SIGCHLD handling ----------------------------------------------------------
 */

/*
 * Avoid our children (Dpid) to become zombies. :-)
 * Notes:
 *   . We let sigaction block SIGCHLD while in the handler.
 *   . Portability is not simple. e.g.
 *       http://www.faqs.org/faqs/unix-faq/faq/part3/section-13.html
 *       man sigaction, waitpid
 */
static void raw_sigchld2(int signum)
{
   pid_t pid;
   int status;

   while (1) {
      pid = waitpid(-1, &status, WNOHANG);
      if (pid > 0) {
         if (WIFEXITED(status))        /* normal exit */
            printf("[dpid]: terminated normally (%d)\n", WEXITSTATUS(status));
         else if (WIFSIGNALED(status)) /* terminated by signal */
            printf("[dpid]: terminated by signal %d\n", WTERMSIG(status));
      } else if (pid == 0 || errno == ECHILD) {
         break;
      } else {
         if (errno == EINTR)
            continue;
         perror("waitpid");
         break;
      }
   }
   ++signum; /* compiler happiness */
}

/*
 * Establish SIGCHLD handler
 */
static void est_sigchld(void)
{
   struct sigaction sigact;
   sigset_t set;

   (void) sigemptyset(&set);
   sigact.sa_handler = raw_sigchld2; /* our custom handler */
   sigact.sa_mask = set;             /* no aditional signal blocks */
   sigact.sa_flags = SA_NOCLDSTOP;   /* ignore stop/resume states */
   if (sigaction(SIGCHLD, &sigact, NULL) == -1) {
      perror("sigaction");
      exit(1);
   }
}

//----------------------------------------------------------------------------

/*
 * Print help text generated from the options structure
 */
static void printHelp(const char *cmdname, const CLI_options *options)
{
   printf("Usage: %s [OPTION]... [--] [URL|FILE]...\n"
          "Options:\n", cmdname);
   while (options && options->help) {
      printf("%s\n", options->help);
      options++;
   }
   printf("  URL                    URL to browse.\n"
          "  FILE                   Local FILE to view.\n"
          "\n");
}

/*
 * Return the maximum number of option arguments
 */
static int numOptions(const CLI_options *options)
{
   int i, max;

   for (i = 0, max = 0; options[i].shortopt; i++)
      if (abs(options[i].opt_argc) > max)
         max = abs(options[i].opt_argc);
   return max;
}

/*
 * Get next command line option.
 */
static OptID getCmdOption(const CLI_options *options, int argc, char **argv,
                           char **opt_argv, int *idx)
{
   typedef enum { O_SEARCH, O_FOUND, O_NOTFOUND, O_DONE } State;
   OptID opt_id = DILLO_CLI_NONE;
   int i = 0;
   State state = O_SEARCH;

   if (*idx >= argc) {
      state = O_DONE;
   } else {
      state = O_NOTFOUND;
      for (i = 0; options[i].shortopt; i++) {
         if (strcmp(options[i].shortopt, argv[*idx]) == 0 ||
             strcmp(options[i].longopt, argv[*idx]) == 0) {
            state = O_FOUND;
            ++*idx;
            break;
         }
      }
   }
   if (state == O_FOUND) {
      int n_arg = options[i].opt_argc;
      opt_id = options[i].id;
      /* Find the required/optional arguments of the option */
      for (i = 0; *idx < argc && i < abs(n_arg) && argv[*idx][0] != '-'; i++)
         opt_argv[i] = argv[(*idx)++];
      opt_argv[i] = NULL;

      /* Optional arguments have opt_argc < 0 */
      if (i < n_arg) {
         fprintf(stderr, "Option %s requires %d argument%s\n",
                 argv[*idx-i-1], n_arg, (n_arg == 1) ? "" : "s");
         opt_id = DILLO_CLI_ERROR;
      }
   }
   if (state == O_NOTFOUND) {
      if (strcmp(argv[*idx], "--") == 0)
         (*idx)++;
      else if (argv[*idx][0] == '-') {
         fprintf(stderr, "Command line option \"%s\" not recognized.\n",
                 argv[*idx]);
         opt_id = DILLO_CLI_ERROR;
      }
   }
   return opt_id;
}

static bool potential_long_press = false, in_long_press = false;

static void long_press_cb(void *data)
{
   if (potential_long_press) {
      potential_long_press = false;
      in_long_press = true;

      Fl::e_number = FL_PUSH;
      Fl::e_state &= ~(FL_BUTTON1 << (Fl::e_keysym - FL_Button - 1));
      Fl::e_state |= FL_BUTTON3;
      Fl::e_keysym = FL_Button + 3;
      Fl::handle_(Fl::event(), (Fl_Window *)data);
      _MSG("synthPUSH ");
   }
}

static int our_dispatch_fn(int e, Fl_Window *w)
{
   static int long_press_x, long_press_y;

   // Pushed button 1
   if (e == FL_PUSH && Fl::e_keysym == FL_Button + 1 &&
      // RIDICULOUS WORKAROUND / BUG, in principle, perhaps.
      // 202407: On PinePhone, in Swmo, which uses Sway: You pop up a menu window (which, incidentally, is modal), but press outside the window to dismiss it rather than choose one of its items. FLTK gets the PUSH but no RELEASE. That messes up long press. I do not know whether we should blame xwayland or wayland.
       !(Fl::modal() && w != Fl::modal())) {
      _MSG("push1 ");
      potential_long_press = true;
      long_press_x = Fl::event_x();
      long_press_y = Fl::event_y();
      Fl::add_timeout(prefs.long_press_length, long_press_cb, w);
   }

   // Button 1 was pushed, but the timeout has not been triggered yet, and a button was released or the mouse has probably been moved enough to be intentional (text selection)
   if (potential_long_press &&
       (e == FL_RELEASE ||
        (e == FL_MOVE && ((abs(Fl::event_x() - long_press_x) >= prefs.drag_threshold) || (abs(Fl::event_y() - long_press_y) >= prefs.drag_threshold))))) {
      _MSG("release/move ");
      potential_long_press = false;
      Fl::remove_timeout(long_press_cb);
   }

   // Ending a long press, the release comes from button 3
   if (in_long_press && e == FL_RELEASE) {
      _MSG("release->3 ");
      in_long_press = false;
      Fl::e_state &= ~(FL_BUTTON1 << (Fl::e_keysym - FL_Button - 1));
      Fl::e_state |= FL_BUTTON3;
      Fl::e_keysym = FL_Button + 3;
   }

   return Fl::handle_(e, w);
}

static void set_text_size() {
   float h_dpi, v_dpi;

   Fl::screen_dpi(h_dpi, v_dpi);
   MSG("DPI reported as %fx%f. ", h_dpi, v_dpi);
   if (!prefs.dpi) {
      prefs.dpi = (int) rint(h_dpi);
      MSG("We’ll use the first one, %d.\n", prefs.dpi);
   } else {
      MSG("The dpi preference is set to %d.\n", prefs.dpi);
   }

   dLib_set_default_font_size(prefs.dpi * prefs.default_text_size_pt / 72.0);
}

/*
 * Set FL_NORMAL_LABEL to interpret neither symbols (@) nor shortcuts (&).
 */
static void custLabelDraw(const Fl_Label* o, int X, int Y, int W, int H,
                          Fl_Align align)
{
   const int interpret_symbols = 0;

   fl_draw_shortcut = 0;
   fl_font(o->font, o->size);
   fl_color((Fl_Color)o->color);
   fl_draw(o->value, X, Y, W, H, align, o->image, interpret_symbols);
}

static void custLabelMeasure(const Fl_Label* o, int& W, int& H)
{
   const int interpret_symbols = 0;

   fl_draw_shortcut = 0;
   fl_font(o->font, o->size);
   fl_measure(o->value, W, H, interpret_symbols);
}

/*
 * Tell the user if default/pref fonts can't be found.
 */
static void checkFont(const char *name, const char *type)
{
   if (! dw::fltk::FltkFont::fontExists(name))
      MSG_WARN("preferred %s font \"%s\" not found.\n", type, name);
}

static void checkPreferredFonts()
{
   checkFont(prefs.font_sans_serif, "sans-serif");
   checkFont(prefs.font_serif, "serif");
   checkFont(prefs.font_monospace, "monospace");
   checkFont(prefs.font_cursive, "cursive");
   checkFont(prefs.font_fantasy, "fantasy");
}

/*
 * Set UI color. 'color' is an 0xrrggbb value, whereas 'default_val' is a fltk
 * color (index 0-0xFF, or 0xrrggbb00).
 */
static void setUIColorWdef(Fl_Color idx, int32_t color, Fl_Color default_val)
{
   if (color != -1)
      Fl::set_color(idx, color << 8);
   else if (default_val != 0xFFFFFFFF)
      Fl::set_color(idx, default_val);
}

static void setColors()
{
   /* The main background is a special case because Fl::background() will
    * set the "gray ramp", which is a set of lighter and darker colors based
    * on the main background and used for box edges and such.
    */
   if (prefs.ui_main_bg_color != -1) {
      uchar r = prefs.ui_main_bg_color >> 16,
            g = prefs.ui_main_bg_color >> 8 & 0xff,
            b = prefs.ui_main_bg_color & 0xff;

      Fl::background(r, g, b);
   }

   setUIColorWdef(FL_BACKGROUND2_COLOR, prefs.ui_text_bg_color, 0xFFFFFFFF);
   setUIColorWdef(FL_FOREGROUND_COLOR, prefs.ui_fg_color, 0xFFFFFFFF);
   setUIColorWdef(FL_SELECTION_COLOR, prefs.ui_selection_color,
                  fl_contrast(FL_SELECTION_COLOR, FL_BACKGROUND2_COLOR));
   setUIColorWdef(PREFS_UI_BUTTON_HIGHLIGHT_COLOR,
                  prefs.ui_button_highlight_color,
                  fl_lighter(FL_BACKGROUND_COLOR));
   setUIColorWdef(PREFS_UI_INSECURE_COLOR, prefs.ui_insecure_color,
                  Fl::get_color(FL_RED));
}

/*
 * Given a command line argument, build a DilloUrl for it.
 */
static DilloUrl *makeStartUrl(char *str, bool local)
{
   char *url_str, *p;
   DilloUrl *start_url;
   struct stat not_used_for_anything;

   /* Relative path to a local file? */
   p = (*str == '/') ? dStrdup(str) :
                       dStrconcat(Paths::getOldWorkingDir(), "/", str, NULL);

   if (lstat(p, &not_used_for_anything) == -1 && errno == ENOENT) {
      /* Not a file, filter URL string.
       * (This test used to use access(), which fails to do the right thing for a dangling symlink. There’s faccessat(), but that only arrived in POSIX.1-2008.)
       */
      url_str = a_Url_string_strip_delimiters(str);
      start_url = a_Url_new(url_str, NULL);
   } else {
      /* absolute path may have non-URL characters */
      url_str = a_Misc_escape_chars(p, "% #");

      start_url = a_Url_new(url_str + 1, "file:///");
   }
   dFree(p);
   dFree(url_str);

   if (local)
      a_Url_set_flags(start_url, URL_FLAGS(start_url) | URL_SpamSafe);

   return start_url;
}

/*
 * MAIN
 */
int main(int argc, char **argv)
{
   uint_t opt_id;
   uint_t options_got = 0;
   int idx = 1;
   int xpos = PREFS_GEOMETRY_DEFAULT_XPOS, ypos = PREFS_GEOMETRY_DEFAULT_YPOS,
       width = PREFS_GEOMETRY_DEFAULT_WIDTH,
       height = PREFS_GEOMETRY_DEFAULT_HEIGHT;
   char **opt_argv;
   FILE *fp;

   srand((uint_t)(time(0) ^ getpid()));

   // Some OSes exit dillo without this (not GNU/Linux).
   signal(SIGPIPE, SIG_IGN);
   // Establish our custom SIGCHLD handler
   est_sigchld();

   /* Handle command line options */
   opt_argv = dNew0(char*, numOptions(Options) + 1);
   while ((opt_id = getCmdOption(Options, argc, argv, opt_argv, &idx))) {
      options_got |= opt_id;
      switch (opt_id) {
      case DILLO_CLI_LOCAL:
      case DILLO_CLI_HTTP:
         break;
      case DILLO_CLI_GEOMETRY:
         if (!a_Misc_parse_geometry(opt_argv[0],&xpos,&ypos,&width,&height)){
            fprintf(stderr, "geometry argument \"%s\" not valid. Must be of "
                            "the form WxH[{+-}X{+-}Y].\n", opt_argv[0]);
            return 2;
         }
         break;
      case DILLO_CLI_VERSION:
         puts("Dillo version " VERSION);
         return 0;
      case DILLO_CLI_HELP:
         printHelp(argv[0], Options);
         return 0;
      default:
         printHelp(argv[0], Options);
         return 2;
      }
   }
   dFree(opt_argv);

   // set the default values for the preferences
   a_Prefs_init();

   // create ~/.dillo if not present
   Paths::init();

   // initialize default key bindings
   Keys::init();

   // parse dillorc
   if ((fp = Paths::getPrefsFP(PATHS_RC_PREFS))) {
      PrefsParser::parse(fp);
   }
   if (strchr(prefs.no_proxy, ' ')) {
      char *s = prefs.no_proxy;

      MSG_WARN("prefs.no_proxy hosts should be separated by commas now, as that’s what libcurl likes.\n");

      while (*s) {
         if (*s == ' ')
            *s = ',';
         s++;
      }
   }

   /* Convenient for us for save_dir to end in / */
   if (*prefs.save_dir && prefs.save_dir[strlen(prefs.save_dir) - 1] != '/') {
      char *s = dStrconcat(prefs.save_dir, "/", NULL);
      dFree(prefs.save_dir);
      prefs.save_dir = s;
   }

   /* command line options override preferences */
   if (options_got & DILLO_CLI_GEOMETRY) {
       prefs.width = width;
       prefs.height = height;
       prefs.xpos = xpos;
       prefs.ypos = ypos;
   }

   if (options_got & DILLO_CLI_HTTP) {
      prefs.http_insecure = TRUE;
   }

   set_text_size();

   // parse keysrc
   if ((fp = Paths::getPrefsFP(PATHS_RC_KEYS))) {
      Keys::parse(fp);
   }
   // parse domainrc
   if ((fp = Paths::getPrefsFP(PATHS_RC_DOMAIN))) {
      a_Domain_parse(fp);
      fclose(fp);
   }
   dLib_show_messages(prefs.show_msg);

   // initialize internal modules
   a_Curl_init();
   a_Http_init();
   a_Mime_init();
   a_Cache_init();
   a_FetchReq_init();
   a_Dicache_init();
   a_Bw_init();
   a_Cookies_init();
   if (prefs.http_insecure) {
      a_Hsts_init(Paths::getPrefsFP(PATHS_HSTS_PRELOAD));
   }
   a_Url_init(Paths::getPrefsFP(PATHS_PUBLIC_SUFFIX));
   a_Auth_init();
   a_Misc_init();
   StyleEngine::init();

   dw::core::Widget::setAdjustMinWidth (prefs.adjust_min_width);
   dw::Table::setAdjustTableMinWidth (prefs.adjust_table_min_width);
   dw::Textblock::setPenaltyHyphen (prefs.penalty_hyphen);
   dw::Textblock::setPenaltyHyphen2 (prefs.penalty_hyphen_2);
   dw::Textblock::setPenaltyEmDashLeft (prefs.penalty_em_dash_left);
   dw::Textblock::setPenaltyEmDashRight (prefs.penalty_em_dash_right);
   dw::Textblock::setPenaltyEmDashRight2 (prefs.penalty_em_dash_right_2);
   dw::Textblock::setStretchabilityFactor (prefs.stretchability_factor);

   MSG("GUI toolkit: FLTK %d.%d.%d\n", FL_MAJOR_VERSION, FL_MINOR_VERSION,
       FL_PATCH_VERSION);

   if (prefs.long_press_length > 0.0)
      Fl::event_dispatch(our_dispatch_fn);

   // Sets WM_CLASS hint on X11
   Fl_Window::default_xclass("dillo");

   Fl::scheme(prefs.theme);

#if FL_API_VERSION >= 10400
   // Don’t let FLTK’s handler interpret <ctrl>[+-0] to scale screens
   Fl::keyboard_screen_scaling(0);
#endif

   // Disable drag and drop as it crashes on MacOSX
   Fl::dnd_text_ops(0);

   setColors();

   if (!prefs.show_ui_tooltip) {
      Fl::option(Fl::OPTION_SHOW_TOOLTIPS, false);
   }

   // Disable '@' and '&' interpretation in normal labels.
   Fl::set_labeltype(FL_NORMAL_LABEL, custLabelDraw, custLabelMeasure);

   checkPreferredFonts();

   /* use preferred font for UI */
   Fl_Font defaultFont = dw::fltk::FltkFont::get (prefs.font_sans_serif, 0);
   Fl::set_font(FL_HELVETICA, defaultFont); // this seems to be the
                                            // only way to set the
                                            // default font in fltk1.3

   fl_message_title_default("Dillo: Message");

   /* Proxy authentication */
   if (prefs.http_proxyuser) {
      char *passwd = a_UIcmd_get_passwd(prefs.http_proxyuser);
      if (passwd) {
         a_Http_set_proxy_passwd(passwd);
         dStrshred(passwd);
         dFree(passwd);
      } else {
         MSG_WARN("Not using proxy authentication.\n");
      }
   }

   /* BUG(?) On X11, if we needed to make a window above to get the proxy password, the UI that we make in a moment will get the focus box in my window manager, but keyboard input still goes into the terminal I started Dillo from (until I click the UI). This could just be my WM’s fault. */

   // Create a new UI/bw pair
   BrowserWindow *bw = a_UIcmd_browser_window_new(prefs.xpos, prefs.ypos, 0, 0);

   /* We need this so that fl_text_extents() in dw/fltkplatform.cc can
    * work when FLTK is configured without XFT and Dillo is opening
    * immediately-available URLs from the cmdline (e.g. about: URLs).
    */
   ((Fl_Widget *)bw->ui)->window()->make_current();

   /* Open URLs/files */
   const bool local = options_got & DILLO_CLI_LOCAL;
   const DilloUrl *referrer = NULL;
   FetchReq req = {0, NULL, referrer, REFERRER_POLICY_NOTSET, HTTP_Sec_Fetch_Dest_document, TRUE};

   if (idx == argc) {
      /* No URLs/files on cmdline. Send startup screen */
      if (strcmp(prefs.start_page, "about:blank") == 0)
         a_UIcmd_open_url(bw, UICMD_LOC_HERE, &req); // NULL URL focuses location
      else {
         DilloUrl *url = a_Url_new(prefs.start_page, NULL);
         req.url = url;
         a_UIcmd_open_url(bw, UICMD_LOC_HERE, &req);
         a_Url_free(url);
         a_UIcmd_set_location_text(bw, prefs.start_page);
      }
   } else {
      for (int i = idx; i < argc; i++) {
         DilloUrl *start_url = makeStartUrl(argv[i], local);
         req.url = start_url;

         if (i > idx) {
            UIcmd_location loc = prefs.middle_click_opens_new_tab ? UICMD_LOC_NEW_TAB : UICMD_LOC_NEW_WINDOW;
            a_UIcmd_open_url(bw, loc, &req);
         } else {
            a_UIcmd_open_url(bw, UICMD_LOC_HERE, &req);
            a_UIcmd_set_location_text(bw, URL_STR(req.url));
         }
         a_Url_free(start_url);
      }
   }

   Fl::run();

   /*
    * Memory deallocating routines
    * (This can be left to the OS, but we'll do it, with a view to test
    *  and fix our memory management)
    */
   a_Domain_freeall();
   a_Cookies_freeall();
   if (prefs.http_insecure) {
      a_Hsts_freeall();
   }
   a_Cache_freeall();
   a_FetchReq_freeall();
   a_Dicache_freeall();
   a_Curl_freeall();
   a_Http_freeall();
   a_History_freeall();
   a_Prefs_freeall();
   a_Misc_freeall();
   Keys::free();
   Paths::free();
   dw::core::freeall();
   dw::fltk::freeall();
   return 0;
}
