/*
 * File: findbar.cc
 *
 * Copyright (C) 2005-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

#include <FL/Fl.H>
#include <FL/Fl_Window.H>
#include <FL/fl_draw.H>
#include "findbar.hh"

#include "msg.h"
#include "uicmd.hh"
#include "bw.h"

const int NOWRAP = 0;

/*
 * Local sub class
 * (Used to handle escape in the findbar, may also avoid some shortcuts).
 */
class MyInput : public MenuInput {
public:
   MyInput (int x, int y, int w, int h, const char* l=0) :
      MenuInput(x,y,w,h,l) {};
   int handle(int e);
};

int MyInput::handle(int e)
{
   _MSG("findbar MyInput::handle()\n");
   int ret = 1, k = Fl::event_key();
   unsigned modifier = Fl::event_state() & (FL_SHIFT| FL_CTRL| FL_ALT|FL_META);

   if (e == FL_KEYBOARD) {
      if (k == FL_Page_Down || k == FL_Page_Up || k == FL_Up || k == FL_Down || k == FL_Home || k == FL_End) {
         // Let them through for key commands and viewport motion.
         return 0;
      }
      if (modifier == FL_SHIFT) {
         if (k == FL_Left || k == FL_Right) {
            // Let these keys get to the UI
            return 0;
         }
      } else if (modifier == FL_CTRL) {
         if (k == 'h' || k == 'i' || k == 'j' || k == 'l' || k == 'm') {
            // Fl_Input wants to use ^H as backspace, and also "insert a few
            // selected control characters literally", but this gets in the way
            // of key commands.
            return 0;
         }
      } else if (k == FL_Escape && modifier == 0) {
         // Avoid clearing the text with Esc, just hide the findbar.
         return 0;
      }
   }

   if (ret)
      ret = MenuInput::handle(e);
   return ret;
}

/*
 * We are following the pattern of the Location input in the UI here.
 */
void Findbar::check_input_fit()
{
   int textwidth;

   fl_font(i->textfont(), i->textsize());
   textwidth = i->textwidth();

   if (row2->contains(i)) {
      if (textwidth > i->w()) {
         row1->add(i);
         row1->resizable(i);
         row2->resizable(row2);
         row2->rearrange();
         row1->rearrange();
         size(w(), 2 * h());
         rearrange();
         a_UIcmd_rearrange_UI(a_UIcmd_get_bw_by_widget(this));
      }
   } else {
      if (textwidth < row2->resizable_width()) {
         row1->resizable(row1);    // necessary?
         row2->insert(*i, hide_btn);
         row2->resizable(i);
         row2->rearrange();
         size(w(), original_height);
         rearrange();
         a_UIcmd_rearrange_UI(a_UIcmd_get_bw_by_widget(this));
      }
   }
}

void Findbar::input_cb(Fl_Widget *, void *data)
{
   ((Findbar *)data)->check_input_fit();
}

/*
 * Find next occurrence of input key
 */
void Findbar::search_cb(Fl_Widget *, void *vfb)
{
   Findbar *fb = (Findbar *)vfb;
   const char *key = fb->i->value();
   bool case_sens = fb->case_btn->value();

   if (key[0] != '\0')
      a_UIcmd_findtext_search(a_UIcmd_get_bw_by_widget(fb),
                              key, case_sens, false);
}

/*
 * Find previous occurrence of input key
 */
void Findbar::searchBackwards_cb(Fl_Widget *, void *vfb)
{
   Findbar *fb = (Findbar *)vfb;
   const char *key = fb->i->value();
   bool case_sens = fb->case_btn->value();

   if (key[0] != '\0') {
      a_UIcmd_findtext_search(a_UIcmd_get_bw_by_widget(fb),
                              key, case_sens, true);
   }
}

/*
 * Hide the search bar
 */
void Findbar::hide_cb(Fl_Widget *, void *vfb)
{
   a_UIcmd_findbar_toggle(a_UIcmd_get_bw_by_widget(vfb), 0);
}

/*
 * Construct text search bar
 */
Findbar::Findbar(int width, int height, int min_button_w_px) :
   CustGroupVertical(0, 0, width, height), original_height(height)
{
   int w = NOWRAP, h;
   int button_width = 70;
   int gap = 0;
   int border = 1;
   int input_width = width - (2 * border + 4 * (button_width + gap));
   int x = 0;

   Fl_Group::current(0);

   height -= 2 * border;

   box(FL_FLAT_BOX);

    row1 = new CustGroupHorizontal(0, 0, width, height);
    add(row1);
    resizable(row1);
    row2 = new CustGroupHorizontal(0, 0, width, height);
    add(row2);

    hide_btn = new CustButton(x, border, 16, height, "×");
    hide_btn->box(FL_FLAT_BOX);
    hide_btn->callback(hide_cb, this);
    hide_btn->clear_visible_focus();
    hide_btn->set_tooltip("Hide");
    hide_btn->measure_label(w, h);
    w = MAX(w, min_button_w_px);
    hide_btn->size(w, hide_btn->h());
    x += w;
   row2->add(hide_btn);

    i = new MyInput(x, border, input_width, height);
    i->box(FL_FLAT_BOX);
    x += input_width + gap;
    i->when(FL_WHEN_CHANGED);
    i->callback(input_cb);
    i->user_data(this);
   row2->add(i);
   row2->resizable(i);

    w = NOWRAP;
    case_btn = new Fl_Toggle_Button(x, border, 2*button_width, height,
                              "Case-sensitive");
    case_btn->clear_visible_focus();
    case_btn->box(FL_FLAT_BOX);
    case_btn->down_box(FL_DOWN_BOX);
    case_btn->measure_label(w, h);
    w = MAX(w, min_button_w_px);
    case_btn->size(w, case_btn->h());
    x += 2 * button_width + gap;
    case_btn->position(width - border - w, border);

    w = NOWRAP;
    prev_btn= new CustButton(x, border, button_width, height, "Prev");
    prev_btn->box(FL_FLAT_BOX);
    prev_btn->shortcut(FL_SHIFT+FL_Enter);
    prev_btn->callback(searchBackwards_cb, this);
    prev_btn->clear_visible_focus();
    prev_btn->set_tooltip("Find previous occurrence of the search phrase\n"
                          "shortcut: Shift+Enter");
    prev_btn->measure_label(w, h);
    w = MAX(w, min_button_w_px);
    prev_btn->size(w, prev_btn->h());
    x += button_width + gap;
    prev_btn->position(case_btn->x() - w, border);

    w = NOWRAP;
    next_btn = new CustButton(x, border, button_width, height, "Next");
    next_btn->box(FL_FLAT_BOX);
    next_btn->shortcut(FL_Enter);
    next_btn->callback(search_cb, this);
    next_btn->clear_visible_focus();
    next_btn->set_tooltip("Find next occurrence of the search phrase\n"
                          "shortcut: Enter");
    next_btn->measure_label(w, h);
    w = MAX(w, min_button_w_px);
    next_btn->size(w, next_btn->h());
    x += w;
    next_btn->position(prev_btn->x() - w, border);

   row2->add(next_btn);
   row2->add(prev_btn);
   row2->add(case_btn);

   i->size(next_btn->x() - i->x(), i->h());

   row2->rearrange();
   rearrange();
   hide();
}

/*
 * Handle events. Used to catch FL_Escape events.
 */
int Findbar::handle(int event)
{
   int k = Fl::event_key();
   unsigned modifier = Fl::event_state() & (FL_SHIFT| FL_CTRL| FL_ALT|FL_META);

   if (event == FL_KEYBOARD && modifier == 0) {
      if (k == FL_Escape) {
         /* let the UI handle it */
         return 0;
      } else if (k == FL_Down || k == FL_Up || k == FL_Page_Down || k == FL_Page_Up || k == FL_Tab) {
         a_UIcmd_findbar_toggle(a_UIcmd_get_bw_by_widget(this), 0);
         return 0;
      }
   }

   return CustGroupVertical::handle(event);
}

/*
 * Show the findbar and focus the input field
 */
void Findbar::show()
{
   // It takes more than just calling show() to do the trick
   CustGroupVertical::show();

   check_input_fit();

   /* select text even if already focused */
   i->take_focus();
   i->insert_position(i->size(), 0);
}

void Findbar::resize(int x, int y, int w, int h)
{
   CustGroupVertical::resize(x,y,w,h);
   if (w && h)
      check_input_fit();
}
