/*
 * File: misc.c
 *
 * Copyright (C) 2000-2007 Jorge Arellano Cid <jcid@dillo.org>,
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <magic.h>

#include <dirent.h>
#include <sys/stat.h>

#include "utf8.hh"
#include "msg.h"
#include "misc.h"

static magic_t MAGIC = NULL;

int a_Misc_init()
{
   MAGIC = magic_open(MAGIC_MIME_TYPE);

   if (!MAGIC)
      MSG_ERR("magic_open failed.\n");
   else if (magic_load(MAGIC, NULL)) {
      MSG_ERR("magic_load failed.\n");
      magic_close(MAGIC);
      MAGIC = NULL;
   }
   return 0;
}

/*
 * Escape characters as %XX sequences.
 * Return value: New string.
 */
char *a_Misc_escape_chars(const char *str, const char *esc_set)
{
   static const char *const hex = "0123456789ABCDEF";
   char *p = NULL;
   Dstr *dstr;
   int i;

   dstr = dStr_sized_new(64);
   for (i = 0; str[i]; ++i) {
      if (str[i] <= 0x1F || str[i] == 0x7F || strchr(esc_set, str[i])) {
         dStr_append_c(dstr, '%');
         dStr_append_c(dstr, hex[(str[i] >> 4) & 15]);
         dStr_append_c(dstr, hex[str[i] & 15]);
      } else {
         dStr_append_c(dstr, str[i]);
      }
   }
   p = dstr->str;
   dStr_free(dstr, FALSE);

   return p;
}

#define TAB_SIZE 8
/*
 * Takes a string and converts any tabs to spaces.
 */
int
a_Misc_expand_tabs(char **start, char *end, char *buf, int buflen)
{
   int j, pos = 0, written = 0, old_pos, char_len;
   uint_t code;
   static const int combining_char_space = 32;

   while (*start < end && written < buflen - TAB_SIZE - combining_char_space) {
      code = a_Utf8_decode(*start, end, &char_len);

      if (code == '\t') {
         /* Fill with whitespaces until the next tab. */
         old_pos = pos;
         pos += TAB_SIZE - (pos % TAB_SIZE);
         for (j = old_pos; j < pos; j++)
            buf[written++] = ' ';
      } else {
         assert(char_len <= 4);
         for (j = 0; j < char_len; j++)
            buf[written++] = (*start)[j];
         pos++;
      }

      *start += char_len;
   }

   /* If following chars are combining chars (e.g. accents) add them to the
    * buffer. We have reserved combining_char_space bytes for this.
    * If there should be more combining chars, we split nevertheless.
    */
   while (*start < end && written < buflen - 4) {
      code = a_Utf8_decode(*start, end, &char_len);

      if (! a_Utf8_combining_char(code))
         break;

      assert(char_len <= 4);
      for (j = 0; j < char_len; j++)
         buf[written++] = (*start)[j];

      *start += char_len;
   }

   return written;
}

/*
 * Detects 'Content-Type' from a data stream sample.
 *
 * 'Data' is a pointer to the first bytes of the raw data.
 *
 * Return value: (0 on success, 1 on doubt, 2 on lack of data).
 */
int a_Misc_get_content_type_from_data(void *Data, size_t Size, const char **PT)
{
   if (!Size)
      return 2;

   if (MAGIC) {
      const char *mimetype = magic_buffer(MAGIC, Data, Size);

      if (mimetype) {
         if (!strcmp(mimetype, "text/xml"))
            mimetype = "text/html";
         *PT = mimetype;
         return 0;
      }
   }
   return 2;
}

/*
 * Parse Content-Type string, e.g., "text/html; charset=utf-8".
 * Content-Type is defined in RFC 2045 section 5.1.
 */
void a_Misc_parse_content_type(const char *type, char **major, char **minor,
                               char **charset)
{
   static const char tspecials_space[] = "()<>@,;:\\\"/[]?= ";
   const char *str, *s;

   if (major)
      *major = NULL;
   if (minor)
      *minor = NULL;
   if (charset)
      *charset = NULL;
   if (!(str = type))
      return;

   for (s = str; *s && isascii((uchar_t)*s) && !iscntrl((uchar_t)*s) &&
        !strchr(tspecials_space, *s); s++) ;
   if (major)
      *major = dStrndup(str, s - str);

   if (*s == '/') {
      for (str = ++s; *s && isascii((uchar_t)*s) && !iscntrl((uchar_t)*s) &&
           !strchr(tspecials_space, *s); s++) ;
      if (minor)
         *minor = dStrndup(str, s - str);
   }
   if (charset && *s &&
       (dStrnAsciiCasecmp(type, "text/", 5) == 0 ||
        dStrnAsciiCasecmp(type, "application/xhtml+xml", 21) == 0)) {
      /* "charset" parameter defined for text media type in RFC 2046,
       * application/xhtml+xml in RFC 3236.
       *
       * Note that RFC 3023 lists some main xml media types and provides
       * the convention of using the "+xml" minor type suffix for other
       * xml types, so it would be reasonable to check for that suffix if
       * we have need to care about various xml types someday.
       */
      const char terminators[] = " ;\t";
      const char key[] = "charset";

      if ((s = dStriAsciiStr(str, key)) &&
          (s == str || strchr(terminators, s[-1]))) {
         s += sizeof(key) - 1;
         for ( ; *s == ' ' || *s == '\t'; ++s);
         if (*s == '=') {
            size_t len;
            for (++s; *s == ' ' || *s == '\t'; ++s);
            if ((len = strcspn(s, terminators))) {
               if (*s == '"' && s[len-1] == '"' && len > 1) {
                 /* quoted string */
                 s++;
                 len -= 2;
               }
               *charset = dStrndup(s, len);
            }
         }
      }
   }
}

/*
 * Parse a geometry string.
 */
int a_Misc_parse_geometry(char *str, int *x, int *y, int *w, int *h)
{
   char *p, *t1, *t2;
   int n1, n2;
   int ret = 0;

   if ((p = strchr(str, 'x')) || (p = strchr(str, 'X'))) {
      n1 = strtol(str, &t1, 10);
      n2 = strtol(++p, &t2, 10);
      if (t1 != str && t2 != p) {
         *w = n1;
         *h = n2;
         ret = 1;
         /* parse x,y now */
         p = t2;
         n1 = strtol(p, &t1, 10);
         n2 = strtol(t1, &t2, 10);
         if (t1 != p && t2 != t1) {
            *x = n1;
            *y = n2;
         }
      }
   }
   _MSG("geom: w,h,x,y = (%d,%d,%d,%d)\n", *w, *h, *x, *y);
   return ret;
}

/*
 * Parse dillorc's search_url string ("[<label> ]<url>")
 * Return value: -1 on error, 0 on success (and label and urlstr pointers)
 */
int a_Misc_parse_search_url(char *source, char **label, char **urlstr)
{
   static char buf[32];
   char *p, *q;
   int ret = -1;

   if ((p = strrchr(source, ' '))) {
      /* label and url pair */
      strncpy(buf,source,MIN(p-source,31));
      buf[MIN(p-source,31)] = 0;
      source = p+1;
      if ((p = strchr(source, '/')) && p[1] && (q = strchr(p+2,'/'))) {
         *urlstr = source;
         ret = 0;
      }
   } else {
      /* url only, make a custom label */
      if ((p = strchr(source, '/')) && p[1] && (q = strchr(p+2,'/'))) {
         strncpy(buf,p+2,MIN(q-p-2,31));
         buf[MIN(q-p-2,31)] = 0;
         *urlstr = source;
         ret = 0;
      }
   }
   *label = buf;
   if (ret == -1)
      MSG_WARN("Invalid search_url: \"%s\"\n", source);
   return ret;
}

/*
 * Encodes string using base64 encoding.
 * Return value: new string or NULL if input string is empty.
 */
char *a_Misc_encode_base64(const char *in)
{
   static const char *const base64_hex = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                                         "abcdefghijklmnopqrstuvwxyz"
                                         "0123456789+/";
   char *out = NULL;
   int len, i = 0;

   if (in == NULL) return NULL;
   len = strlen(in);

   out = (char *)dMalloc((len + 2) / 3 * 4 + 1);

   for (; len >= 3; len -= 3) {
      out[i++] = base64_hex[in[0] >> 2];
      out[i++] = base64_hex[((in[0]<<4) & 0x30) | (in[1]>>4)];
      out[i++] = base64_hex[((in[1]<<2) & 0x3c) | (in[2]>>6)];
      out[i++] = base64_hex[in[2] & 0x3f];
      in += 3;
   }

   if (len > 0) {
      unsigned char fragment;
      out[i++] = base64_hex[in[0] >> 2];
      fragment = (in[0] << 4) & 0x30;
      if (len > 1) fragment |= in[1] >> 4;
      out[i++] = base64_hex[fragment];
      out[i++] = (len < 2) ? '=' : base64_hex[(in[1] << 2) & 0x3c];
      out[i++] = '=';
   }
   out[i] = '\0';
   return out;
}

/*
 * Given a hex octet (e.g., e3, 2F, 20), return the corresponding
 * character if the octet is valid, and -1 otherwise
 */
int a_Misc_decode_hex_octet(const char *s)
{
   int hex_value;
   char *tail, hex[3];

   if (s && (hex[0] = s[0]) && (hex[1] = s[1])) {
      hex[2] = 0;
      hex_value = strtol(hex, &tail, 16);
      if (tail - hex == 2)
        return hex_value;
   }
   return -1;
}

char *a_Misc_make_save_filename(const char *path, const char *query)
{
   size_t MaxLen = 64;
   const char *dir = prefs.save_dir, *path2;
   char *name, *free1, *free2, *n1, *n2;

   free1 = free2 = NULL;

   if (!path)
      path = "";
   if (!query)
      query = "";

   /* get the last component of the path */
   path2 = strrchr(path, '/');
   path = path2 ? path2 + 1 : path;

   /* truncate the path if necessary */
   if (strlen(path) > MaxLen) {
      path = free1 = dStrndup(path, MaxLen);
   }

   /* is there a query? */
   if (*query) {
      /* truncate the query if necessary */
      if (strlen(query) > MaxLen) {
         query = free2 = dStrndup(query, MaxLen);
      }
      name = dStrconcat(dir, path, "?", query, NULL);
   } else {
      name = dStrconcat(dir, path, NULL);
   }

   dFree(free1);
   dFree(free2);

   /* Replace %20 and ' ' with '_' */
   for (n1 = n2 = name; *n1; n1++, n2++) {
      *n2 =
         (n1[0] == ' ')
         ? '_' :
         (n1[0] == '%' && n1[1] == '2' && n1[2] == '0')
         ? (n1 += 2, '_') :
         n1[0];
   }
   *n2 = 0;
   return name;
}

/*
 * Load a local file into a dStr.
 * Return value: dStr on success, NULL on error.
 * TODO: a filesize threshold may be implemented.
 */
Dstr *a_Misc_file2dstr(const char *filename)
{
   FILE *F_in;
   int n;
   char buf[4096];
   Dstr *dstr = NULL;

   if ((F_in = fopen(filename, "r"))) {
      dstr = dStr_sized_new(4096);
      while ((n = fread (buf, 1, 4096, F_in)) > 0) {
         dStr_append_l(dstr, buf, n);
      }
      fclose(F_in);
   }
   return dstr;
}

static int common_prefix_len(const char *s1, const char *s2)
{
   int len = 0;

   if (s1 && s2)
      while (*s1 && (*s1++ == *s2++))
        len++;

   return len;
}

/*
 * If we were able to add anything, return an extended path string. Otherwise, NULL.
 */
char *a_Misc_tabcomplete(const char *path)
{
   const char *last_slash = strrchr(path, '/');
   const int dirlen = last_slash ? last_slash - path + 1 : 0, // including slash
             afterlen = strlen(path) - dirlen;

   _MSG("Tabcomplete: path is \"%s\"\n", path);
   if (dirlen) {
      DIR *dir;
      Dstr *filename = dStr_new("");

      dStr_append_l(filename, path, dirlen);
      dir = opendir(filename->str);

      if (dir) {
         struct dirent *ent;
         struct stat ss;
         int matches = 0;

         while ((ent = readdir(dir))) {
            if (!strcmp(ent->d_name, ".") || !strcmp(ent->d_name, ".."))
               continue;
            if (!strncmp(path + dirlen, ent->d_name, afterlen)) {

               if (!matches++) {
                  dStr_append(filename, ent->d_name);
                  _MSG("First match \"%s\" gives \"%s\"\n", ent->d_name, filename->str);
               } else {
                  int common = common_prefix_len(filename->str+dirlen, ent->d_name);
                  dStr_truncate(filename, dirlen+common);
                  _MSG("Match \"%s\" leaves \"%s\"\n", ent->d_name, filename->str);
               }
            }
         }
         if (matches == 1 && stat(filename->str, &ss) == 0 && S_ISDIR(ss.st_mode)) {
            // Let’s not assume d_type will be available in dirent
            dStr_append_c(filename, '/');
         }
         closedir(dir);
      }
      if ((size_t)filename->len > strlen(path)) {
         char *str = filename->str;
         dStr_free(filename, 0);
         return str;
      } else {
         dStr_free(filename, 1);
      }
   }
   return NULL;
}

void a_Misc_freeall()
{
   magic_close(MAGIC);
}
