/*
 * File: nav.c
 *
 * Copyright (C) 2000-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

/* Support for a navigation stack */

#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include "msg.h"
#include "nav.h"
#include "history.h"
#include "client.hh"
#include "uicmd.hh"
#include "capi.h"
#include "timeout.hh"

/*
 * For back and forward navigation, each bw keeps an url index,
 * and its scroll position.
 */
typedef struct {
   int url_idx;
   int posx, posy;
} nav_stack_item;



/*
 * Free memory used by this module
 * TODO: this may be removed or called by a_Bw_free().
  *      Currently is not called from anywhere.
 */
void a_Nav_free(BrowserWindow *bw)
{
   a_Nav_cancel_expect(bw);
   dFree(bw->nav_stack);
}


/* Navigation stack methods ------------------------------------------------ */

/*
 * Return current nav_stack index [0 based; -1 = empty]
 */
int a_Nav_stack_idx(BrowserWindow *bw)
{
   return bw->nav_stack_idx;
}

/*
 * Return the url index of i-th element in the stack. [-1 = Error]
 */
int a_Nav_get_uidx(BrowserWindow *bw, int i)
{
   nav_stack_item *nsi = dList_nth_data (bw->nav_stack, i);
   return (nsi) ? nsi->url_idx : -1;
}

/*
 * Return the url index of the current element in the stack.
 */
int a_Nav_get_curr_uidx(BrowserWindow *bw)
{
   nav_stack_item *nsi;

   nsi = dList_nth_data (bw->nav_stack, a_Nav_stack_idx(bw));
   return (nsi) ? nsi->url_idx : -1;
}

/*
 * Move the nav_stack pointer
 */
static void Nav_stack_move_ptr(BrowserWindow *bw, int offset)
{
   int idx;

   dReturn_if_fail (bw != NULL);
   if (offset != 0) {
      idx = bw->nav_stack_idx + offset;
      dReturn_if_fail (idx >= 0 && idx < a_Nav_stack_size(bw));
      bw->nav_stack_idx = idx;
   }
}

/*
 * Return size of nav_stack [1 based]
 */
int a_Nav_stack_size(BrowserWindow *bw)
{
   return dList_length(bw->nav_stack);
}

/*
 * Truncate the navigation stack including 'pos' and upwards.
 */
static void Nav_stack_truncate(BrowserWindow *bw, int pos)
{
   void *data;

   dReturn_if_fail (bw != NULL);

   while (pos < dList_length(bw->nav_stack)) {
      data = dList_nth_data(bw->nav_stack, pos);
      dList_remove_fast (bw->nav_stack, data);
      dFree(data);
   }
}

/*
 * Insert a nav_stack_item into the stack at a given position.
 */
static void Nav_stack_append(BrowserWindow *bw, int url_idx)
{
   nav_stack_item *nsi;

   dReturn_if_fail (bw != NULL);

   /* Insert the new element */
   nsi = dNew(nav_stack_item, 1);
   nsi->url_idx = url_idx;
   nsi->posx = 0;
   nsi->posy = 0;
   dList_append (bw->nav_stack, nsi);
}

/*
 * Get the scrolling position of the current page.
 */
static void Nav_get_scroll_pos(BrowserWindow *bw, int *posx, int *posy)
{
   nav_stack_item *nsi;

   if ((nsi = dList_nth_data (bw->nav_stack, a_Nav_stack_idx(bw)))) {
      *posx = nsi->posx;
      *posy = nsi->posy;
   } else {
      *posx = *posy = 0;
   }
}

/*
 * Save the scrolling position of the current page.
 */
static void Nav_save_scroll_pos(BrowserWindow *bw, int idx, int posx, int posy)
{
   nav_stack_item *nsi;

   if ((nsi = dList_nth_data (bw->nav_stack, idx))) {
      nsi->posx = posx;
      nsi->posy = posy;
   }
}

/*
 * Remove equal adjacent URLs at the top of the stack.
 * (It may happen with redirections)
 */
static void Nav_stack_clean(BrowserWindow *bw)
{
   int i;

   dReturn_if_fail (bw != NULL);

   if ((i = a_Nav_stack_size(bw)) >= 2 &&
       a_Nav_get_uidx(bw,i - 2) == a_Nav_get_uidx(bw,i - 1)) {
      void *data = dList_nth_data (bw->nav_stack, i - 1);
      dList_remove_fast (bw->nav_stack, data);
      dFree(data);
      if (bw->nav_stack_idx >= a_Nav_stack_size(bw))
         bw->nav_stack_idx = a_Nav_stack_size(bw) - 1;
   }
}


/* General methods --------------------------------------------------------- */

/*
 * Create a ClientData structure for 'url' and ask the cache to send it back.
 *  - Also set a few things related to the browser window.
 * This function requests the page's root-URL; images and related stuff
 * are fetched directly by the HTML module.
 *
 * We return a cache client key for the sake of handling redirection, ugh.
 */
static int Nav_open_url(BrowserWindow *bw, FetchReq *req, int offset)
{
   const DilloUrl *old_url;
   int x, y, idx, key = 0;
   ClientData *client;

   MSG("URL: %s\n", URL_STR_(req->url));

   /* Get the url of the current page */
   idx = a_Nav_stack_idx(bw);
   old_url = a_History_get_url(a_Nav_get_uidx(bw, idx));
   _MSG("Nav_open_url:  old_url='%s' idx=%d\n", URL_STR(old_url), idx);
   /* Record current scrolling position */
   if (old_url) {
      a_UIcmd_get_scroll_xy(bw, &x, &y);
      Nav_save_scroll_pos(bw, idx, x, y);
      _MSG("Nav_open_url:  saved scroll of '%s' at x=%d y=%d\n",
          URL_STR(old_url), x, y);
   }

   /* Update navigation-stack-pointer (offset may be zero) */
   Nav_stack_move_ptr(bw, offset);

   if (URL_FLAGS(req->url) & (URL_E2EQuery | URL_Repush) ||
       !old_url ||
       a_Url_cmp(old_url, req->url) || strcmp(URL_FRAGMENT(old_url), URL_FRAGMENT(req->url))) {
      a_Bw_stop_clients(bw, BW_Root + BW_Img);
      a_Bw_cleanup(bw);

      client = a_Client_new(bw, req->url);
      client->flags |= CLIENT_RootUrl;
      key = a_Capi_open_url(client, req, NULL, NULL);
   }
   return key;
}

/*
 * Cancel the last expected url if present. The responsibility
 * for actually aborting the data stream remains with the caller.
 */
void a_Nav_cancel_expect(BrowserWindow *bw)
{
   if (a_Bw_expecting(bw)) {
      a_Bw_cancel_expect(bw);
      a_UIcmd_set_buttons_sens(bw);
   }
}

/*
 * We have an answer! Set things accordingly.
 * This function is called for root URLs only.
 * Beware: this function is much more complex than it looks
 *         because URLs with repush pass twice through it.
 */
void a_Nav_expect_done(BrowserWindow *bw)
{
   int posx, posy;
   bool_t goto_old_scroll=TRUE;
   char *fragment = NULL;

   dReturn_if_fail(bw != NULL);

   if (a_Bw_expecting(bw)) {
      int url_idx;
      size_t unused;
      DilloUrl *url = a_Url_dup(a_Bw_expected_url(bw));
      int repush = (URL_FLAGS(url) & URL_Repush);
      int e2equery = (URL_FLAGS(url) & URL_E2EQuery);

      fragment = a_Url_decode_hex_str(URL_FRAGMENT_(url), &unused);

      /* Unset E2EQuery and Repush before adding this url to history */
      a_Url_set_flags(url, URL_FLAGS(url) & ~(URL_E2EQuery|URL_Repush));
      url_idx = a_History_add_url(url);
      a_Url_free(url);

      if (repush) {
         _MSG("a_Nav_expect_done: repush!\n");
      } else if (e2equery) {
         MSG("a_Nav_expect_done: reload!\n");
      } else {
         Nav_stack_truncate(bw, a_Nav_stack_idx(bw) + 1);
         Nav_stack_append(bw, url_idx);
         Nav_stack_move_ptr(bw, 1);
      }

      if (fragment) {
         goto_old_scroll = FALSE;
         if (repush) {
            Nav_get_scroll_pos(bw, &posx, &posy);
            if (posx || posy)
               goto_old_scroll = TRUE;
         } else if (e2equery) {
            /* Reset scroll, so repush goes to fragment in the next pass */
            Nav_save_scroll_pos(bw, a_Nav_stack_idx(bw), 0, 0);
         }
      }
      a_Nav_cancel_expect(bw);
   }

   if (goto_old_scroll) {
      /* Scroll to where we were in this page */
      Nav_get_scroll_pos(bw, &posx, &posy);
      if (posx || posy)
         a_UIcmd_set_scroll_xy(bw, posx, posy);
      _MSG("Nav: expect_done scrolling to x=%d y=%d\n", posx, posy);
   } else if (fragment) {
      /* Scroll to fragment */
      a_UIcmd_set_scroll_by_fragment(bw, fragment);
   } else {
      /* Scroll to origin */
      a_UIcmd_set_scroll_xy(bw, 0, 0);
   }

   dFree(fragment);
   Nav_stack_clean(bw);

   const DilloUrl *url = a_History_get_url(a_Nav_get_curr_uidx(bw));

   /*
    * I haven’t traced all of the cases, but there’s no expected url when you’re jumping through the nav stack. This function that we’re in looks so hacky.
    */
   if (url)
      a_UIcmd_set_location_text(bw, URL_STR(url));
   else
      MSG_ERR("a_Nav_expect_done() couldn’t get current URL from history\n");

   _MSG("Nav: a_Nav_expect_done\n");
}

/*
 * Make 'url' the current browsed page (upon data arrival)
 * - Set bw to expect the URL data
 * - Ask the cache to feed back the requested URL (via Nav_open_url)
 *
 * We return a cache client key for the sake of handling redirection, ugh.
 */
int a_Nav_push(BrowserWindow *bw, FetchReq *req)
{
   int key;
   const DilloUrl *e_url;
   const int offset = 0;

   dReturn_val_if_fail (bw != NULL, 0);

   e_url = a_Bw_expected_url(bw);
   if (e_url && !a_Url_cmp(e_url, req->url) &&
       !strcmp(URL_FRAGMENT(e_url),URL_FRAGMENT(req->url))) {
      /* we're already expecting that url (most probably a double-click) */
      return 0;
   }
   a_Nav_cancel_expect(bw);
   a_Bw_expect(bw, req->url);
   key = Nav_open_url(bw, req, offset);
   return key;
}

/*
 * This one does a_Nav_repush's job.
 */
static void Nav_repush(BrowserWindow *bw)
{
   DilloUrl *url;

   a_Nav_cancel_expect(bw);
   if (a_Nav_stack_size(bw)) {
      const DilloUrl *referrer = NULL;
      const bool_t user_saw_url = FALSE;
      const int offset = 0;

      url = a_Url_dup(a_History_get_url(a_Nav_get_curr_uidx(bw)));
      /* Let's make reload be from Cache */
      a_Url_set_flags(url, URL_FLAGS(url) | URL_Repush);
      a_Bw_expect(bw, url);

      FetchReq req = {0, url, referrer, REFERRER_POLICY_NOTSET, HTTP_Sec_Fetch_Dest_document, user_saw_url};
      Nav_open_url(bw, &req, offset);
      a_Url_free(url);
   }
}

static void Nav_repush_callback(void *data)
{
   _MSG(">>>> Nav_repush_callback <<<<\n");
   Nav_repush(data);
}

/*
 * Repush current URL: not an end-to-end reload but from cache.
 * - Currently used to switch to a charset decoder given by the META element.
 * - Delayed to let dillo finish the call flow into a known state.
 *
 * There's no need to stop the parser before calling this function:
 * When the timeout activates, a_Bw_stop_clients will stop the data feed.
 */
void a_Nav_repush(BrowserWindow *bw)
{
   dReturn_if_fail (bw != NULL);
   _MSG(">>>> a_Nav_repush <<<<\n");
   a_Timeout_add(0.0, Nav_repush_callback, (void*)bw);
}

static void Nav_do_reload(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow *)vbw;
   const DilloUrl *h_url = a_History_get_url(a_Nav_get_curr_uidx(bw));
   const bool_t user_saw_url = TRUE;
   DilloUrl *r_url = a_Url_dup(h_url);
   const DilloUrl *referrer = NULL;
   FetchReq req = {0, r_url, referrer, REFERRER_POLICY_NOTSET, HTTP_Sec_Fetch_Dest_document, user_saw_url};
   const int offset = 0;

   /* Let's make reload be end-to-end */
   a_Url_set_flags(r_url, URL_FLAGS(r_url) | URL_E2EQuery);
   /* This is an explicit reload, so clear the SpamSafe flag */
   a_Url_set_flags(r_url, URL_FLAGS(r_url) & ~URL_SpamSafe);
   a_Bw_expect(bw, r_url);
   Nav_open_url(bw, &req, offset);
   a_Url_free(r_url);
}

/*
 * Implement the RELOAD button functionality.
 * (Currently it only reloads the page, not its images)
 */
void a_Nav_reload(BrowserWindow *bw)
{
   dReturn_if_fail (bw != NULL);

   a_Nav_cancel_expect(bw);
   if (a_Nav_stack_size(bw)) {
      const DilloUrl *h_url = a_History_get_url(a_Nav_get_curr_uidx(bw));

      if (URL_FLAGS(h_url) & URL_Post) {
         /* Note the user will be able to change the current page while the question is up, so they can end up reloading a different page if they eventually choose Yes. I don’t see any value in preventing use of the browser while the question is up. Also, the user can press reload repeatedly and grow a stack of these questions, but if that amuses them, let them. */

         a_UIcmd_ask_user(bw, "Repost form data?", Nav_do_reload, bw);
      } else {
         Nav_do_reload(bw);
      }
   }
}

/*
 * Jump to a URL in the Navigation stack.
 */
void a_Nav_jump(BrowserWindow *bw, int idx, int new_bw)
{
   if (idx >= 0 && idx < a_Nav_stack_size(bw)) {
      const DilloUrl *url = a_History_get_url(a_Nav_get_uidx(bw,idx)),
                     *referrer = NULL;
      const bool_t user_saw_url = TRUE;
      FetchReq req = {0, url, referrer, REFERRER_POLICY_NOTSET, HTTP_Sec_Fetch_Dest_document, user_saw_url};

      if (new_bw) {
         a_UIcmd_open_url(bw, UICMD_LOC_FROM_EVENT_STATE, &req);
      } else {
         const int offset = idx - a_Nav_stack_idx(bw);
         Nav_open_url(bw, &req, offset);
      }
   }
}


/* Specific methods -------------------------------------------------------- */

/*
 * Save a URL (from cache or from the net).
 */
void a_Nav_save_url(BrowserWindow *bw,
                    const DilloUrl *url, const char *filename)
{
   char *buf;
   int buf_size;

   MSG("The plan is to save %s.\n", filename);
   if (a_Capi_get_buf(url, &buf, &buf_size)) {
      int fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);

      if (fd == -1) {
         MSG_WARN("Open failed with %s.\n", strerror(errno));
      } else {
         ssize_t just_written = 0;
         size_t total_written = 0;

         while (total_written < (size_t)buf_size) {
            just_written = write(fd, buf + total_written, buf_size - total_written);
            if (just_written == -1) {
               MSG_WARN("Write failed with %s.\n", strerror(errno));
               a_UIcmd_set_msg(bw, "Write failed with %s.", strerror(errno));
               break;
            } else {
               total_written += (size_t)just_written;
            }
         }
         if (just_written != -1) {
            MSG("Wrote %s (%zu bytes).\n", filename, total_written);
            a_UIcmd_set_msg(bw, "Wrote %s (%zu bytes).", filename, total_written);
         }
         close(fd);
      }
   }
}

void a_Nav_view_page_source(BrowserWindow *bw, const DilloUrl *url)
{
   ClientData *client;
   const DilloUrl *referrer = NULL;
   const bool_t user_saw_url = FALSE;
   FetchReq req = {0, url, referrer, REFERRER_POLICY_NOTSET, HTTP_Sec_Fetch_Dest_document, user_saw_url};

   dReturn_if_fail (url);

   a_Bw_stop_clients(bw, BW_Root + BW_Img);
   a_Bw_cleanup(bw);

   client = a_Client_new(bw, url);
   client->flags |= CLIENT_Source | CLIENT_RootUrl;
   (void) a_Capi_open_url(client, &req, NULL, NULL);
}

