#include "../dlib/dlib.h"
#include "msg.h"
#include "prefs.h"
#include "url.h"
#include "referrer.h"


#define NO_CROSS_ORIGIN 1 /* If you allow cross origin, you will have to add a check to prevent data: and file: referrers */


static bool_t Referrer_downgrade(const DilloUrl *destination, const DilloUrl *referring)
{
   return !dStrAsciiCasecmp(URL_SCHEME(destination), "http") &&
           dStrAsciiCasecmp(URL_SCHEME(referring), "http");
}

char *a_Referrer_make(ReferrerPolicyType p, const DilloUrl *destination, const DilloUrl *referring)
{
   char *r = NULL;

   if (referring && destination && strcmp(prefs.http_referer, "none")) {
      bool_t dont_return_anything = FALSE;
      bool_t dont_return_path = FALSE;

      switch(p) {
      case REFERRER_POLICY_NO_REFERRER:
         dont_return_anything = TRUE;
         break;
      case REFERRER_POLICY_NO_REFERRER_WHEN_DOWNGRADE:
         if (Referrer_downgrade(destination, referring))
            dont_return_anything = TRUE;
         break;
      case REFERRER_POLICY_SAME_ORIGIN:
         if (!a_Url_same_origin(destination, referring))
            dont_return_anything = TRUE;
         break;
      case REFERRER_POLICY_ORIGIN:
            dont_return_path = TRUE;
         break;
      case REFERRER_POLICY_STRICT_ORIGIN:
         if (Referrer_downgrade(destination, referring))
            dont_return_anything = TRUE;
         else
            dont_return_path = TRUE;
         break;
      case REFERRER_POLICY_ORIGIN_WHEN_CROSS_ORIGIN:
         if (!a_Url_same_origin(destination, referring))
            dont_return_path = TRUE;
         break;
      case REFERRER_POLICY_NOTSET:
      case REFERRER_POLICY_STRICT_ORIGIN_WHEN_CROSS_ORIGIN:
         if (Referrer_downgrade(destination, referring))
            dont_return_anything = TRUE;
         else if (!a_Url_same_origin(destination, referring))
            dont_return_path = TRUE;
         break;
      case REFERRER_POLICY_UNSAFE_URL:
         break;
      default:
         MSG_ERR("Unrecognized referrer policy value %d\n", p);
      }

      if (!dont_return_anything) {
         const DilloUrl *u;

         if (NO_CROSS_ORIGIN && !a_Url_same_origin(destination, referring)) {
            /* We don’t want to supply a cross-origin referrer here, so fake the referrer to be the root of the destination site. */
            u = destination;
            dont_return_path = TRUE;
         } else {
            u = referring;
         }

         if (dont_return_path || !strcmp(prefs.http_referer, "host")) {
            r = dStrconcat(URL_SCHEME(u), "://", URL_AUTHORITY(u), "/", NULL);
         } else {
            r = dStrconcat(URL_SCHEME(u), "://",
                           URL_AUTHORITY(u),
                           URL_PATH_(u) ? URL_PATH_(u) : "/", NULL);
         }
      }
   }
   return r;

}

ReferrerPolicyType a_Referrer_read(const char *policy)
{
   ReferrerPolicyType p;

   /* Would be nicer to build a struct to loop through, but this is quick to write and that would be slower to write. */
   if (!policy)
      p = REFERRER_POLICY_NOTSET;
   else if (!dStrAsciiCasecmp(policy, "no-referrer"))
      p = REFERRER_POLICY_NO_REFERRER;
   else if (!dStrAsciiCasecmp(policy, "no-referrer-when-downgrade"))
      p = REFERRER_POLICY_NO_REFERRER_WHEN_DOWNGRADE;
   else if (!dStrAsciiCasecmp(policy, "origin"))
      p = REFERRER_POLICY_ORIGIN;
   else if (!dStrAsciiCasecmp(policy, "origin-when-cross-origin"))
      p = REFERRER_POLICY_ORIGIN_WHEN_CROSS_ORIGIN;
   else if (!dStrAsciiCasecmp(policy, "same-origin"))
      p = REFERRER_POLICY_SAME_ORIGIN;
   else if (!dStrAsciiCasecmp(policy, "strict-origin"))
      p = REFERRER_POLICY_STRICT_ORIGIN;
   else if (!dStrAsciiCasecmp(policy, "strict-origin-when-cross-origin"))
      p = REFERRER_POLICY_STRICT_ORIGIN_WHEN_CROSS_ORIGIN;
   else if (!dStrAsciiCasecmp(policy, "unsafe-url"))
      p = REFERRER_POLICY_UNSAFE_URL;
   else p = REFERRER_POLICY_NOTSET;

   return p;
}
