/*
 * File: tabs.cc
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

#include <math.h> /* rint */
#include <errno.h>
#include <unistd.h> /* W_OK */
#include <dirent.h> /* DIR, opendir, readdir, etc. for save/select file */
#include <assert.h>

#include "../dw/overlay_window.hh"
#include "../dw/base_input.hh"
#include <FL/Fl.H>
#include <FL/Fl_Widget.H>
/* #include <FL/names.h> */

#include "history.h"
#include "prefs.h"
#include "nav.h"
#include "msg.h"
#include "timeout.hh"
#include "tabs.hh"
#include "uicmd.hh"
#include "keys.hh"
#include "buttons.hh"
#include "misc.h" /* a_Misc_parse_search_url */
#include "html.hh" /* for the form menu */

#include "hsts.h"
#include "paths.hh"

extern int EXPERIMENTAL_PAGING_MODE;

// Handy macro
#define BW2UI(bw) ((UI*)((bw)->ui))

static void set_window_labels(Fl_Window *win, const char *str)
{
   const char *copy;

   win->Fl_Widget::copy_label(str);
   copy = win->label();
   win->label(copy, copy);
}

// --------------- BrowserTab ----------------------


BrowserTab::BrowserTab(int ww, int wh) : Fl_Browser(0,0,ww,wh)
{
   /* Fl_Browser behaviour seems bizarre. HOLD lets us use arrow keys to select, and I have to make the enter key work manually in handle(). We are able to get the callback triggered by a mouse click. */
   type(FL_HOLD_BROWSER);
   when(FL_WHEN_RELEASE_ALWAYS);
   box(FL_FLAT_BOX);
   scrollbar.selection_color(FL_SELECTION_COLOR);
   hscrollbar.selection_color(FL_SELECTION_COLOR);

   dragging = false;
   may_increase_text_size = false;
}

int BrowserTab::item_height(void *item) const
{
   if (!visible(lineno(item)))
      return 0;

   const char *str = item_text(item);

   fl_font(textfont(), textsize());

   if (!*str) {
      return fl_height();
   } else {
      int X, Y, W, H;

      bbox(X, Y, W, H);
      fl_measure(str, W, H, 0);
      return H;
   }
}

int BrowserTab::item_width(void *item) const
{
   if (!visible(lineno(item)))
      return 0;

   int X, Y, W, H;

   bbox(X, Y, W, H);

   fl_font(textfont(), textsize());
   fl_measure(item_text(item), W, H, 0);

   return W;
}

/*
 * Work around a FLTK bug where the selected item is forgotten (but the item itself still knows [or thinks] it’s selected, and so continues to be drawn that way). textsize() was not the first place I looked to find out what the fuck was going on. Changing the text size should be the most innocuous, so you don’t start there. Maybe second most innocuous after changing the color.
 */
void BrowserTab::textsize(int sz)
{
   int n = 0;
   void *sel = selection();
   if (sel)
      n = lineno(sel);

   Fl_Browser::textsize(sz);

   if (n)
      select(n);
}

int BrowserTab::textsize() const
{
   return Fl_Browser::textsize();
}

void BrowserTab::item_draw(void* item, int X, int Y, int W, int H) const
{
   int visible_lines = 1;

   for (int i = lineno(item) - 1; i >= 1; i--)
      if (visible(i))
         visible_lines++;

   if (!(item_selected(item) || !(visible_lines % 2))) {
      // An alternating background color to improve readability. As a text widget, the standard background color is FL_BACKGROUND2_COLOR.
      fl_color(FL_BACKGROUND_COLOR);
      fl_rectf(X, Y, W, H);
   }
   fl_color(item_selected(item) ? fl_contrast(textcolor(), selection_color()) : textcolor());
   fl_draw(item_text(item), X, Y, W, H, FL_ALIGN_TOP_LEFT|FL_ALIGN_WRAP, 0, 0);
}

void BrowserTab::draw()
{
   // Get rid of those dashed focus boxes.
   clear_visible_focus();
   Fl_Browser::draw();
   set_visible_focus();
}

void BrowserTab::back_to_UI()
{
   UI *ui = (UI*)BW2UI(bw);

   ui->tabs()->switch_tab(ui);
}

int BrowserTab::handle(int e)
{
   if (e == FL_KEYBOARD) {
      int event_key = Fl::event_key();

      if (event_key == FL_Enter) {
         do_callback();
         return 1;
      } else if (event_key == FL_Escape) {
         back_to_UI();
         return 1;
      }
   } else if (e == FL_SHORTCUT) {
      // The scrollbar doesn’t get Page_Up/Down until FL_KEYBOARD becomes FL_SHORTCUT.
      int event_key = Fl::event_key();

      if (event_key == FL_Page_Up || event_key == FL_Page_Down) {
         // make the current item something that’s visible so if you’re using a proper keyboard you can then use the arrow keys to move among what’s currently shown rather than having to use the mouse.
         void *item;
         int ret = Fl_Browser::handle(e);

         int X, Y, W, H;
         bbox(X, Y, W, H);

         // NOTE: Due to handle() above, the scrollbar has changed the position, but Fl_Browser doesn’t update all of its state to reflect that until we call find_item().

         item = find_item(event_key == FL_Page_Up ? Y+H-1 : Y);
         if (item && !Fl_Browser_::displayed(item_at(value())))
            Fl_Browser_::select(item);
         return ret;
      }
   } else if (e == FL_MOUSEWHEEL && Fl::e_dy) {
      int i, initialval = value();

      if (initialval) {
         if (Fl::e_dy < 0) {
            for (i = initialval + Fl::e_dy; i >= 1; i--) {
               if (visible(i)) {
                  value(i);
                  break;
               }
            }
         } else {
            int s = size();
            for (i = initialval + Fl::e_dy; i <= s; i++) {
               if (visible(i)) {
                  value(i);
                  break;
               }
            }
         }
      }
      if (value() != initialval) {
         do_callback();
         return 1;
      }
   } else if (e == FL_PUSH) {
      mouse_y = Fl::event_y();
   } else if (e == FL_RELEASE) {
      if (dragging) {
         dragging = false;

         /* Don’t trigger callback() */
         int w = when();
         when(FL_WHEN_NEVER);
         int ret = Fl_Browser:: handle(e);
         when(w);

         return ret;
      }
   } else if (e == FL_DRAG) {
      int pos;

      if (!dragging && (find_item(mouse_y) != find_item(Fl::event_y()))) {
         // only initiate dragging if we go far enough to be on a different item
         dragging = true;
      }
      if (dragging) {
         pos = vposition() + FL_NORMAL_SIZE * (Fl::event_y() - mouse_y) / 4;
         if (pos > full_height() - h())
            pos = full_height() - h();
         vposition(pos);
         mouse_y = Fl::event_y();
         return 1;
      }
   }
   return Fl_Browser::handle(e);
}

char *BrowserTab::make_label(const char *title, const DilloUrl *url)
{
   char *label;

   if (url) {
      const char *data = dStr_printable(URL_DATA(url), 350);
      Dstr *dl = dStr_new(NULL);

      if (title && *title)
         dStr_sprintfa(dl, "%s\n    ", title);
      dStr_sprintfa(dl, "%s", URL_STR(url));
      if (data && *data)
         dStr_sprintfa(dl, " POST:%s", data);
      label = dl->str;
      dStr_free(dl, 0);
   } else {
      label = dStrdup("[empty — no URL]");
   }
   return label;
}

void BrowserTab::build (BrowserWindow *bw)
{
   this->bw = bw;
   clear();
   textsize(FL_NORMAL_SIZE);
}

void BrowserTab::bigger_if_theres_room()
{
   may_increase_text_size = true;

   if (h() >= full_height()) {
      int original_size = textsize(), current = 2 * original_size;

      while (textsize(current), h() < full_height()) {
         current--;
      }
   }
}

void BrowserTab::resize(int x, int y, int w, int h)
{
   Fl_Browser::resize(x, y, w, h);

   if (may_increase_text_size) {
      /* Jiggling the textsize is a workaround because FLTK caches full_height_, and it will only check it again if Fl_Browser::textsize() is called with a different value from the current one. */
      if (textsize() == FL_NORMAL_SIZE)
         textsize(FL_NORMAL_SIZE + 1);
      textsize(FL_NORMAL_SIZE);
      bigger_if_theres_room();
   }

   OverlayWindow *win = (OverlayWindow *)window();
   if (win)
      win->update_overlay_bounds(x, y, w, h);
}

// --------------- TablistTab ----------------------


TablistTab::TablistTab(int ww, int wh, CustTabs *tabs) : BrowserTab(ww,wh)
{
   label("Dillo tab list");
   this->tabs = tabs;
}

void TablistTab::do_tablist_cb()
{
   int selected_item = value();

   /* value() can be 0 if you click empty space outside the item list. We could use this to send you back to your old tab, but you already have a list of tabs in front of you if you want that, and it was probably an accidental click. */
   if (selected_item) {
      UI *ui = (UI *)data(selected_item);
      CustTabs *tabs = (CustTabs *)ui->parent();

      tabs->value(ui);
   }
}

void TablistTab::tablist_cb(Fl_Widget *wid, void *)
{
   if (Fl::event() == FL_RELEASE || Fl::event_key() == FL_Enter) {
      TablistTab *self = (TablistTab *)wid;
      self->do_tablist_cb();
   }
}

void TablistTab::build (BrowserWindow *bw)
{
   int i, j;
   callback(tablist_cb);

   BrowserTab::build(bw);

   for (i = tabs->non_UI_tabs; i < tabs->children(); i++) {
      UI *ui = (UI *)tabs->child(i);

      for (j = 0; j < a_Bw_num(); j++) {
         BrowserWindow *bw = a_Bw_get(j);

         if (ui == (UI*)(bw->ui)) {
            int idx = a_Nav_get_curr_uidx(bw);
            const DilloUrl *url = a_History_get_url(idx);
            const char *title = a_History_get_title(idx, 0);
            char *label = make_label(title, url);

            add(label, ui);
            dFree(label);
            break;
         }
      }
   }
   select(1);
}

LinkMenuTab::LinkMenuTab(int ww, int wh) : BrowserTab(ww,wh)
{
   label("Dillo link menu");
}

LinkMenuTab::~LinkMenuTab()
{
   a_FetchReq_free(req);
}

void LinkMenuTab::do_linkmenu_cb()
{
   bool went_elsewhere = false;
   int i = value();

   OverlayWindow *win = (OverlayWindow *)window();
   win->set_overlay_link_dest("");

   if (i >= 1) {
      switch (i) {
      case 1:
         a_UIcmd_open_url(bw, UICMD_LOC_HERE, req);
         break;
      case 2:
         a_UIcmd_open_url(bw, UICMD_LOC_NEW_TAB_FOCUSED, req);
         went_elsewhere = true; // the new tab
         break;
      case 3:
         a_UIcmd_open_url(bw, UICMD_LOC_NEW_WINDOW, req);
         break;
      case 4:
         a_UIcmd_add_bookmark(bw, req->url);
         break;
      case 5:
         a_UIcmd_copy_urlstr(bw, URL_STR(req->url));
         break;
      case 6:
         a_UIcmd_save_link(bw, req->url);
         went_elsewhere = true; // to the savefile interface
         break;
      default:
         break;
      }
   }
   if (!went_elsewhere)
      back_to_UI();
}

void LinkMenuTab::linkmenu_cb(Fl_Widget *wid, void *)
{
   if (Fl::event() == FL_RELEASE || Fl::event_key() == FL_Enter) {
      LinkMenuTab *self = (LinkMenuTab *)wid;
      self->do_linkmenu_cb();
   }
}

int LinkMenuTab::handle(int event)
{
   if ((event == FL_DRAG && Fl::event_inside(this)) || event == FL_MOVE)
      ((OverlayWindow *)window())->mouse_moved(Fl::event_x(), Fl::event_y());

   return BrowserTab::handle(event);
}

void LinkMenuTab::build (BrowserWindow *bw, FetchReq *req, bool_t link_cached)
{
   BrowserTab::build(bw);

   this->req = req;

   callback(linkmenu_cb);

   add("Open link");
   add("Open in new tab");
   add("Open in new window");
   add("Bookmark link");
   add("Copy link URL");
   if (link_cached) {
      add("Save link");
   } else {
      add("");
      hide(size());
   }
   add("Close this menu");
   select(1);

   bigger_if_theres_room();
}

ImageMenuTab::ImageMenuTab(int ww, int wh) : BrowserTab(ww,wh)
{
   label("Dillo image menu");
   showing_link = false;
}

ImageMenuTab::~ImageMenuTab()
{
   a_FetchReq_free(imgreq);
   if (linkreq)
      a_FetchReq_free(linkreq);
   a_Url_free(page_url);
}

void ImageMenuTab::do_imagemenu_cb()
{
   bool went_elsewhere = false;
   int i = value();

   OverlayWindow *win = (OverlayWindow *)window();
   win->set_overlay_link_dest("");

   if (i >= 1) {
      switch (i) {
      case 1:
         a_UIcmd_open_url(bw, UICMD_LOC_HERE, imgreq);
         break;
      case 2:
         a_UIcmd_open_url(bw, UICMD_LOC_NEW_TAB_FOCUSED, imgreq);
         went_elsewhere = true; // the new tab
         break;
      case 3:
         a_UIcmd_open_url(bw, UICMD_LOC_NEW_WINDOW, imgreq);
         break;
      case 4:
         if (*text(4)) {
            const bool_t user_saw_url = TRUE, is_background = FALSE;
            a_Bw_load_images(bw, (void*)imgreq->url, user_saw_url, is_background);
         }
         break;
      case 5:
         a_UIcmd_add_bookmark(bw, imgreq->url);
         break;
      case 6:
         a_UIcmd_copy_urlstr(bw, URL_STR(imgreq->url));
         break;
      case 7:
         a_UIcmd_save_link(bw, imgreq->url);
         went_elsewhere = true; // to the savefile interface
         break;

      case 9:
         a_UIcmd_open_url(bw, UICMD_LOC_HERE, linkreq);
         break;
      case 10:
         a_UIcmd_open_url(bw, UICMD_LOC_NEW_TAB_FOCUSED, linkreq);
         went_elsewhere = true; // the new tab
         break;
      case 11:
         a_UIcmd_open_url(bw, UICMD_LOC_NEW_WINDOW, linkreq);
         break;
      case 12:
         a_UIcmd_add_bookmark(bw, linkreq->url);
         break;
      case 13:
         a_UIcmd_copy_urlstr(bw, URL_STR(linkreq->url));
         break;
      case 14:
         a_UIcmd_save_link(bw, linkreq->url);
         went_elsewhere = true; // to the savefile interface
         break;
      default:
         break;
      }
   }
   if (!went_elsewhere)
      back_to_UI();
}

void ImageMenuTab::overlay_link_state()
{
      OverlayWindow *win = (OverlayWindow *)window();
      int v = value();

      if (v >= 9 && !showing_link) {
         win->set_overlay_link_dest(URL_STR(linkreq->url));
         showing_link = true;
      } else if (v <= 7 && showing_link) {
         win->set_overlay_link_dest(URL_STR(imgreq->url));
         showing_link = false;
      }
}

void ImageMenuTab::imagemenu_cb(Fl_Widget *wid, void *)
{
   ImageMenuTab *self = (ImageMenuTab *)wid;

   if (Fl::event() == FL_RELEASE || Fl::event_key() == FL_Enter) {
      self->do_imagemenu_cb();
   } else {
      self->overlay_link_state();
   }
}

int ImageMenuTab::handle(int event)
{
   if ((event == FL_DRAG && Fl::event_inside(this)) || event == FL_MOVE)
      ((OverlayWindow *)window())->mouse_moved(Fl::event_x(), Fl::event_y());

   return BrowserTab::handle(event);
}

void ImageMenuTab::build (BrowserWindow *bw, FetchReq *imgreq, FetchReq *linkreq, const DilloUrl *page_url, bool_t loaded, bool_t img_cached, bool_t link_cached)
{
   BrowserTab::build(bw);

   this->imgreq = imgreq;
   this->linkreq = linkreq;
   this->page_url = a_Url_dup(page_url);

   callback(imagemenu_cb);

   add("Isolate image");
   add("Open in new tab");
   add("Open in new window");
   if (loaded) {
      add("");
      hide(size());
   } else {
      add("Load image in page");
   }
   add("Bookmark image");
   add("Copy image URL");
   if (img_cached) {
      add("Save image");
   } else {
      add("");
      hide(size());
   }
   if (linkreq) {
      add("");
      add("Open link");
      add("Open in new tab");
      add("Open in new window");
      add("Bookmark link");
      add("Copy link URL");
      if (link_cached) {
         add("Save link");
      } else {
         add("");
         hide(size());
      }
   }
   add("Close this menu");
   select(1);
   bigger_if_theres_room();
}

PageMenuTab::PageMenuTab(int ww, int wh) : BrowserTab(ww,wh)
{
   label("Dillo page menu");
}

PageMenuTab::~PageMenuTab()
{
   a_Url_free(url);
}

void PageMenuTab::do_pagemenu_cb()
{
   bool went_elsewhere = false;
   int i = value();

   if (i >= 1) {
      switch (i) {
      case 1:
         {
         bool was_set = a_UIcmd_get_selection_mode(bw);
         a_UIcmd_set_selection_mode(bw, !was_set);
         break;
         }
      case 2:
         ((UI *)bw->ui)->findbar_toggle(1);
         break;
      case 3:
         a_UIcmd_view_page_source(bw, url);
         break;
      case 4:
         if (*text(4)) {
            a_UIcmd_view_page_bugs(bw);
            went_elsewhere = true;
         }
         break;
      case 5:
         a_UIcmd_add_bookmark(bw, url);
         break;
      case 6:
         a_UIcmd_save_link(bw, url);
         went_elsewhere = true; // to the savefile interface
         break;
      default:
         break;
      }
   }
   if (!went_elsewhere)
      back_to_UI();
}

void PageMenuTab::pagemenu_cb(Fl_Widget *wid, void *)
{
   if (Fl::event() == FL_RELEASE || Fl::event_key() == FL_Enter) {
      PageMenuTab *self = (PageMenuTab *)wid;
      self->do_pagemenu_cb();
   }
}

void PageMenuTab::build (BrowserWindow *bw, const DilloUrl *url, bool_t bugs)
{
   BrowserTab::build(bw);

   this->url = a_Url_dup(url);

   callback(pagemenu_cb);

   add(a_UIcmd_get_selection_mode(bw) ? "Turn off text selection" : "Turn on text selection");
   add("Find text");
   add("View page source");
   if (bugs) {
      add("View page bugs");
   } else {
      add("");
      hide(size());
   }
   add("Bookmark this page");
   add("Save page");
   add("Close this menu");
   select(1);
   bigger_if_theres_room();
}

FormMenuTab::FormMenuTab(int ww, int wh) : BrowserTab(ww,wh)
{
   label("Dillo form menu");
}

FormMenuTab::~FormMenuTab()
{
   a_Url_free(url);
}

void FormMenuTab::do_formmenu_cb()
{
   int i = value();

   if (i >= 1) {
      switch (i) {
      case 1:
         {
         void *doc = a_Bw_get_url_doc(bw, url);

         if (doc)
            a_Html_form_submit(doc, vform);
         }
         break;
      case 2:
         {
         void *doc = a_Bw_get_url_doc(bw, url);

         if (doc)
            a_Html_form_reset(doc, vform);
         }
         break;
      case 3:
         {
            bool visible = !strcmp(text(3), "Hide hidden fields");

            void *doc = a_Bw_get_url_doc(bw, url);

            if (doc)
               a_Html_form_display_hiddens(doc, vform, !visible);
         }
         break;
      default:
         break;
      }
   }
   back_to_UI();
}

void FormMenuTab::formmenu_cb(Fl_Widget *wid, void *)
{
   if (Fl::event() == FL_RELEASE || Fl::event_key() == FL_Enter) {
      FormMenuTab *self = (FormMenuTab *)wid;
      self->do_formmenu_cb();
   }
}

void FormMenuTab::build (BrowserWindow *bw, const DilloUrl *url, void *vform, bool_t showing_hiddens)
{
   BrowserTab::build(bw);

   this->url = a_Url_dup(url);
   this->vform = vform;

   callback(formmenu_cb);

   add("Submit form");
   add("Reset form");
   add(showing_hiddens ? "Hide hidden fields": "Show hidden fields");
   add("Close this menu");
   select(1);
   bigger_if_theres_room();
}

InputMenuTab::InputMenuTab(int ww, int wh) : BrowserTab(ww,wh)
{
   label("Dillo input menu");
}

InputMenuTab::~InputMenuTab()
{
   delete tracker;
}

void InputMenuTab::do_inputmenu_cb()
{
   Fl_Input *input = (Fl_Input *)tracker->widget();

   if (input) {
      int i = value(), when = input->when();

      if (i >= 1) {
         switch (i) {
         case 1:
            input->value("");

            // This is a FL_PASTE event, but Fl::e_number (of Fl::event()) is not updated when it is sent, so if the recipient checks Fl::event(), it sees whatever event triggered the callback we’re in right now. Then the text editor’s callback can be triggered even though it was really just a paste. So disable the callback temporarily.

            input->when(FL_WHEN_NEVER);
            Fl::paste(*input, false);
            input->when(when);
            break;
         case 2:
            input->value("");
            input->when(FL_WHEN_NEVER);
            Fl::paste(*input, true);
            input->when(when);
            break;
         case 3:
            input->when(FL_WHEN_NEVER);
            Fl::paste(*input, false);
            input->when(when);
            break;
         case 4:
            input->when(FL_WHEN_NEVER);
            Fl::paste(*input, true);
            input->when(when);
            break;
         case 5:
            input->value("");
            break;
         case 6:
            input->undo();
            break;
         default:
            break;
         }
      }
      if (i >= 1 && i <= 6)
         input->take_focus();

   } else {
      /* As of this writing, all uses of this input are in the UI, so I don’t think this can even happen except maybe some microsecond race condition. (Contrasted with the form inputs where a requested page comes in and replaces the current page.) */
      MSG("tracker saved us from input deletion disaster in InputMenuTab!\n");
   }
   back_to_UI();
}

void InputMenuTab::inputmenu_cb(Fl_Widget *wid, void *)
{
   if (Fl::event() == FL_RELEASE || Fl::event_key() == FL_Enter) {
      InputMenuTab *self = (InputMenuTab *)wid;
      self->do_inputmenu_cb();
   }
}

void InputMenuTab::build (BrowserWindow *bw, void *vinput)
{
   BrowserTab::build(bw);

   tracker = new Fl_Widget_Tracker((Fl_Widget *)vinput);

   callback(inputmenu_cb);

   add("Replace with primary");
   add("Replace with clipboard");
   add("Paste primary");
   add("Paste clipboard");
   add("Clear");
   add("Undo");
   add("Close this menu");

   select(1);
   bigger_if_theres_room();
}

// --------------- HistoryTab ----------------------


HistoryTab::HistoryTab(int ww, int wh) : BrowserTab(ww,wh)
{
   label("Dillo history list");
}

void HistoryTab::do_history_cb()
{
   int i = value() - 1;

   if (i >= 0) {
      UI *ui = (UI*)bw->ui;
      const DilloUrl *url = a_History_get_url(i);

      if (navmode) {
         // to meet fwd/back expectations, we have to move in the nav stack
         a_UIcmd_nav_jump(bw, i);
      } else {
         const DilloUrl *referrer = NULL;
         FetchReq req = {0, url, referrer, REFERRER_POLICY_NOTSET, HTTP_Sec_Fetch_Dest_document, TRUE};
         a_UIcmd_open_url(bw, UICMD_LOC_FROM_EVENT_STATE, &req);
      }
      if (ui->contains(Fl::focus())) {
         /* We’re being shown, so I thought we must have focus, but it doesn’t work that way. Anyway, if focus didn’t switch to a new tab, go... */
         back_to_UI();
      }
   } else {
      /* value() can be 0 if you click in the empty space outside of the list of items. */
      back_to_UI();
   }
}

void HistoryTab::history_cb(Fl_Widget *wid, void *)
{
   if (Fl::event() == FL_RELEASE || Fl::event_key() == FL_Enter) {
      HistoryTab *self = (HistoryTab *)wid;
      self->do_history_cb();
   }
}

void HistoryTab::build (BrowserWindow *bw)
{
   BrowserTab::build(bw);

   callback(history_cb);
}

void HistoryTab::populate_from_session()
{
   const DilloUrl *url;
   int i = 0;

   navmode = false;

   while ((url = a_History_get_url(i))) {
      const char *title = a_History_get_title(i, 0);
      char *label = make_label(title, url);
      add(label);
      dFree(label);
      i++;
   }
   if (i) {
      select(i);
      lineposition(i, BOTTOM); // Fl_Browser likes to move it to the middle even when it is the last one, with blank space below
   }
}

void HistoryTab::populate_from_nav(BrowserWindow *bw)
{
   const DilloUrl *url;
   int n, i = 0;
   int *nav_history = a_UIcmd_get_history(bw);

   navmode = true;

   while ((n = nav_history[i]) != -1) {
      char *label;
      const char *title = a_History_get_title(n, 0);

      url = a_History_get_url(n);
      label = make_label(title, url);
      add(label);
      dFree(label);
      i++;
   }
   dFree(nav_history);

   i = a_Nav_stack_idx(bw);
   if (i >= 0) {
      select(i+1);
   }
}


// --------------- CommandsTab ----------------------


CommandsTab::CommandsTab(int ww, int wh) : BrowserTab(ww,wh)
{
   label("Dillo menu");
}

void CommandsTab::do_websearch_cb()
{
   int i = value();

   if (i >= 1 && i < size()) {
      UI *ui = (UI*)bw->ui;
      const char *s = (char *)dList_nth_data(prefs.search_urls, i - 1);
      a_UIcmd_websearch(bw, s);
      ui->tabs()->value(ui);
   } else {
      /* value() can be 0 if you click in the empty space outside of the list of items. */
      back_to_UI();
   }
}

void CommandsTab::websearch_cb(Fl_Widget *wid, void *)
{
   if (Fl::event() == FL_RELEASE || Fl::event_key() == FL_Enter) {
      CommandsTab *self = (CommandsTab *)wid;
      self->do_websearch_cb();
   }
}

void CommandsTab::websearch (BrowserWindow *bw)
{
   BrowserTab::build(bw);

   int n_it = dList_length(prefs.search_urls);

   for (int i = 0; i < n_it; i++) {
      char *label, *url, *source;
      source = (char *)dList_nth_data(prefs.search_urls, i);
      if (!source || a_Misc_parse_search_url(source, &label, &url) < 0)
         continue;
      /* Fl_Browser has columns, where you separate parts of your string with tabs to put those parts in different columns. It was doing something, but not at all what I wanted. Maybe a multiples-of-8 scheme and assuming monospace? So no columns for now. */
      add(label);
   }
   add("Close this menu");
   bigger_if_theres_room();
   callback(websearch_cb);
}

/*
 * Ugh.
 *
 * In most cases, we go back to the tab we came from, but there are a few returns for opening a new tab, opening the history tab, opening the savefile tab.
 */
void CommandsTab::do_commands_cb()
{
   int i = value();

   if (i >= 1) {
      UI *ui = (UI*)bw->ui;

      switch (i) {
      case 1:
         a_UIcmd_smaller();
         break;
      case 2:
         a_UIcmd_larger();
         break;
      case 3:
         a_UIcmd_open_url(bw, UICMD_LOC_NEW_TAB_FOCUSED, NULL);
         return;
      case 4:
         a_UIcmd_open_url(bw, UICMD_LOC_NEW_WINDOW, NULL);
         break;
      case 5:
         a_UIcmd_open_file(bw);
         break;
      case 6:
         websearch(bw);
         return;
      case 7:
         a_UIcmd_reload(bw);
         break;
      case 8:
         a_UIcmd_save(bw); // we’re going to another tab
         return;
      case 9:
         a_UIcmd_book(bw);
         break;
      case 10:
         a_UIcmd_session_history(bw); // we’re going to another tab
         return;
      case 11:
         {
            DilloUrl *url;
            const DilloUrl *referrer = NULL;
            char *urlstr, *path = dStrconcat(DILLO_DOCDIR, "user_help.html", NULL);

            if (access(path, R_OK) == 0) {
               urlstr = dStrconcat("file://", path, NULL);
               url = a_Url_new(urlstr, NULL);
               dFree(urlstr);
            } else {
               MSG("Can't read local help file at \"%s\"."
                   " Getting remote help...\n", path);
               a_UIcmd_set_msg(bw, "Help file not found at '%s'.", path);
                url = a_Url_new("https://web.archive.org/web/20161117005912/http://www.dillo.org/dillo3-help.html", NULL);
            }
            FetchReq req = {0, url, referrer, REFERRER_POLICY_NOTSET, HTTP_Sec_Fetch_Dest_document, TRUE};
            a_UIcmd_open_url(bw, UICMD_LOC_FROM_EVENT_STATE, &req);
            a_Url_free(url);
         }
         break;
      case 12:
         a_Timeout_add(0.0, a_UIcmd_close_all_bw, bw);
         break;
      case 13:
         break;
      case 14:
         prefs.load_stylesheets = prefs.load_stylesheets ? FALSE : TRUE;
         a_UIcmd_repush(bw);
         break;
      case 15:
         prefs.parse_embedded_css = prefs.parse_embedded_css ? FALSE : TRUE;
         a_UIcmd_repush(bw);
         break;
      case 16:
         prefs.load_images = prefs.load_images ? FALSE : TRUE;
         if (prefs.load_images) {
            bool_t user_saw_url = FALSE;
            bool_t is_background = FALSE;
            a_Bw_load_images(bw, NULL, user_saw_url, is_background);
         }
         break;
      case 17:
         prefs.load_background_images = prefs.load_background_images ? FALSE : TRUE;
         if (prefs.load_background_images) {
            bool_t user_saw_url = FALSE;
            bool_t is_background = TRUE;
            a_Bw_load_images(bw, NULL, user_saw_url, is_background);
         } else {
            a_UIcmd_repush(bw);
         }
         break;
      case 18:
         prefs.http_insecure = prefs.http_insecure ? FALSE : TRUE;
         a_UIcmd_update_location_color();

         if (prefs.http_insecure)
            a_Hsts_init(Paths::getPrefsFP(PATHS_HSTS_PRELOAD));
         break;
      case 19:
         prefs.ui_left_handed = prefs.ui_left_handed ? FALSE : TRUE;
         a_UIcmd_update_handedness();
         break;
      case 20:
         EXPERIMENTAL_PAGING_MODE = !EXPERIMENTAL_PAGING_MODE;
         break;
      default:
         break;
      }

      ui->tabs()->value(ui);
   } else {
      /* value() can be 0 if you click in the empty space outside of the list of items. */
      back_to_UI();
   }
}

void CommandsTab::commands_cb(Fl_Widget *wid, void *)
{
   if (Fl::event() == FL_RELEASE || Fl::event_key() == FL_Enter) {
      CommandsTab *self = (CommandsTab *)wid;
      self->do_commands_cb();
   }
}

void CommandsTab::build (BrowserWindow *bw)
{
   char *s;
   BrowserTab::build(bw);

#if 0
   add("Scale down");
   add("Scale up");
#else
   add("Smaller text (and new tabs)");
   add("Larger text (and new tabs)");
#endif
   add("New Tab"); // we will select this as default
   add("New Window");
   add("Open File");
   add("Web Search");
   add("Reload");
   add("Save");
   add("View Bookmarks");
   add("History");
   add("Help");
   add("Close browser");
   add("");
   s = dStrconcat(prefs.load_stylesheets ? "Dis" : "En", "able loading CSS stylesheets", NULL);
   add(s);
   dFree(s);
   s = dStrconcat(prefs.parse_embedded_css ? "Dis" : "En", "able embedded CSS", NULL);
   add(s);
   dFree(s);
   s = dStrconcat(prefs.load_images ? "Dis" : "En", "able loading images", NULL);
   add(s);
   dFree(s);
   s = dStrconcat(prefs.load_background_images ? "Dis" : "En", "able loading BG images", NULL);
   add(s);
   dFree(s);
   s = dStrconcat(prefs.http_insecure ? "Disa" : "A", "llow insecure HTTP", NULL);
   add(s);
   dFree(s);
   s = dStrconcat(prefs.ui_left_handed ? "Right" : "Left", "-handed", NULL);
   add(s);
   dFree(s);
   s = dStrconcat("Turn ", EXPERIMENTAL_PAGING_MODE ? "off" : "on", " experimental paging mode", NULL);
   add(s);
   dFree(s);

   add("Close this menu");
   select(3); // new tab
   callback(commands_cb);
   bigger_if_theres_room();
}

// ---------------------- SavefileTab ------------------------

/* USED BOTH FOR SAVING AND SELECTING FILES. THE TASKS ARE SO SIMILAR THAT OTHERWISE YOU WOULD HAVE TO DUPLICATE ALL OF THE CODE FOR A FEW LITTLE CHANGES, BUT IT’S ALSO A BIT UGLY TO HAVE TO HAVE TWO “OPPOSITE” TASKS RUNNING THROUGH THE SAME CODE. So, sorry, but it had to be done. */

class SavefileInput : public BaseInput {
public:
   SavefileInput (int x, int y, int w, int h) : BaseInput(x,y,w,h) {};
   virtual int handle(int e);
};

int SavefileInput::handle(int e)
{
   if (e == FL_KEYBOARD) {
      int k = Fl::event_key();
      unsigned modifier = Fl::event_state() & (FL_SHIFT | FL_CTRL | FL_ALT);

      if (k == FL_Tab && !modifier) { // tab completion
         char *completed = a_Misc_tabcomplete(value());

         if (completed) {
            value(completed);
            set_changed();
            dFree(completed);
         }
         do_callback();
         return 1;
      } else if (k == '/' && size() && *value() == '~' &&
                 ((size() == 1 && insert_position() == mark()) ||
                  ((MIN(insert_position(), mark()) == 1) &&
                   (MAX(insert_position(), mark()) == size())))) {
         // from "~/" to "file://" + $HOME. The text input had contained an unselected '~', optionally followed by selected text.
         value(dGethomedir());
      }
   }
   return BaseInput::handle(e);
}

void SavefileTab::message(const char *s) {
   msgbox->message(s);
   rearrange();
}

void SavefileTab::show_matching_filenames()
{
   const char *s = input->value(),
              *last_slash = strrchr(s, '/');
   const int dirlen = last_slash ? last_slash - s + 1 : 0, // including slash
             afterlen = strlen(s) - dirlen;

   overwrite->hide();
   ok->hide();
   filebrowser->clear();

   if (dirlen) {
      DIR *dir;
      Dstr *filename = dStr_new("");

      dStr_append_l(filename, s, dirlen);
      dir = opendir(filename->str);

      if (dir) {
         struct dirent *ent;

         while ((ent = readdir(dir))) {
            if (!strcmp(ent->d_name, ".") || !strcmp(ent->d_name, ".."))
               continue;
            if (!strncmp(s + dirlen, ent->d_name, afterlen)) {
               struct stat ss;
               dStr_truncate(filename, dirlen);
               dStr_append(filename, ent->d_name);
               if (stat(filename->str, &ss) == 0 && S_ISDIR(ss.st_mode)) {
                  // Let’s not assume d_type will be available in dirent
                  dStr_append_c(filename, '/');
               }
               filebrowser->add(filename->str);
            }
         }
         filebrowser->sort(FL_SORT_ASCENDING);
         closedir(dir);
      } else {
         char *s = dStrconcat("Can’t read that directory: ", strerror(errno), NULL);
         message(s);
         dFree(s);
      }
      dStr_free(filename, 1);
   }
}

void SavefileTab::back_to_UI()
{
   UI *ui = (UI*)BW2UI(bw);

   a_Url_free(url);
   ui->tabs()->switch_tab(ui);
}

void SavefileTab::consider_saving()
{
   struct stat ss;
   const char *name = input->value();

   if (stat(name, &ss) == 0 && S_ISDIR(ss.st_mode)) {
      message("That’s a directory.");
   } else if (access(name, W_OK) == 0) {
      overwrite->show();
      overwrite->take_focus();
   } else if (errno == ENOENT) {
      bool_t proceed = TRUE;
      const char *dirend = strrchr(name, '/');
      if (dirend) {
         char *base = dStrndup(name, dirend-name+1);

         if (access(base, W_OK)) {
            char *s = dStrconcat("Can’t write in that directory: ", strerror(errno), NULL);
            message(s);
            dFree(s);
            proceed = FALSE;
         }
         dFree(base);
      }
      if (proceed) {
         a_Nav_save_url(bw, url, name);
         back_to_UI();
      }
   } else {
      char *s = dStrconcat("Can’t write that file: ", strerror(errno), NULL);
      message(s);
      dFree(s);
   }
}

void SavefileTab::consider_selecting()
{
   struct stat ss;
   const char *name = input->value();

   if (stat(name, &ss) == 0 && S_ISDIR(ss.st_mode)) {
      message("That’s a directory.");
   } else if (access(name, R_OK) == 0) {
      ok->show();
      ok->take_focus();
   } else if (errno == ENOENT) {
      message("File does not exist.");
   } else {
      char *s = dStrconcat("Can’t read that file: ", strerror(errno), NULL);
      message(s);
      dFree(s);
   }
}

void SavefileTab::do_input_cb()
{
   message(NULL);

   if (Fl::event_key() == FL_Enter) {
      /* cursor at end. For some reason, pressing enter causes the whole filename to be selected. */
      input->insert_position(input->size());

      if (OUR_JOB_IS_TO_SAVE)
         consider_saving();
      else
         consider_selecting();
   } else {
      show_matching_filenames();
   }
}

void SavefileTab::input_cb(Fl_Widget *, void *vself)
{
   SavefileTab *self = (SavefileTab *)vself;
   self->do_input_cb();
}

void SavefileTab::do_cancel_cb(Fl_Widget *wid)
{
   back_to_UI();
}

void SavefileTab::cancel_cb(Fl_Widget *wid, void *vself)
{
   SavefileTab *self = (SavefileTab *)vself;
   self->do_cancel_cb(wid);
}

void SavefileTab::do_overwrite_cb(Fl_Widget *wid)
{
   a_Nav_save_url(bw, url, input->value());
   overwrite->hide();
   back_to_UI();
}

void SavefileTab::overwrite_cb(Fl_Widget *wid, void *vself)
{
   SavefileTab *self = (SavefileTab *)vself;
   self->do_overwrite_cb(wid);
}

void SavefileTab::do_ok_cb(Fl_Widget *wid)
{
   MSG("File selected: %s\n", input->value());
   dStr_truncate(filename, 0);
   dStr_append(filename, input->value());
   ok->hide();
   back_to_UI();
}

void SavefileTab::ok_cb(Fl_Widget *wid, void *vself)
{
   SavefileTab *self = (SavefileTab *)vself;
   self->do_ok_cb(wid);
}

void SavefileTab::do_filebrowser_cb(Fl_Widget *wid)
{
   int line = filebrowser->value();

   input->take_focus();

   if (line) {
      struct stat ss;
      const char *filename = filebrowser->text(line);

      message(NULL);
      input->value(filename);
      if (stat(filename, &ss) == 0 && S_ISDIR(ss.st_mode)) {
         show_matching_filenames();
      } else {
         if (OUR_JOB_IS_TO_SAVE)
            consider_saving();
         else
            consider_selecting();
      }
   }
}

void SavefileTab::filebrowser_cb(Fl_Widget *wid, void *vself)
{
   SavefileTab *self = (SavefileTab *)vself;
   self->do_filebrowser_cb(wid);
}

SavefileTab::SavefileTab(int ww, int wh) : CustGroupVertical(0,0,ww,wh)
{
   int min_button_h_px;

   Fl_Group::current(0);

   min_button_h_px = (int) rint(prefs.dpi / 25.4 * prefs.ui_min_button_h_mm);

   input = new SavefileInput(0,0,ww, min_button_h_px);
   input->when(FL_WHEN_ENTER_KEY_CHANGED); // at least for a Fl_Input, this amounts to when changed or when enter pressed.
   input->callback(input_cb, this);

   add(input);

   buttons = new Fl_Group(0,min_button_h_px, ww, min_button_h_px);
   add(buttons);

   cancel = new FocusAndEnterButton(0, min_button_h_px, ww/2, min_button_h_px);
   cancel->label("Cancel");
   cancel->callback(cancel_cb, this);
   buttons->add(cancel);

   overwrite = new FocusAndEnterButton(ww/2,min_button_h_px, ww - cancel->w(), min_button_h_px);
   overwrite->label("Overwrite");
   overwrite->callback(overwrite_cb, this);
   buttons->add(overwrite);

   ok = new FocusAndEnterButton(ww/2, min_button_h_px, ww - cancel->w(), min_button_h_px);
   ok->label("OK");
   ok->callback(ok_cb, this);
   buttons->add(ok);

   msgbox = new MsgBox(0,2*min_button_h_px, ww, 0);
   msgbox->align(FL_ALIGN_TOP_LEFT|FL_ALIGN_INSIDE|FL_ALIGN_WRAP);
   add(msgbox);

   filebrowser = new Fl_Browser(0,2*min_button_h_px, ww, wh-2*min_button_h_px);
   filebrowser->callback(filebrowser_cb, this);
   filebrowser->type(FL_HOLD_BROWSER);
   filebrowser->scrollbar.selection_color(FL_SELECTION_COLOR);
   filebrowser->hscrollbar.selection_color(FL_SELECTION_COLOR);
   add(filebrowser);
   resizable(filebrowser);
}

int SavefileTab::handle(int e)
{
   if (e == FL_KEYBOARD) {
      int k = Fl::event_key();

      if (k == FL_Escape) {
         cancel->do_callback();
         return 1;
      }
      if (k == FL_Tab) {
         unsigned modifier = Fl::event_state() & (FL_SHIFT | FL_CTRL | FL_ALT);

         // Let the Input have it for Tab completion.
         if (!modifier)
            return 0;
      }
   }
   return CustGroupVertical::handle(e);
}

void SavefileTab::buildSave (int ww, int wh, const DilloUrl *url, BrowserWindow *bw)
{
   char *filename = a_Misc_make_save_filename(URL_PATH_(url), URL_QUERY_(url));

   label("Dillo: save file");
   this->OUR_JOB_IS_TO_SAVE = true;
   this->url = a_Url_dup(url);
   this->bw = bw;
   this->filename = NULL;

   msgbox->labelsize(FL_NORMAL_SIZE);
   input->value(filename);
   show_matching_filenames();
   dFree(filename);
   input->take_focus();
}

void SavefileTab::buildSelect (int ww, int wh, BrowserWindow *bw, Dstr *filename)
{
   label("Dillo: select file");
   this->OUR_JOB_IS_TO_SAVE = false;
   this->url = NULL;
   this->bw = bw;
   this->filename = filename;

   msgbox->labelsize(FL_NORMAL_SIZE);
   input->value(filename->str);
   show_matching_filenames();
   input->take_focus();
}

// ---------------------- DisplayTab -----------------

void close_cb(Fl_Widget *wid, void *vui)
{
   UI *ui = (UI*) vui;

   ui->tabs()->switch_tab(ui);
}

DisplayTab::DisplayTab(int ww, int wh) : Fl_Group(0,0,ww,wh)
{
   int min_button_h_px;

   buffer = new Fl_Text_Buffer();

   Fl_Group::current(0);

   min_button_h_px = (int) rint(prefs.dpi / 25.4 * prefs.ui_min_button_h_mm);

   display = new Fl_Text_Display(0, 0, ww, wh-min_button_h_px);
   add(display);
   resizable(display);

   close = new FocusAndEnterButton(0,wh-min_button_h_px, ww, min_button_h_px);
   close->label("Close");
   close->box(FL_FLAT_BOX);
   close->callback(close_cb);
   add(close);
}

DisplayTab::~DisplayTab ()
{
   display->buffer(NULL);
   delete buffer;
}

int DisplayTab::handle(int e)
{
   if (e == FL_KEYBOARD && Fl::event_key() == FL_Escape) {
      close->do_callback();
      return 1;
   }
   return Fl_Group::handle(e);
}

void DisplayTab::build(BrowserWindow *bw, const char *title, const char *contents)
{
   label(title);

   close->user_data(BW2UI(bw));

   buffer->text(contents);
   display->textsize(FL_NORMAL_SIZE);
   display->buffer(buffer);

   /* enable wrapping lines; text uses entire width of window */
   display->wrap_mode(Fl_Text_Display::WRAP_AT_BOUNDS, 0);
}

/*
 * ----------------------CustTabs ----------------------------
 */

CustTabs::CustTabs (int ww, int wh, const char *lbl) :
      Fl_Wizard(0,0,ww,wh,lbl)
{
   Fl_Widget *placeholder;

   box(FL_FLAT_BOX);

   Fl_Group::current(0);
   placeholder = new Fl_Group(ww, wh, 0, 0);
   add(placeholder);
   assert(non_UI_tabs == 1);

}

void CustTabs::use_nonUI_tab(Fl_Widget *t)
{
   Fl_Widget *was_shown = value(); // WE ABSOLUTELY HAVE TO GET value() BEFORE INSERTING A VISIBLE CHILD

   insert(*t, 0);
   was_shown->hide();
   if (window())
      window()->cursor(FL_CURSOR_DEFAULT);

   Fl_Widget *old_nonUI_tab = child(1);
   remove(1);
   delete old_nonUI_tab;
}

void CustTabs::show_tablist(BrowserWindow *bw)
{

   TablistTab *t = new TablistTab(w(),h(), this);

   t->build(bw);
   use_nonUI_tab(t);
}

void CustTabs::show_pagemenu(BrowserWindow *bw, const DilloUrl *url, bool_t bugs)
{

   PageMenuTab *t = new PageMenuTab(w(),h());

   t->build(bw, url, bugs);
   use_nonUI_tab(t);
}

void CustTabs::show_linkmenu(BrowserWindow *bw, FetchReq *req, bool_t cached)
{

   LinkMenuTab *t = new LinkMenuTab(w(),h());

   t->build(bw, req, cached);
   use_nonUI_tab(t);

   OverlayWindow *win = (OverlayWindow *)window();
   win->update_overlay_bounds(x(), y(), w(), h());
   win->set_overlay_link_dest(URL_STR(req->url));
}

void CustTabs::show_imagemenu(BrowserWindow *bw, FetchReq *imgreq, FetchReq *linkreq, const DilloUrl *page_url, bool_t loaded, bool_t img_cached, bool_t link_cached)
{

   ImageMenuTab *t = new ImageMenuTab(w(),h());

   t->build(bw, imgreq, linkreq, page_url, loaded, img_cached, link_cached);
   use_nonUI_tab(t);

   OverlayWindow *win = (OverlayWindow *)window();
   win->update_overlay_bounds(x(), y(), w(), h());
   win->set_overlay_link_dest(URL_STR(imgreq->url));
}

void CustTabs::show_formmenu(BrowserWindow *bw, const DilloUrl *url, void *vform, bool_t showing_hiddens)
{

   FormMenuTab *t = new FormMenuTab(w(),h());

   t->build(bw, url, vform, showing_hiddens);
   use_nonUI_tab(t);
}

void CustTabs::show_inputmenu(BrowserWindow *bw, void *vinput)
{

   InputMenuTab *t = new InputMenuTab(w(),h());

   t->build(bw, vinput);
   use_nonUI_tab(t);
}

void CustTabs::show_session_history(BrowserWindow *bw)
{

   HistoryTab *t = new HistoryTab(w(),h());
   t->build(bw);
   t->populate_from_session();
   use_nonUI_tab(t);
}

void CustTabs::show_nav_history(BrowserWindow *bw)
{

   HistoryTab *t = new HistoryTab(w(),h());
   t->build(bw);
   t->populate_from_nav(bw);
   use_nonUI_tab(t);
}

void CustTabs::show_savefile(const DilloUrl *url, BrowserWindow *bw)
{

   SavefileTab *t = new SavefileTab(w(),h());
   t->buildSave(w(), h(), url, bw);
   use_nonUI_tab(t);
}

void CustTabs::show_selectfile(BrowserWindow *bw, Dstr *filename)
{

   SavefileTab *t = new SavefileTab(w(),h());
   t->buildSelect(w(), h(), bw, filename);
   use_nonUI_tab(t);
}

void CustTabs::show_display(BrowserWindow *bw, const char *title, const char *contents)
{

   DisplayTab *t = new DisplayTab(w(),h());
   t->build(bw, title, contents);
   use_nonUI_tab(t);
}

void CustTabs::show_commands(BrowserWindow *bw)
{
   CommandsTab *t = new CommandsTab(w(),h());
   t->build(bw);
   use_nonUI_tab(t);
}

int CustTabs::handle(int e)
{
   int ret = 0;
   _MSG("CustTabs::handle e=%s\n", fl_eventnames[e]);

   if (e == FL_KEYBOARD) {
      return 0; // Receive as shortcut
   } else if (e == FL_SHORTCUT) {
      UI *ui = (UI*)value();
      BrowserWindow *bw = a_UIcmd_get_bw_by_widget(ui);
      KeysCommand_t cmd = Keys::getKeyCmd();
      if (cmd == KEYS_NOP) {
         // Do nothing
         _MSG("CustTabs::handle KEYS_NOP\n");
      } else if (cmd == KEYS_CLOSE_ALL) {
         if (!bw) {
            // We need to be in a regular tab for this to work. The question bar sometimes doesn’t naturally belong in a tab, and then we sometimes need to get there from the depths. We have BrowserWindow, which is really a sort of Dillo-world bag that mostly goes along with the FLTK-world UI, which is really a tab, with UIcmd stuck in the middle of everything… The question bar could be a peer to the DilloTabs, but do you want to tell DilloTabs that it exists so it can tell the UIs and so on? You can’t cast the win() to some type that has a question bar because the window is either an Xembed or a Fl_Window or a Fl_Double_Window…
            MSG_WARN("What a bad hack, but you need a hack of some sort, or you need to rearchitect uicmd/tabs/bw/ui…\n");
            bw = a_UIcmd_get_bw_by_widget(child(non_UI_tabs));
            value(child(non_UI_tabs));
         }
         a_Timeout_add(0.0, a_UIcmd_close_all_bw, bw);
         ret = 1;
      } else if (bw) {
         if (cmd == KEYS_NEW_TAB) {
            a_UIcmd_open_url(bw, UICMD_LOC_NEW_TAB_FOCUSED, NULL);
            ret = 1;
         } else if (cmd == KEYS_CLOSE_TAB) {
            a_UIcmd_close_bw(bw);
            ret = 1;
         } else if (cmd == KEYS_TABLIST) {
            show_tablist(bw);
            ret = 1;
         } else if (cmd == KEYS_LEFT_TAB) {
            prev_tab();
            ret = 1;
         } else if (cmd == KEYS_RIGHT_TAB) {
            next_tab();
            ret = 1;
         } else if (cmd == KEYS_NEW_WINDOW) {
            a_UIcmd_open_url(bw, UICMD_LOC_NEW_WINDOW, NULL);
            ret = 1;
         }
      }
   }

   return (ret) ? ret : Fl_Group::handle(e);
}

/*
 * Create a new tab with its own UI
 */
UI *CustTabs::add_new_tab(Fl_Widget *main, int focus)
{
   int i = 0, n;

   /* The UI is constructed in a comfortable fitting size, and then resized
    * so FLTK doesn't get confused later with even smaller dimensions! */
   current(0);
   UI *new_ui = new UI(0,0,w(), h(), main);
// new_ui->resize(0,0,w(),h());
   new_ui->tabs(this);

Fl_Widget *was_shown = value(); // Best to get old value before adding a second visible tab because value() will hide() us and then we have to show() ourselves again manually.
   add(new_ui);

   if (focus) {
      was_shown->hide();
      if (window())
      window()->cursor(FL_CURSOR_DEFAULT);
   } else {
      new_ui->hide();
   }
   n = children() - non_UI_tabs;
   for (i = non_UI_tabs; i < children(); i++) {
      UI *ui = (UI *)child(i);
      ui->update_tab_count(n);
   }
   return new_ui;
}

/*
 * Remove tab by UI
 */
void CustTabs::remove_tab(UI *ui)
{
   int i = 0, n;

   if (children() - non_UI_tabs > 1) {
      int active = find(value());
      int new_active = active > non_UI_tabs ? active - 1 : children() - 1;
      switch_tab(child(new_active));
   }

   remove(ui);
   delete(ui);

   n = children() - non_UI_tabs;
   for (i = non_UI_tabs; i < children(); i++) {
      UI *ui = (UI *)child(i);
      ui->update_tab_count(n);
   }
}

void CustTabs::value(Fl_Widget *new_tab)
{
   const char *title = new_tab->label();

   /* So there’s a child array, and there are special ones in the front. Which
    * has been mostly a pain. But not so avoidable.
    */
   Fl_Wizard::value(new_tab);

   // Update window title
   set_window_labels(window(), title ? title : "");
}

// 2025: value() without arguments is also simpleminded. If you insert a new child, expecting to switch away from an old child, it will fuck up because value() just returns the first visible child. It does no more careful remembering than that.

// In fact, Fl_Wizard is so tiny and simpleminded that I imagine we could get rid of it without difficulty.
void CustTabs::switch_tab(Fl_Widget *new_tab)
{
   Fl_Widget *old_tab = value();

   if (new_tab != old_tab) {
      /* We make a point of calling show() before value() is changed because
       * the wizard may hide the old one before showing the new one. In that
       * case, the new UI gets focus with Fl::e_keysym set to whatever
       * triggered the switch, and this is a problem when it's Tab/Left/Right/
       * Up/Down because some widgets (notably Fl_Group and Fl_Input) exhibit
       * unwelcome behaviour in that case. If the new widgets are already
       * shown, fl_fix_focus will fix everything with Fl::e_keysym temporarily
       * cleared.
       */
      new_tab->show();
      value(new_tab);
   }
}

void CustTabs::prev_tab()
{
   if (children() - non_UI_tabs > 1) {
      int active = find(value());
      int new_idx = active > non_UI_tabs ? active - 1 : children() - 1;
      Fl_Widget *new_tab = child(new_idx);
      value(new_tab);
   }
}

void CustTabs::next_tab()
{
   if (children() - non_UI_tabs > 1) {
      int active = find(value());
      int new_idx = active < children() - 1 ? active + 1 : non_UI_tabs;
      Fl_Widget *new_tab = child(new_idx);
      value(new_tab);
   }
}

