/*
 * File: ui.cc
 *
 * Copyright (C) 2005-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

// UI for Dillo

#include <unistd.h>
#include <stdio.h>
#include <math.h>      /* rint */

#include "keys.hh"
#include "ui.hh"
#include "msg.h"
#include "tipwin.hh"
#include "misc.h" /* a_Misc_tabcomplete */

#include <FL/Fl.H>
#include <FL/Fl_Box.H>
#include <FL/Fl_Secret_Input.H>
#include <FL/fl_draw.H> /* fl_font */

#include "../dw/overlay_window.hh"

#include "uicmd.hh"
#include "findbar.hh"
#include "questionbar.hh"
#include "usernamepasswordbar.hh"
#include "buttons.hh"
#include "history.h"
#include "nav.h"

const int NOWRAP = 0;

/*
 * Local sub classes
 */

//----------------------------------------------------------------------------

/*
 * (Used to avoid certain shortcuts in the location bar)
 */
class CustInput : public TipWinInput {
private:
   UI *ui;
public:

   // is there a way not to have to store the UI like this?

   CustInput (int x, int y, int w, int h, UI* ui) :
      TipWinInput(x,y,w,h) {this->ui = ui;};
   virtual int handle(int e);
};

/*
 * Disable keys: Up, Down, Page_Up, Page_Down, Tab and
 * CTRL+{o,r,Home,End}  SHIFT+{Left,Right}.
 */
int CustInput::handle(int e)
{
   int k = Fl::event_key();

   _MSG("CustInput::handle event=%d\n", e);

   // We're only interested in some flags
   unsigned modifier = Fl::event_state() & (FL_SHIFT | FL_CTRL | FL_ALT);

   // Don't focus with arrow keys
   if (e == FL_FOCUS &&
       (k == FL_Up || k == FL_Down || k == FL_Left || k == FL_Right)) {
      return 0;
   } else if (e == FL_KEYBOARD) {
      if (k == FL_Escape && modifier == 0) {
         // Let the parent group handle this Esc key
         return 0;
      } else if (k == FL_Tab && modifier == 0) {
         DilloUrl *url = a_Url_new(value(), NULL);
         if (!dStrAsciiCasecmp(URL_SCHEME(url), "file")) {
            char *completed = a_Misc_tabcomplete(URL_PATH(url));
            if (completed) {
               char *s = dStrconcat("file://", completed, NULL);
               ui->set_location(s);
               set_changed();
               dFree(s);
               dFree(completed);
            } else {
               _MSG("No matches.\n");
            }
         }
         a_Url_free(url);
      } else if (k == '/' && !MIN(insert_position(), mark()) && MAX(insert_position(), mark()) == size()) {
         // from "/" to "file:///". The text input had been empty or it had been all selected.
         value("file://");
      } else if (k == '/' && size() && *value() == '~' &&
                 ((size() == 1 && insert_position() == mark()) ||
                  ((MIN(insert_position(), mark()) == 1) &&
                   (MAX(insert_position(), mark()) == size())))) {
         // from "~/" to "file://" + $HOME. The text input had contained an unselected '~', optionally followed by selected text.
         char *urlstr = dStrconcat("file://", dGethomedir(), NULL);
         value(urlstr);
         dFree(urlstr);
      } else if (modifier == FL_SHIFT) {
         if (k == FL_Left || k == FL_Right) {
            // Let these keys get to the UI
            return 0;
         }
         if (k == FL_Insert) {
            // Fl_Input uses Shift-Insert to paste from the clipboard, but we want to use it to paste from primary selection. (Clipboard paste already has Ctrl-V, anyway.)
            //In FLTK 1.4, we could use handle_key() instead of handle(), but it’s private in 1.3.
            Fl::paste(*this, false);
            return 1;
        }

      } else if (modifier == FL_CTRL) {
         if (k == 'l') {
            // Make text selected when already focused.
            insert_position(size(), 0);
            return 1;
         } else if (k == 'h' || k == 'o' || k == 'r' ||
                    k == FL_Home || k == FL_End) {
            // Let these keys get to the UI
            return 0;
         }
      } else if (modifier == 0) {
         if (k == FL_Down || k == FL_Up ||
             k == FL_Page_Down || k == FL_Page_Up || k == FL_Tab) {
            // Give up focus and honor the key
            a_UIcmd_focus_main_area(a_UIcmd_get_bw_by_widget(this));
            return 0;
         }
      }
      if (k == FL_Page_Down || k == FL_Page_Up) {
         // These do nothing of interest when FL_MULTILINE_INPUT isn't set.
         // Let them through for key commands.
         return 0;
      }
   }
   return TipWinInput::handle(e);
}

//----------------------------------------------------------------------------

/* The callback will be triggered as usual on release for left button, but on push for right button. */
class CustForwBackButton : public CustButton {
public:
   CustForwBackButton(int x, int y, int w, int h, const char *l=0) :
      CustButton(x,y,w,h,l) {};
   int handle(int e);
};

int CustForwBackButton::handle(int e)
{
   if (e == FL_LEAVE)
      a_UIcmd_display_link_dest(a_UIcmd_get_bw_by_widget(this), "");

   if (Fl::event_button() == FL_RIGHT_MOUSE) {
      if (e == FL_PUSH) {
         value(0);
         do_callback();
         return 1;
      } else if (e == FL_RELEASE) {
         int rc;
         // don’t do callback
         when(0);
         rc = CustButton::handle(e);
         when(FL_WHEN_RELEASE);
         return rc;
      }
   }
   return CustButton::handle(e);
}

class CustBackButton : public CustForwBackButton {
public:
   CustBackButton(int x, int y, int w, int h, const char *l=0) :
      CustForwBackButton(x,y,w,h,l) {};
   int handle(int e);
};

int CustBackButton::handle(int e)
{
   if (e == FL_ENTER) {
      BrowserWindow *bw = a_UIcmd_get_bw_by_widget(this);
      if (bw) {
// TODO a middle click makes a new tab, and there’s an enter on the new tab before the outside code knows the bw and can add it to the list of bw. It doesn’t really matter because there will be no link destination to display here, but it’s ugly to have to make this test
         int i = a_Nav_stack_idx(bw);
         if (i) {
            const DilloUrl *url = a_History_get_url(a_Nav_get_uidx(bw, i-1));
            a_UIcmd_display_link_dest(bw, URL_STR(url));
         }
      }
   }
   return CustForwBackButton::handle(e);
}

class CustForwButton : public CustForwBackButton {
public:
   CustForwButton(int x, int y, int w, int h, const char *l=0) :
      CustForwBackButton(x,y,w,h,l) {};
   int handle(int e);
};

int CustForwButton::handle(int e)
{
   if (e == FL_ENTER) {
      BrowserWindow *bw = a_UIcmd_get_bw_by_widget(this);
      if (bw) {
// TODO a middle click makes a new tab, and there’s an enter on the new tab before the outside code knows the bw and can add it to the list of bw. It doesn’t really matter because there will be no link destination to display here, but it’s ugly to have to make this test
         int i = a_Nav_stack_idx(bw);
         if (i < a_Nav_stack_size(bw) - 1) {
            const DilloUrl *url = a_History_get_url(a_Nav_get_uidx(bw, i+1));
            a_UIcmd_display_link_dest(bw, URL_STR(url));
         }
      }
   }
   return CustForwBackButton::handle(e);
}

//----------------------------------------------------------------------------

/* Trigger the callback on push, not release. It might be possible to trick a button into doing this by using FL_WHEN_CHANGED and manipulating value(), but this wasn’t immediately working quite right for me, and why not do something clean rather than hacky? */
class CustPushButton : public CustButton {
public:
   CustPushButton(int x, int y, int w, int h, const char *l=0) :
      CustButton(x,y,w,h,l) { when(0); };
   int handle(int e);
};

int CustPushButton::handle(int e)
{
   if (e == FL_PUSH) {
      do_callback();
      return 1;
   }
   return CustButton::handle(e);
}


//
// Callback functions --------------------------------------------------------
//

/*
 * Callback for the Menu button.
 */
static void menu_cb(Fl_Widget *wid, void *)
{
   int b = Fl::event_button();
   if (b == FL_LEFT_MOUSE || b == FL_RIGHT_MOUSE) {
      BrowserWindow *bw = a_UIcmd_get_bw_by_widget(wid);
      a_UIcmd_commands_tab(bw);
   }
}

/* The location (url bar) group can be in the nav bar iff there’s room. Otherwise goes in a separate row above (loc bar) */
void UI::check_location_fit()
{
   int textwidth;

   fl_font(Location->textfont(), Location->textsize());
   textwidth = Location->textwidth();

   if (NavBar->contains(LocationGroup)) {

      if (textwidth > Location->w()) {
         _MSG("textwidth %d > location w %d\n", textwidth , Location->w());
         LocBar->show();
         LocBar->add(LocationGroup);
         LocBar->resizable(LocationGroup);
         NavBar->resizable(NavBar);
         NavBar->rearrange();
         LocBar->rearrange();
         TopGroup->rearrange();
      }
   } else {
      if (textwidth < NavBar->resizable_width()) {
         _MSG("textwidth %d < NavBar->resizable_width() %d\n", textwidth, NavBar->resizable_width());
         if (prefs.ui_left_handed)
            NavBar->add(*LocationGroup);
         else
            NavBar->insert(*LocationGroup, 0);
         NavBar->resizable(LocationGroup);
         NavBar->rearrange();
         LocBar->hide();
         TopGroup->rearrange();
      }
   }
}

/*
 * Send the browser to the new URL in the location.
 */
static void location_cb(Fl_Widget *wid, void *data)
{
   Fl_Input *i = (Fl_Input*)wid;

   _MSG("location_cb()\n");
   if (Fl::event() == FL_KEYDOWN && Fl::event_key() == FL_Enter) {

      /* Had to check for KEYDOWN because the input menu can trigger paste or replacement using the Enter key, and that’s a change, which is a reason to trigger this callback. */

      a_UIcmd_open_location_urlstr(a_UIcmd_get_bw_by_widget(i), i->value());
   } else {
      UI *ui = (UI*) data;
      ui->check_location_fit();
   }
}

/*
 * Callback handler for button press on the panel
 */
static void b1_cb(Fl_Widget *wid, void *cb_data)
{
   BrowserWindow *bw = a_UIcmd_get_bw_by_widget(wid);
   int ui_button = VOIDP2INT(cb_data);
   int mouse_button = Fl::event_button();

   if (mouse_button >= FL_LEFT_MOUSE && mouse_button <= FL_RIGHT_MOUSE) {
      _MSG("[%s], mouse button %d was pressed\n", button_names[bn], b);
      _MSG("mouse button %d was pressed\n", b);
   }
   switch (ui_button) {
   case UI_BACK:
      if (mouse_button == FL_RIGHT_MOUSE)
         a_UIcmd_nav_history(bw);
      else
         a_UIcmd_back(bw);
      break;
   case UI_FORW:
      if (mouse_button == FL_RIGHT_MOUSE)
         a_UIcmd_nav_history(bw);
      else
         a_UIcmd_forw(bw);
      break;
   case UI_RELOAD:
         a_UIcmd_reload(bw);
      break;
   case UI_BOOKMARKS:
         a_UIcmd_book(bw);
      break;
   case UI_STOP:
         a_UIcmd_stop(bw);
      break;
   case UI_CLOSE:
         a_UIcmd_close_bw(bw);
      break;
   case UI_TABS:
         a_UIcmd_tab_list(bw);
      break;
   default:
      break;
   }
}

//////////////////////////////////////////////////////////////////////////////
// UI class methods
//

//----------------------------
// Panel construction methods
//----------------------------

/*
 * Make a generic navigation button
 */
CustButton *UI::make_button(const char *label, const char *tooltip, int b_n)
{
   int w = NOWRAP, h;
   // Forward and back need special event handling
   CustButton *b = b_n == UI_BACK ?
                   new CustBackButton(p_xpos, 0, bw, bh, label):
                   b_n == UI_FORW ?
                   new CustForwButton(p_xpos, 0, bw, bh, label):
                   new CustButton(p_xpos, 0, bw, bh, label);
   b->callback(b1_cb, INT2VOIDP(b_n));
   b->clear_visible_focus();
   b->box(FL_FLAT_BOX);
   b->down_box(FL_THIN_DOWN_FRAME);
   b->set_tooltip(tooltip);
   b->measure_label(w, h);
   w = MAX(w, min_button_w_px);
   b->size(w, b->h());
   p_xpos += w;
   return b;
}

/*
 * Create the location box (Input/Search)
 */
void UI::make_location(int ww)
{
    // LocationGroup does almost nothing here, but it does mean we don’t have to fight with fltk boxtypes to get a non-jarring separation between Location and Main.
    LocationGroup = new Fl_Group(p_xpos,0,ww-p_xpos,lh,0);
    LocationGroup->begin();

     CustInput *i = new CustInput(p_xpos,1,ww-p_xpos,lh-1,this);
     Location = i;
     i->when(FL_WHEN_ENTER_KEY_CHANGED);
     i->callback(location_cb, this);
     i->set_tooltip("Location");
     i->box(FL_FLAT_BOX);
     if (prefs.http_insecure)
        i->color(PREFS_UI_INSECURE_COLOR);
     p_xpos += i->w();
    LocationGroup->box(FL_FLAT_BOX);
    LocationGroup->resizable(i);
    LocationGroup->end();
}

/*
 * Create the Menu button
 */
void UI::make_menu_button()
{
   CustButton *btn;
   int w = NOWRAP, h;

   MenuButton = btn = new CustPushButton(p_xpos,0,bw,bh,"Menu");
   btn->box(FL_FLAT_BOX);
   btn->measure_label(w, h);
   w = MAX(w, min_button_w_px);
   btn->size(w, h);
   p_xpos += w;
   btn->callback(menu_cb, this);
   btn->set_tooltip("Menu");
   btn->clear_visible_focus();
   if (!prefs.show_menubutton)
      btn->hide();
}

void UI::update_nav_button_order()
{
   if (prefs.ui_left_handed) {
      if (NavBar->contains(LocationGroup))
         NavBar->insert(*LocationGroup, 0);
      NavBar->insert(*TabButton, 0);
      NavBar->insert(*Close, 0);
      NavBar->insert(*Forw, 0);
      NavBar->insert(*Back, 0);
      NavBar->insert(*Bookmarks, 0);
      NavBar->insert(*Reload, 0);
      NavBar->insert(*Stop, 0);
      NavBar->insert(*MenuButton, 0);
   } else {
      NavBar->insert(*MenuButton, 0);
      NavBar->insert(*Stop, 0);
      NavBar->insert(*Reload, 0);
      NavBar->insert(*Bookmarks, 0);
      NavBar->insert(*Back, 0);
      NavBar->insert(*Forw, 0);
      NavBar->insert(*Close, 0);
      NavBar->insert(*TabButton, 0);
      if (NavBar->contains(LocationGroup))
         NavBar->insert(*LocationGroup, 0);
   }
   NavBar->rearrange();
}

/*
 * Create the control panel
 */
void UI::make_panel(int ww)
{
   p_xpos = 0;

   current(0);
   LocBar = new CustGroupHorizontal(0,0,ww,lh);
   LocBar->box(FL_NO_BOX);
   LocBar->hide();

    p_xpos = 0;
   NavBar = new CustGroupHorizontal(0,0,ww,nh);
   NavBar->box(FL_NO_BOX);
    make_location(ww);
    make_menu_button();

    Stop = make_button("stop", "Stop loading", UI_STOP);
    Reload = make_button("reload", "Reload", UI_RELOAD);
    Bookmarks = make_button("book", "Bookmarks", UI_BOOKMARKS);
    Back = make_button("back", "Previous page", UI_BACK);
    Forw = make_button("fwd", "Next page", UI_FORW);
    Close = make_button("close", "Close tab", UI_CLOSE);
    TabButton = make_button("tabs", "Tab list", UI_TABS);

    NavBar->resizable(LocationGroup);

    update_nav_button_order();
    NavBar->rearrange();
}

void UI::status_close_cb(Fl_Widget *, void *data)
{
   UI *ui = (UI *)data;

   ui->StatusBar->hide();
   ui->TopGroup->rearrange();
}

/*
 * Create the status panel
 */
void UI::make_status_bar(int ww, int wh)
{
   int w, h;

   sh = dLib_get_default_font_size() * 6.0 / 5.0 ;
   p_xpos = 0;
   StatusBar = new CustGroupHorizontal(0, wh-sh, ww, sh);
   StatusBar->clear_visible_focus();
   StatusBar->box(FL_NO_BOX);

   Fl_Group::current(0);

    StatusClose = new CustButton(p_xpos, wh-sh, 16, sh, "× ");
    StatusClose->box(FL_FLAT_BOX);
    StatusClose->callback(status_close_cb, this);
    StatusClose->clear_visible_focus();
    StatusClose->set_tooltip("Close");
    w = NOWRAP;
    StatusClose->measure_label(w, h);
    w = MAX(w, min_button_w_px);
    StatusClose->size(w, sh);
    p_xpos += w;

    // Status box
    StatusOutput = new Fl_Box(p_xpos, wh-sh, 0, sh);
    StatusOutput->box(FL_FLAT_BOX); // if changed to a box that uses box_dx, etc, then account for that in determine_status_height()
    StatusOutput->align(FL_ALIGN_LEFT|FL_ALIGN_INSIDE|FL_ALIGN_WRAP);
    StatusOutput->color(FL_FOREGROUND_COLOR);
    StatusOutput->labelcolor(FL_BACKGROUND_COLOR);

   StatusBar->add(StatusClose);
   StatusBar->add(StatusOutput);
   StatusBar->resizable(StatusOutput);

   StatusBar->rearrange();
   StatusBar->hide();
}

/* How tall does the status bar have to be for the output box to have enough room to show the message? */
void UI::determine_status_height()
{
   int H, W = StatusOutput->w();

   // this is from Fl_Widget::draw_label(). I’ve just expressed my irritation in the QuestionBox measurement code.
   if (W > 11 && StatusOutput->align()&(FL_ALIGN_LEFT|FL_ALIGN_RIGHT))
      W -= 6;

   fl_font(StatusOutput->labelfont(), StatusOutput->labelsize());
   fl_measure(StatusOutput->label(), W, H, 0);
   StatusBar->size(StatusBar->w(), H);
   StatusBar->rearrange();
}

/*
 * Set a new message in the status bar.
 */
void UI::set_status(const char *str)
{
   if (str && *str) {
      StatusOutput->copy_label(str);
      determine_status_height();
      StatusBar->show();
   } else {
      StatusBar->hide();
   }
   TopGroup->rearrange();
}

void UI::update_location_color()
{
   Location->color(prefs.http_insecure ? PREFS_UI_INSECURE_COLOR : FL_BACKGROUND2_COLOR);
   Location->redraw();
}

void UI::update_tab_count(int n)
{
   char buf[16];
   int w = NOWRAP, h;

   snprintf(buf, 16, "%d tabs", n);
   TabButton->copy_label(buf);
   TabButton->measure_label(w, h);
   w = MAX(w, min_button_w_px);
   TabButton->size(w, TabButton->h());
   n > 1 ? TabButton->show() : TabButton->hide();
   NavBar->rearrange();
}

/*
 * User Interface constructor
 */
UI::UI(int x, int y, int ui_w, int ui_h, Fl_Widget *main) :
  CustGroupVertical(x, y, ui_w, ui_h, "Dillo")
{
   min_button_h_px = (int) rint(prefs.dpi / 25.4 * prefs.ui_min_button_h_mm);
   min_button_w_px = (int) rint(prefs.dpi / 25.4 * prefs.ui_min_button_w_mm);

   fh = nh = bh = lh = min_button_h_px; // the heights have all become vestigial
   bw = min_button_w_px;

   LocBar = NavBar = NULL;

   Tabs = NULL;
   TopGroup = this;
   TopGroup->box(FL_NO_BOX);
   clear_flag(SHORTCUT_LABEL);

   Fl_Group::current(0);


    // Render area
    Main = main;

    // Find text bar
    FindBar = new Findbar(ui_w, MAX(fh, min_button_h_px), min_button_w_px);

    make_panel(ui_w);  // Location, and Nav buttons

    // Status Panel
    make_status_bar(ui_w, ui_h);

   TopGroup->add(Main);
   TopGroup->resizable(Main);
   TopGroup->add(FindBar);
   TopGroup->add(LocBar);
   TopGroup->add(NavBar);
   TopGroup->add(StatusBar);

   TopGroup->rearrange();

   customize();
}

/*
 * UI destructor
 */
UI::~UI()
{
   _MSG("UI::~UI()\n");
}

/*
 * FLTK event handler for this window.
 */
int UI::handle(int event)
{
   _MSG("UI::handle event=%s\n", fl_eventnames[event]);

   int ret = 0;
   if (event == FL_KEYBOARD) {
      return 0; // Receive as shortcut
   } else if (event == FL_SHORTCUT) {
      KeysCommand_t cmd = Keys::getKeyCmd();
      if (cmd == KEYS_NOP) {
         // Do nothing
      } else if (cmd == KEYS_SCREEN_UP || cmd == KEYS_SCREEN_DOWN ||
                 cmd == KEYS_SCREEN_LEFT || cmd == KEYS_SCREEN_RIGHT ||
                 cmd == KEYS_LINE_UP || cmd == KEYS_LINE_DOWN ||
                 cmd == KEYS_LEFT || cmd == KEYS_RIGHT ||
                 cmd == KEYS_TOP || cmd == KEYS_BOTTOM) {
         a_UIcmd_scroll(a_UIcmd_get_bw_by_widget(this), cmd);
         ret = 1;
      } else if (cmd == KEYS_BACK) {
         a_UIcmd_back(a_UIcmd_get_bw_by_widget(this));
         ret = 1;
      } else if (cmd == KEYS_FORWARD) {
         a_UIcmd_forw(a_UIcmd_get_bw_by_widget(this));
         ret = 1;
      } else if (cmd == KEYS_BOOKMARKS) {
         a_UIcmd_book(a_UIcmd_get_bw_by_widget(this));
         ret = 1;
      } else if (cmd == KEYS_BOOKMARK_PAGE) {
         BrowserWindow *bw = a_UIcmd_get_bw_by_widget(this);
         const DilloUrl *url = a_History_get_url(a_Nav_get_curr_uidx(bw));
         a_UIcmd_add_bookmark(bw, url);
         ret = 1;
      } else if (cmd == KEYS_FIND) {
         findbar_toggle(1);
         ret = 1;
      } else if (cmd == KEYS_GOTO) {
         focus_location();
         ret = 1;
      } else if (cmd == KEYS_OPEN) {
         a_UIcmd_open_file(a_UIcmd_get_bw_by_widget(this));
         ret = 1;
      } else if (cmd == KEYS_RELOAD) {
         a_UIcmd_reload(a_UIcmd_get_bw_by_widget(this));
         ret = 1;
      } else if (cmd == KEYS_STOP) {
         a_UIcmd_stop(a_UIcmd_get_bw_by_widget(this));
         ret = 1;
      } else if (cmd == KEYS_SAVE) {
         a_UIcmd_save(a_UIcmd_get_bw_by_widget(this));
         ret = 1;
      } else if (cmd == KEYS_MENU) {
         BrowserWindow *bw = a_UIcmd_get_bw_by_widget(this);
         a_UIcmd_commands_tab(bw);
         ret = 1;
      } else if (cmd == KEYS_VIEW_SOURCE) {
         BrowserWindow *bw = a_UIcmd_get_bw_by_widget(this);
         const DilloUrl *url = a_History_get_url(a_Nav_get_curr_uidx(bw));
         a_UIcmd_view_page_source(bw, url);
         ret = 1;
      } else if (cmd == KEYS_LARGER) {
         a_UIcmd_larger();
         ret = 1;
      } else if (cmd == KEYS_SMALLER) {
         a_UIcmd_smaller();
         ret = 1;
      }
   } else if (event == FL_HIDE) {
      // hide any overlay window

      /* The usual Dillo bizarreness of
       * BrowserWindow *bw = a_UIcmd_get_bw_by_widget(this);
       * doesn’t work when we’re in the middle of changing tabs, so let’s forget about that and do this directly.
       */

      OverlayWindow *win = (OverlayWindow *)window();
      win->clear_entire_overlay();
   }

   if (!ret) {
      ret = Fl_Group::handle(event);
   }
   return ret;
}

void UI::resize(int x, int y, int w, int h)
{
   int oldW = this->w();
   CustGroupVertical::resize(x,y,w,h);
   check_location_fit();

   if (w != oldW) {
      determine_status_height();
      rearrange();
   }
}

void UI::rearrange()
{
   TopGroup->rearrange();
}

//----------------------------
// API for the User Interface
//----------------------------

/*
 * Get the text from the location input-box.
 */
const char *UI::get_location()
{
   return Location->value();
}

/*
 * Set a new URL in the location input-box.
 */
void UI::set_location(const char *str)
{
   if (!str) str = "";
   Location->value(str);
   Location->insert_position((Fl::focus() == Location) ? strlen(str) : 0);

   /* The Location input is set to trigger the callback when the value changes, but apparently
    * directly changing the value yourself doesn’t count. And set_changed() just sets the
    * flag. So:
    */
   check_location_fit();
}

/*
 * Focus location entry.
 * If it's not visible, show it until the callback is done.
 */
void UI::focus_location()
{
   Location->take_focus();
   // Make text selected when already focused.
   Location->insert_position(Location->size(), 0);
}

/*
 * Focus Main area.
 */
void UI::focus_main()
{
   Main->take_focus();
}

void UI::username_password(const char *msg, UIcmd_username_password_cb cb, void *data)
{
   UsernamePasswordBar *u = new UsernamePasswordBar(w(), nh, msg);
   u->callback(cb, data);
   TopGroup->insert(*u, Main);
   TopGroup->rearrange();
}

void UI::ask_user(const char *question, UIcmd_ask_cb callback, void *data)
{
   Questionbar *q = new Questionbar(w(), nh, min_button_w_px, question, callback, data);
   TopGroup->insert(*q, Main);
   TopGroup->rearrange();
}

/*
 * Customize the UI's panel (show/hide buttons)
 */
void UI::customize()
{
   if ( !prefs.show_back )
      Back->hide();
   if ( !prefs.show_forw )
      Forw->hide();
   if ( !prefs.show_reload )
      Reload->hide();
   if ( !prefs.show_bookmarks )
      Bookmarks->hide();
   if ( !prefs.show_stop )
      Stop->hide();

   if (NavBar)
      NavBar->rearrange();
   if (LocBar)
      LocBar->rearrange();
}

/*
 * Set button sensitivity (Back/Forw/Stop)
 */
void UI::button_set_sens(UIButton btn, int sens)
{
   switch (btn) {
   case UI_BACK:
      (sens) ? Back->activate() : Back->deactivate();
      break;
   case UI_FORW:
      (sens) ? Forw->activate() : Forw->deactivate();
      break;
   case UI_STOP:
      if (sens) {
         Stop->labelcolor(FL_RED);
         Stop->activate();
      } else {
         Stop->labelcolor(FL_FOREGROUND_COLOR);
         Stop->deactivate();
      }
      break;
   default:
      break;
   }
}

/* true if from clipboard, false if from primary selection */
void UI::paste(bool clipboard)
{
   Fl::paste(*Location, clipboard);
}

/*
 * Adjust space for the findbar (if necessary) and show or remove it
 */
void UI::findbar_toggle(bool add)
{
   /* WORKAROUND:
    * This is tricky: As fltk-1.3 resizes hidden widgets (which it
    * doesn't resize when visible!). We need to set the size to (0,0) to
    * get the desired behaviour.
    * (STR#2639 in FLTK bug tracker).
    */

   if (add) {
      if (!FindBar->visible())
         FindBar->size(w(), fh);
      FindBar->show();
   } else {
      // hide
      FindBar->size(0,0);
      FindBar->hide();
      // reset state
      a_UIcmd_findtext_reset(a_UIcmd_get_bw_by_widget(this));
      // focus main area
      focus_main();
   }
   TopGroup->rearrange();
}
