/*
 * File: uicmd.cc
 *
 * Copyright (C) 2005-2011 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

// Functions/Methods for commands triggered from the UI


#include <string.h>
#include <math.h>       /* for rint */

#include <FL/Fl.H>
#include <FL/Fl_Widget.H>
#include <FL/Fl_Wizard.H>

#include "../dw/overlay_window.hh"

#include "paths.hh"
#include "keys.hh"
#include "ui.hh"
#include "uicmd.hh"
#include "tabs.hh"
#include "utf8.hh"
#include "passwordwindow.hh"
#include "bookmark.h"
#include "history.h"
#include "msg.h"
#include "prefs.h"
#include "misc.h"
#include "../dlib/dlib.h"

#include "dw/fltkviewport.hh"

#include "nav.h"

// Handy macro
#define BW2UI(bw) ((UI*)((bw)->ui))

// Platform idependent part
using namespace dw::core;
// FLTK related
using namespace dw::fltk;


/*
 * Forward declarations
 */
static BrowserWindow *UIcmd_tab_new(CustTabs *tabs, int focus);
static char *UIcmd_make_search_str(const char *str);
static void UIcmd_set_window_labels(Fl_Window *win, const char *str);

//----------------------------------------------------------------------------

static void close_window(void *data)
{
   CustTabs *tabs = (CustTabs*) data;
   int ntabs = tabs->num_UIs();

   while (ntabs-- > 0)
      a_UIcmd_close_bw(a_UIcmd_get_bw_by_widget(tabs->value()));
}

static void win_cb (Fl_Widget *w, void *cb_data) {
   CustTabs *tabs = (CustTabs*) cb_data;
   int ntabs = tabs->num_UIs();

   if (Fl::event_key() == FL_Escape) {
      // Don't let FLTK close a browser window due to unhandled Escape
      // (most likely with modifiers).
      return;
   }

   if (ntabs == 1) {
      close_window(tabs);
   } else if (prefs.show_quit_dialog) {
      a_UIcmd_ask_user(a_UIcmd_get_bw_by_widget(tabs->value()), "Window contains more than one tab. Close it?", close_window, tabs);
   }
}

/*
 * Given a UI or UI child widget, return its bw.
 */
BrowserWindow *a_UIcmd_get_bw_by_widget(void *v_wid)
{
   BrowserWindow *bw;
   for (int i = 0; i < a_Bw_num(); ++i) {
      bw = a_Bw_get(i);
      if (((UI*)bw->ui)->contains((Fl_Widget*)v_wid)) {
         return bw;
      }
   }
   return NULL;
}

void a_UIcmd_update_location_color()
{
   BrowserWindow *bw;
   for (int i = 0; i < a_Bw_num(); ++i) {
      bw = a_Bw_get(i);
      ((UI*)bw->ui)->update_location_color();
   }
}

void a_UIcmd_update_handedness()
{
   BrowserWindow *bw;
   Layout *layout;

   for (int i = 0; i < a_Bw_num(); ++i) {
      bw = a_Bw_get(i);
      layout = (Layout*)bw->render_layout;
      layout->setScrollbarOnLeft(prefs.ui_left_handed);
      ((UI*)bw->ui)->update_nav_button_order();
   }
}

/*
 * Create a new UI and its associated BrowserWindow data structure.
 * If x and y are nonnegative, try to position the window.
 */
BrowserWindow *a_UIcmd_browser_window_new(int x, int y, int ww, int wh)
{
   BrowserWindow *new_bw = NULL;
   Fl_Window *win;

   if (ww <= 0 || wh <= 0) {
      // Set default geometry from dillorc.
      ww = prefs.width;
      wh = prefs.height;
   }

   win = new OverlayWindow(ww, wh);
   win->box(FL_NO_BOX);

   CustTabs *DilloTabs = new CustTabs(ww, wh);
   win->end();
   win->resizable(DilloTabs);

   int focus = 1;
   new_bw = UIcmd_tab_new(DilloTabs, focus);
   win->show();

   if (x >= 0 && y >= 0) {
      // position the first window according to preferences
      DilloTabs->window()->position(x, y);
   }

   win->callback(win_cb, DilloTabs);

   return new_bw;
}

/*
 * Set the window name and icon name.
 */
static void UIcmd_set_window_labels(Fl_Window *win, const char *str)
{
   const char *copy;

   win->Fl_Widget::copy_label(str);
   copy = win->label();
   win->label(copy, copy);
}

/*
 * Create a new Tab button, UI and its associated BrowserWindow data
 * structure.
 */
static BrowserWindow *UIcmd_tab_new(CustTabs *tabs, int focus)
{
   int min_button_h_px;

   min_button_h_px = (int) rint(prefs.dpi / 25.4 * prefs.ui_min_button_h_mm);

   _MSG(" UIcmd_tab_new\n");

   // Now create the Dw render layout and viewport
   FltkPlatform *platform = new FltkPlatform ();
   Layout *layout = new Layout (platform);
   style::Color *bgColor = style::Color::create (layout, prefs.ui_main_bg_color);
   layout->setBgColor (bgColor);
   layout->setBgImage (NULL, style::BACKGROUND_REPEAT,
                       style::BACKGROUND_ATTACHMENT_SCROLL,
                       style::createPerLength (0), style::createPerLength (0));

   FltkViewport *viewport = new FltkViewport (0, 0, 1, 1);
   viewport->box(FL_NO_BOX);
   viewport->setDragScroll (prefs.drag_scrolling_button ? true : false);
   viewport->setDragThreshold (prefs.drag_threshold);
   viewport->setScrollThickness(min_button_h_px * 2 / 3);
   viewport->setScrollbarOnLeft (prefs.ui_left_handed ? true : false);
   viewport->setScrollStep((int) rint(2.0 * dLib_get_default_font_size()));
   layout->attachView (viewport);

   // Create and set the UI
   UI *new_ui = tabs->add_new_tab(viewport, focus);

   // Now, create a new browser window structure
   BrowserWindow *new_bw = a_Bw_new();

   // Reference the UI from the bw
   new_bw->ui = (void *)new_ui;
   // Copy the layout pointer into the bw data
   new_bw->render_layout = (void*)layout;

   // Clear the window title
   if (focus)
      UIcmd_set_window_labels(new_ui->window(), new_ui->label());

   a_UIcmd_set_selection_mode(new_bw, prefs.drag_scrolling_button != 1);

   return new_bw;
}

/*
 * Close one browser window
 */
void a_UIcmd_close_bw(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow *)vbw;
   UI *ui = BW2UI(bw);
   CustTabs *tabs = ui->tabs();
   Layout *layout = (Layout*)bw->render_layout;

   _MSG("a_UIcmd_close_bw\n");
   a_Bw_stop_clients(bw, BW_Root + BW_Img + BW_Force);
   delete(layout);
   if (tabs) {
      tabs->remove_tab(ui);
      if (tabs->num_UIs() == 0)
         delete tabs->window();
   }
   a_Bw_free(bw);
}

static void close_all(void *data)
{
   BrowserWindow *bw;

   while ((bw = a_Bw_get(0)))
      a_UIcmd_close_bw((void*)bw);
}

/*
 * Close all the browser windows
 */
void a_UIcmd_close_all_bw(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow *)vbw;

   if (a_Bw_num() == 1) {
      close_all(NULL);
   } else if (prefs.show_quit_dialog) {
      a_UIcmd_ask_user(bw, "Window contains more than one tab. Close it?", close_all, NULL);
   }
}

/*
 * Return a search string of the suffix if str starts with a
 * prefix of a search engine name and a blank
 */
static char *UIcmd_find_search_str(const char *str)
{
   int p;
   char *url = NULL;
   int len = strcspn(str, " ");

   if (len > 0 && str[len] != '\0') {
      /* we found a ' ' in str, check whether the first part of str
       * is a prefix of a search_url label
       */
      for (p = 0; p < dList_length(prefs.search_urls); p++) {
         const char *search =
            (const char *)dList_nth_data(prefs.search_urls, p);
         if (search && strncasecmp(str, search, len) == 0) {
            prefs.search_url_idx = p;
            url = UIcmd_make_search_str(str + len + 1);
            break;
         }
      }
   }
   return url;
}

/*
 * Open URL from given string. This is intended for use from the location bar and search dialog, detecting and handling a search string.
 *
 * our custom "file:" URIs are normalized here too.
 */
void a_UIcmd_open_location_urlstr(void *vbw, const char *urlstr)
{
   char *new_urlstr;
   char *search_urlstr = NULL;
   DilloUrl *url;
   int ch;
   BrowserWindow *bw = (BrowserWindow*)vbw;
   Dstr *POST_data = NULL;

   if ((search_urlstr = UIcmd_find_search_str(urlstr))) {
      /* Search strings are all encoded to use GET, but the great majority of real
       * search requests use POST. Search engines using POST generally tolerate GET,
       * but sometimes they grumble, and POST and GET are intended to have different
       * semantics, and not showing arguments in a URL is supposed to be somewhat
       * good for privacy, and so on.
       *
       * In the interest of being realistic about this browser and doing something
       * simple - even if the search urls are already too complex to belong in dillorc,
       * let’s just prefix URLs with "POST@" when we want to use POST.
       */
      if (!strncmp(search_urlstr, "POST@", 5)) {
         char *query;

         urlstr = search_urlstr + 5;
         query = strchr(search_urlstr, '?');

         if (query) {
            *query = '\0'; /* we don’t want the question mark at all */
            POST_data = dStr_new(query+1);
         }
      } else {
         urlstr = search_urlstr;
      }
   }

   /* If it wasn’t a search string, don’t allow any spaces.
    * This is almost invariably the result of failing to prefix
    * a search string, which would leak your search string (and
    * not get you anything of any value).
    */
   if (urlstr && *urlstr && !strchr(urlstr, ' ')) {
      /* Filter URL string */
      new_urlstr = a_Url_string_strip_delimiters(urlstr);

      if (!dStrnAsciiCasecmp(new_urlstr, "file:", 5)) {
         /* file URI */
         ch = new_urlstr[5];
         if (!ch || ch == '.') {
            url = a_Url_new(Paths::getOldWorkingDir(), "file:");
         } else if (ch == '~') {
            url = a_Url_new(dGethomedir(), "file:");
         } else {
            url = a_Url_new(new_urlstr, "file:");
         }

      } else {
         /* common case */
         url = a_Url_new(new_urlstr, NULL);

         if (POST_data)
            a_Url_set_post(url, &POST_data);
      }
      dFree(new_urlstr);

      if (url) {
         const DilloUrl *referrer = NULL;
         FetchReq req = {0, url, referrer, REFERRER_POLICY_NOTSET, HTTP_Sec_Fetch_Dest_document, TRUE};
         a_UIcmd_set_location_text(bw, URL_STR(url)); // in case http was automatically upgraded to https (or to show what happened to search URLs)
         a_UIcmd_open_url(bw, UICMD_LOC_HERE, &req);
         a_Url_free(url);
      }
   }
   dFree(search_urlstr);
   dStr_free(POST_data, 1);
}

void a_UIcmd_open_url(BrowserWindow *bw, UIcmd_location loc, FetchReq *req)
{
   if (loc == UICMD_LOC_FROM_EVENT_STATE) {
      int e = Fl::event();

      if (e == FL_RELEASE && Fl::event_button() == 2) {
         if (prefs.middle_click_opens_new_tab) {
            int focus_tab = prefs.focus_new_tab;

            if (Fl::event_shift()) focus_tab = !focus_tab;
            loc = focus_tab ? UICMD_LOC_NEW_TAB_FOCUSED : UICMD_LOC_NEW_TAB;
         } else {
            loc = UICMD_LOC_NEW_WINDOW;
         }
      } else {
         loc = UICMD_LOC_HERE;
      }
      if (e != FL_RELEASE && e != FL_KEYDOWN && e != FL_SHORTCUT) {
         MSG_ERR("a_UIcmd_open_url() called with event %d. Not really a big problem, just want to know this can happen.\n", e); // FL_KEYDOWN is known from the commands menu and FL_SHORTCUT from the UI key commands.
      }
   }

   if (loc == UICMD_LOC_NEW_TAB || loc == UICMD_LOC_NEW_TAB_FOCUSED)
      bw = UIcmd_tab_new(BW2UI(bw)->tabs(), loc == UICMD_LOC_NEW_TAB_FOCUSED);
   else if (loc == UICMD_LOC_NEW_WINDOW) {
      int w, h;
      a_UIcmd_get_wh(bw, &w, &h);
      bw = a_UIcmd_browser_window_new(-1, -1, w, h);
   }
   if (req && req->url) {
      req->user_saw_url = TRUE;
      a_Nav_push(bw, req);
      if (loc != UICMD_LOC_HERE)
         a_UIcmd_set_location_text(bw, URL_STR(req->url)); // so they know what’s happening
      BW2UI(bw)->focus_main();
   } else {
      // Used to start a blank bw
      BW2UI(bw)->focus_location();
      a_UIcmd_set_buttons_sens(bw);
   }

}

/*
 * Send the browser back to previous page
 */
void a_UIcmd_back(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow *)vbw;
   a_UIcmd_nav_jump(bw, a_Nav_stack_idx(bw) - 1);
}

/*
 * Send the browser to next page
 */
void a_UIcmd_forw(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow *)vbw;
   a_UIcmd_nav_jump(bw, a_Nav_stack_idx(bw) + 1);
}

/*
 * Reload current URL
 */
void a_UIcmd_reload(void *vbw)
{
   a_Nav_reload((BrowserWindow*)vbw);
}

/*
 * Repush current URL
 */
void a_UIcmd_repush(void *vbw)
{
   a_Nav_repush((BrowserWindow*)vbw);
}

void a_UIcmd_select_tab(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow*)vbw;
   UI *ui = BW2UI(bw);
   UIcmd_set_window_labels(ui->window(), ui->label());
   ((Fl_Wizard *) ui->parent())->value(ui);
}

/*
 * Save a URL
 */
static void UIcmd_save(BrowserWindow *bw, const DilloUrl *url)
{
   BW2UI(bw)->tabs()->show_savefile(url, bw);
}

/*
 * Save current URL
 */
void a_UIcmd_save(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow *)vbw;
   const DilloUrl *url = a_History_get_url(a_Nav_get_curr_uidx(bw));

   if (url) {
      UIcmd_save(bw, url);
   }
}

/*
 * Select a file
 */
void a_UIcmd_select_file(BrowserWindow *bw, Dstr *filename)
{
   dStr_append(filename, Paths::getOldWorkingDir());
   dStr_append_c(filename, '/');
   BW2UI(bw)->tabs()->show_selectfile(bw, filename);
}

/*
 * Stop network activity on this bw.
 * The stop button was pressed: stop page (and images) downloads.
 */
void a_UIcmd_stop(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow *)vbw;

   MSG("a_UIcmd_stop()\n");
   a_Nav_cancel_expect(bw);
   a_Bw_stop_clients(bw, BW_Root + BW_Img + BW_Force);
}

/*
 * Open URL with dialog chooser
 */
void a_UIcmd_open_file(void *vbw)
{
   char *urlstr = dStrconcat("file://", dGethomedir(), "/", NULL);
   DilloUrl *url = a_Url_new(urlstr, NULL);
   const DilloUrl *referrer = NULL;
   FetchReq req = {0, url, referrer, REFERRER_POLICY_NOTSET, HTTP_Sec_Fetch_Dest_document, TRUE};
   a_UIcmd_open_url((BrowserWindow *)vbw, UICMD_LOC_FROM_EVENT_STATE, &req);
   dFree(urlstr);
   a_Url_free(url);
}

/*
 * Returns a newly allocated string holding a search url generated from
 * a string of keywords (separated by blanks) and the current search_url.
 * The search string is urlencoded.
 */
static char *UIcmd_make_search_str(const char *str)
{
   char *search_url, *l, *u, *c;
   char *keys = a_Url_encode_hex_str(str),
        *src = (char*)dList_nth_data(prefs.search_urls, prefs.search_url_idx);
   Dstr *ds = dStr_sized_new(128);

   /* parse search_url into label and url */
   if (a_Misc_parse_search_url(src, &l, &u) == 0) {
      for (c = u; *c; c++) {
         if (*c == '%')
            switch(*++c) {
            case 's':
               dStr_append(ds, keys); break;
            case '%':
               dStr_append_c(ds, '%'); break;
            case 0:
               MSG_WARN("search_url ends with '%%'\n"); c--; break;
            default:
               MSG_WARN("illegal specifier '%%%c' in search_url\n", *c);
            }
         else
            dStr_append_c(ds, *c);
      }
   }
   dFree(keys);

   search_url = ds->str;
   dStr_free(ds, 0);
   return search_url;
}

/*
 * Get a query from a dialog and open it
 */
void a_UIcmd_websearch(BrowserWindow *bw, const char *item)
{
   if (item) {
      const char *space = strchr(item, ' ');
      if (space) {
         char *prefix = dStrndup(item, space - item + 1);
         a_UIcmd_focus_location(bw);
         a_UIcmd_set_location_text(bw, prefix);
         dFree(prefix);
      }
   }
}

char *a_UIcmd_get_passwd(const char *username)
{
   char *passwd;
   const char *title = "Dillo: Password";
   char *msg = dStrconcat("Password for user \"", username, "\"", NULL);
   passwd = a_Password_window(title, msg);
   dFree(msg);
   return passwd;
}

/*
 * Save link URL
 */
void a_UIcmd_save_link(BrowserWindow *bw, const DilloUrl *url)
{
   UIcmd_save(bw, url);
}

/*
 * The special tab showing the history of URLs opened in this session
 */
void a_UIcmd_session_history(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow*)vbw;

   BW2UI(bw)->tabs()->show_session_history(bw);
}

/*
 * The special tab showing the history of URLs on this nav stream (forward and back buttons)
 */
void a_UIcmd_nav_history(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow*)vbw;

   if (a_Nav_stack_size(bw) > 0)
      BW2UI(bw)->tabs()->show_nav_history(bw);
}

/*
 * Request the bookmarks page
 */
void a_UIcmd_book(void *vbw)
{
   char *filename = dStrconcat("file://", dGethomedir(), "/.dillo/bm.html", NULL);
   DilloUrl *url = a_Url_new(filename, NULL);
   const DilloUrl *referrer = NULL;
   FetchReq req = {0, url, referrer, REFERRER_POLICY_NOTSET, HTTP_Sec_Fetch_Dest_document, TRUE};
   a_UIcmd_open_url((BrowserWindow*)vbw, UICMD_LOC_FROM_EVENT_STATE, &req);
   dFree(filename);
   a_Url_free(url);
}

/*
 * Add a bookmark for a certain URL
 */
void a_UIcmd_add_bookmark(BrowserWindow *bw, const DilloUrl *url)
{
   a_Bookmark_add(bw, url);
}


/*
 * Popup the page menu
 */
void a_UIcmd_page_popup(void *vbw, void *v_toplevelStylesheets, void *v_importStylesheets)
{
   BrowserWindow *bw = (BrowserWindow*)vbw;
   const DilloUrl *url = a_History_get_url(a_Nav_get_curr_uidx(bw));
   BW2UI(bw)->tabs()->show_pagemenu(bw, url, bw->num_page_bugs != 0);
}

/*
 * Popup the link menu
 */
void a_UIcmd_link_popup(BrowserWindow *bw, FetchReq *req, bool_t cached)
{
   BW2UI(bw)->tabs()->show_linkmenu(bw, req, cached);
}

/*
 * Pop up the image menu
 */
void a_UIcmd_image_popup(BrowserWindow *bw, FetchReq *imgreq,
                         FetchReq *linkreq, const DilloUrl *page_url, bool_t loaded_img, bool_t img_cached, bool_t link_cached)
{
   BW2UI(bw)->tabs()->show_imagemenu(bw, imgreq, linkreq, page_url, loaded_img, img_cached, link_cached);
}

/*
 * Pop up the form menu
 */
void a_UIcmd_form_popup(BrowserWindow *bw, const DilloUrl *url, void *vform,
                        bool_t showing_hiddens)
{
   BW2UI(bw)->tabs()->show_formmenu(bw, url, vform, showing_hiddens);
}

/*
 * Pop up the input menu
 */
void a_UIcmd_input_popup(BrowserWindow *bw, void *input)
{
   if (!bw) {
      // Some rearchitecting would be good.
      MSG_ERR("These menus only work for inputs in regular browser tabs. Try BaseInput instead.\n");
   } else {
      BW2UI(bw)->tabs()->show_inputmenu(bw, input);
   }
}

/*
 * Show the commands tab
 */
void a_UIcmd_commands_tab(BrowserWindow *bw)
{
   BW2UI(bw)->tabs()->show_commands(bw);
}

// In selection mode, left mouse selects rather than scrolls.
bool_t a_UIcmd_get_selection_mode(BrowserWindow *bw)
{
   Layout *layout = (Layout*)bw->render_layout;
   return layout->getSelectionMode();
}

void a_UIcmd_set_selection_mode(BrowserWindow *bw, bool_t v)
{
   Layout *layout = (Layout*)bw->render_layout;
   layout->setSelectionMode(v);
}

/*
 * Copy url string to paste buffer
 */
void a_UIcmd_copy_urlstr(BrowserWindow *bw, const char *urlstr)
{
   Layout *layout = (Layout*)bw->render_layout;
   layout->copySelection(urlstr);
}

/*
 * Show a text window with the URL's source
 */
void a_UIcmd_view_page_source(BrowserWindow *bw, const DilloUrl *url)
{
   a_Nav_view_page_source(bw, url);
}

#if 0
static void UIcmd_notify_zoom()
{
   /* Need to rescale backing buffers */
   for (int i = 0; i < a_Bw_num(); ++i) {
      Fl::handle(FL_ZOOM_EVENT, BW2UI(a_Bw_get(i))->window());
   }
}
#endif

/* Something like zoom in */
void a_UIcmd_larger()
{
   const double stepup = 6.0 / 5.0;

#if 0
   int i;

   for (i = Fl::screen_count() - 1; i >= 0; i--)
      Fl::screen_scale(i, Fl::screen_scale(i) * stepup);
   UIcmd_notify_zoom();
#else

   BrowserWindow *bw;

   dLib_set_default_font_size(dLib_get_default_font_size() * stepup);
MSG("FL_NORMAL_SIZE is: %d\n", FL_NORMAL_SIZE);
   prefs.ui_min_button_h_mm = prefs.ui_min_button_h_mm * stepup;
   prefs.ui_min_button_w_mm = prefs.ui_min_button_w_mm * stepup;
   for (int i = 0; i < a_Bw_num(); ++i) {
      bw = a_Bw_get(i);
      a_Nav_repush(bw);
   }
#endif
}

/* Something like zoom out */
void a_UIcmd_smaller()
{
   const double stepdown = 5.0 / 6.0;

#if 0
   int i;

   for (i = Fl::screen_count() - 1; i >= 0; i--)
      Fl::screen_scale(i, Fl::screen_scale(i) * stepdown);
   UIcmd_notify_zoom();
#else

   BrowserWindow *bw;

   dLib_set_default_font_size(dLib_get_default_font_size() * stepdown);
MSG("FL_NORMAL_SIZE is: %d\n", FL_NORMAL_SIZE);
   prefs.ui_min_button_h_mm = prefs.ui_min_button_h_mm * stepdown;
   prefs.ui_min_button_w_mm = prefs.ui_min_button_w_mm * stepdown;
   for (int i = 0; i < a_Bw_num(); ++i) {
      bw = a_Bw_get(i);
      a_Nav_repush(bw);
   }
#endif
}

/*
 * Show the browser window's HTML errors in a text window
 */
void a_UIcmd_view_page_bugs(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow*)vbw;

   if (bw->num_page_bugs > 0) {
      BW2UI(bw)->tabs()->show_display(bw, "Detected HTML errors", bw->page_bugs->str);
   }
}

/*
 * The special tab listing the UIs belonging to this window
 */
void a_UIcmd_tab_list(BrowserWindow *bw)
{
   BW2UI(bw)->tabs()->show_tablist(bw);
}

/*
 * Make a list of URL indexes for the history tab
 */
int *a_UIcmd_get_history(BrowserWindow *bw)
{
   int i;
   int *hlist;

   // Count the number of URLs
   i = a_Nav_stack_size(bw);
   hlist = dNew(int, i + 1);

   // Fill the list
   for (i = 0; i < a_Nav_stack_size(bw); i++) {
      hlist[i] = a_Nav_get_uidx(bw,i);
   }
   hlist[i] = -1;

   return hlist;
}

/*
 * Jump to a certain URL in the navigation stack.
 */
void a_UIcmd_nav_jump(BrowserWindow *bw, int index)
{
   int new_bw = Fl::event() == FL_RELEASE && Fl::event_button() == 2;

   a_Nav_jump(bw, index, new_bw);

   /* History tab needs this to know that we ended up opening in same bw, unfortunately. */
   BW2UI(bw)->focus_main();
}

/*
 * This seems a little random, but, given the shape of Dillo, UIcmd seems to be the most suitable of the options to handle this. bw is who knows what’s in the UIs, and who is friendly with bw in the right sort of way?
 *
 * And of course this assumes we’re dealing with a document URL, not an image, etc.
 */
void a_UIcmd_authenticated(const DilloUrl *url)
{
   BrowserWindow *bw;
   bool reloaded = false;

   for (int i = 0; i < a_Bw_num(); ++i) {
      bw = a_Bw_get(i);
      if (a_Bw_has_url(bw, url)) {
         if (reloaded) {
            a_UIcmd_repush(bw);
         } else {
            a_UIcmd_reload(bw);
            reloaded = true;
         }
      }
   }
}

// UI binding functions -------------------------------------------------------

/*
 * Return browser window width and height
 */
void a_UIcmd_get_wh(BrowserWindow *bw, int *w, int *h)
{
   *w = BW2UI(bw)->w();
   *h = BW2UI(bw)->h();
   _MSG("a_UIcmd_wh: w=%d, h=%d\n", *w, *h);
}

/*
 * Get the scroll position (x, y offset pair)
 */
void a_UIcmd_get_scroll_xy(BrowserWindow *bw, int *x, int *y)
{
   Layout *layout = (Layout*)bw->render_layout;

   if (layout) {
     *x = layout->getScrollPosX();
     *y = layout->getScrollPosY();
   }
}

/*
 * Set the scroll position ({x, y} offset pair)
 */
void a_UIcmd_set_scroll_xy(BrowserWindow *bw, int x, int y)
{
   Layout *layout = (Layout*)bw->render_layout;

   if (layout) {
      layout->scrollTo(HPOS_LEFT, VPOS_TOP, x, y, 0, 0);
   }
}

/*
 * Set the scroll position by fragment (from URL)
 */
void a_UIcmd_set_scroll_by_fragment(BrowserWindow *bw, const char *f)
{
   Layout *layout = (Layout*)bw->render_layout;

   if (layout && f) {
      layout->setAnchor(f);
   }
}

/*
 * Pass scrolling command to dw.
 */
void a_UIcmd_scroll(BrowserWindow *bw, int icmd)
{
   Layout *layout = (Layout*)bw->render_layout;

   if (layout) {
      typedef struct {
         KeysCommand_t keys_cmd;
         ScrollCommand dw_cmd;
      } mapping_t;

      const mapping_t map[] = {
         {KEYS_SCREEN_UP, SCREEN_UP_CMD},
         {KEYS_SCREEN_DOWN, SCREEN_DOWN_CMD},
         {KEYS_SCREEN_LEFT, SCREEN_LEFT_CMD},
         {KEYS_SCREEN_RIGHT, SCREEN_RIGHT_CMD},
         {KEYS_LINE_UP, LINE_UP_CMD},
         {KEYS_LINE_DOWN, LINE_DOWN_CMD},
         {KEYS_LEFT, LEFT_CMD},
         {KEYS_RIGHT, RIGHT_CMD},
         {KEYS_TOP, TOP_CMD},
         {KEYS_BOTTOM, BOTTOM_CMD},
      };
      KeysCommand_t keycmd = (KeysCommand_t)icmd;

      for (uint_t i = 0; i < sizeof(map) / sizeof(map[0]); i++) {
         if (keycmd == map[i].keys_cmd) {
            layout->scroll(map[i].dw_cmd);
            break;
         }
      }
   }
}

/*
 * Get location's text
 */
char *a_UIcmd_get_location_text(BrowserWindow *bw)
{
   return dStrdup(BW2UI(bw)->get_location());
}

/*
 * Set location's text
 */
void a_UIcmd_set_location_text(void *vbw, const char *text)
{
   BrowserWindow *bw = (BrowserWindow*)vbw;
   BW2UI(bw)->set_location(text);
}

/*
 * Set the page progress bar
 */
void a_UIcmd_set_page_prog(BrowserWindow *bw, size_t nbytes, double proportion, uint_t bugs, uint_t cookies)
{
   const char *prefix, *format;
   double magnitude;

   Dstr *str = dStr_sized_new(48);
   Layout *layout = (Layout*)bw->render_layout;
   _MSG("page prog got %zu bytes, %f of total.\n", nbytes, proportion);
   if (nbytes >= 1024 * 1024) {
      prefix = "M";
      magnitude = nbytes / (1024 * 1024.0);
   } else if (nbytes >= 1024) {
      prefix = "K";
      magnitude = nbytes / 1024.0;
   } else {
      prefix = "";
      magnitude = nbytes;
   }
   // The 3 significant digits of %.3g is generally nice until you give it a value above 999 and then it uses scientific notation. Let’s avoid that.
   format = magnitude > 999 ? "%.0f %sB" : "%.3g %sB";
   dStr_sprintfa(str, format, magnitude, prefix);
   if (cookies)
      dStr_sprintfa(str, ", %u cookie%s", cookies, cookies == 1 ? "" : "s");
   if (bugs)
      dStr_sprintfa(str, ", %u bug%s", bugs, bugs == 1 ? "" : "s");

   layout->setOverlayProgressMsg(str->str, proportion);
   dStr_free(str, 1);
}

/*
 * Set the page title in the tab label and window titlebar.
 * (Update window titlebar for the current tab only)
 */
void a_UIcmd_set_page_title(BrowserWindow *bw, const char *label)
{
   const int size = 128;
   char title[size];

   if (snprintf(title, size, "Dillo: %s", label ? label : "") >= size) {
      uint_t i = MIN(size - 4, 1 + a_Utf8_end_of_char(title, size - 8));
      snprintf(title + i, 4, "...");
   }
   BW2UI(bw)->copy_label(title);

   if (a_UIcmd_get_bw_by_widget(BW2UI(bw)->tabs()->value()) == bw) {
      // This is the focused bw, set window title
      UIcmd_set_window_labels(BW2UI(bw)->window(), title);
   }
}

/*
 * Set a printf-like status string on the bottom of the dillo window.
 * Beware: The safe way to set an arbitrary string is
 *         a_UIcmd_set_msg(bw, "%s", str)
 */
void a_UIcmd_set_msg(BrowserWindow *bw, const char *format, ...)
{
   va_list argp;
   Dstr *ds = dStr_sized_new(128);
   va_start(argp, format);
   dStr_vsprintf(ds, format, argp);
   va_end(argp);
   BW2UI(bw)->set_status(ds->str);
   dStr_free(ds, 1);
}

void a_UIcmd_display_link_dest(BrowserWindow *bw, const char *str)
{
   Layout *layout = (Layout*)bw->render_layout;
   layout->setOverlayLinkDest(str);
}

/*
 * Set the sensitivity of back/forw/stop buttons.
 */
void a_UIcmd_set_buttons_sens(BrowserWindow *bw)
{
   int sens;

   // Stop
   sens = (dList_length(bw->ImageClients) || dList_length(bw->RootClients));
   BW2UI(bw)->button_set_sens(UI_STOP, sens);
   // Back
   sens = (a_Nav_stack_idx(bw) > 0);
   BW2UI(bw)->button_set_sens(UI_BACK, sens);
   // Forward
   sens = (a_Nav_stack_idx(bw) < a_Nav_stack_size(bw) - 1 &&
           !a_Bw_expecting(bw));
   BW2UI(bw)->button_set_sens(UI_FORW, sens);
}

/*
 * Search for next/previous occurrence of key.
 */
void a_UIcmd_findtext_search(BrowserWindow *bw, const char *key,
                             int case_sens, int backward)
{
   Layout *l = (Layout *)bw->render_layout;

   switch (l->search(key, case_sens, backward)) {
   case FindtextState::RESTART:
      a_UIcmd_set_msg(bw, backward?"Top reached; restarting from the bottom."
                                  :"Bottom reached; restarting from the top.");
      break;
   case FindtextState::NOT_FOUND:
      a_UIcmd_set_msg(bw, "\"%s\" not found.", key);
      break;
   case FindtextState::SUCCESS:
   default:
      a_UIcmd_set_msg(bw, "");
   }
}

/*
 * Reset text search state.
 */
void a_UIcmd_findtext_reset(BrowserWindow *bw)
{
   Layout *l = (Layout *)bw->render_layout;
   l->resetSearch();

   a_UIcmd_set_msg(bw, "");
}

/*
 * Tell the UI to hide/show the findbar
 */
void a_UIcmd_findbar_toggle(BrowserWindow *bw, int on)
{
   BW2UI(bw)->findbar_toggle(on);
}

void a_UIcmd_username_password(void *vbw, const char *msg, UIcmd_username_password_cb callback, void *data)
{
   BrowserWindow *bw = (BrowserWindow*)vbw;
   BW2UI(bw)->username_password(msg, callback, data);
}

void a_UIcmd_ask_user(BrowserWindow *bw, const char *question, UIcmd_ask_cb callback, void *data)
{
   BW2UI(bw)->ask_user(question, callback, data);
}

void a_UIcmd_rearrange_UI(BrowserWindow *bw)
{
   BW2UI(bw)->rearrange();
}

/*
 * Focus the rendered area.
 */
void a_UIcmd_focus_main_area(BrowserWindow *bw)
{
   BW2UI(bw)->focus_main();
}

/*
 * Focus the location bar.
 */
void a_UIcmd_focus_location(void *vbw)
{
   BrowserWindow *bw = (BrowserWindow*)vbw;
   BW2UI(bw)->focus_location();
}

